/**
	\file
	\brief		Header file for art color preference data storage
	\copyright	Hot Door, Inc. 2010-2025
*/

#ifndef __HDI_CORE_ART_COLOR_PREF__
#define __HDI_CORE_ART_COLOR_PREF__

#if defined(HDI_CORE_AIP_MODE)

#include "hdicoreArtColor.h"
#include "hdicorePrefData.h"

namespace hdi
{
	namespace core
	{
		/**
			\brief	Allows for storage of art colors
		*/
		class ArtColorPref : public PrefData
		{
			public:
				/**
					\brief	Constructs an empty ArtColorPref object
					\author	GW
					\date	09/2013
					
					\note	To test if a ArtColorPref object is empty, call isEmpty() on it
					\note	Empty ArtColorPref objects do not relate to any actual preference data; they are designed to
							be "receivers" of some other ArtColorPref object via the overloaded assignment operator.
							Empty ArtColorPref objects are useless until such time (though it is safe to call any of
							their methods).
				*/
				ArtColorPref();
			
				/**
					\brief	Constructs a new ArtColorPref object from an existing ArtColorPref object (copy constructor)
					\author	GW
					\date	09/2013

					\param	ac_		Existing ArtColorPref object
				*/
				ArtColorPref(const ArtColorPref& ac_);

				/**
					\brief	Constructs an ArtColorPref object, with new data, to be added to a container at a later time
					\author	GW
					\date	09/2013

					\param	value_	Initial color value
					\param	name_	Name for the data once it has been added to its container, as UTF-8

					\note	The name_ argument can only be "" (omitted) if the object will be added directly to an
							ArrayPref object.
				*/
				ArtColorPref(const ArtColor& value_, const std::string& name_ = "");

				/**
					\brief	Destructs an ArtColorPref object
					\author	GW
					\date	09/2013
				*/
				virtual ~ArtColorPref();

				/**
					\brief		Allows one ArtColorPref object to be assigned from another
					\author		GW
					\date		09/2013

					\param		rhs_	Righthand side of the = operator; the object to copy values from
					\returns	The target ArtColorPref object, but with its value updated to match that of rhs_
				*/
				virtual ArtColorPref& operator=(const ArtColorPref& rhs_);
				
				/**
					\brief		Convenience method to clone an ArtColorPref object on the heap
					\author		GW
					\date		10/2013

					\returns	A pointer to the new ArtColorPref object
					
					\note		If you subclass ArtColorPref, you MUST overload this method yourself! If you don't
								and/or your clone() method does not return an instance of your ArtColorPref subclass,
								you will experience "object slicing" when adding the pref to a container.

					\warning	The caller must manage the memory for the returned ArtColorPref object.
				*/
				virtual ArtColorPref* clone() const;

				/**
					\brief		Gets the color for this object
					\author		GW
					\date		09/2013

					\returns	The ArtColor object being described by this wrapper
				*/
				virtual ArtColor artColor() const;

				/**
					\brief		Sets the color for this object
					\author		GW
					\date		01/2012

					\param		color_	New color value
					\returns	true if the color is successfully updated, or false otherwise
				*/
				virtual bool setArtColor(const ArtColor& color_);
		};
		
		typedef std::unique_ptr<ArtColorPref> ArtColorPrefUP;
		typedef std::shared_ptr<ArtColorPref> ArtColorPrefSP;
		typedef std::weak_ptr<ArtColorPref> ArtColorPrefWP;
	}
}

#endif
// HDI_CORE_AIP_MODE

#endif
// __HDI_CORE_ART_COLOR_PREF__
