/**
	\file
	\brief		Header file for Illustrator art style manipulation
	\copyright	Hot Door, Inc. 2010-2025
*/

#ifndef __HDI_CORE_ART_STYLE__
#define __HDI_CORE_ART_STYLE__

#if defined(HDI_CORE_AIP_MODE)

#include "hdicoreArtboardPoint.h"
#include "hdicoreLiveEffect.h"
#include "hdicorePathStyle.h"
#include "hdicoreTypes.h"

namespace hdi
{
	namespace aip
	{
		class ArtStyle;
	}
	
	namespace core
	{
		/**
			\brief	Describes an art style
		*/
		class ArtStyle
		{
			public:
				class BlendField;
				class Effect;
				class PaintField;
				
				/**
					\brief	Stores art style gradient information
				*/
				struct PaintData
				{
					/**
						\brief	The fill gradient origin, relative to the bounding box of the object to which the style
								would be applied
					*/
					ArtboardPoint fillRelativeGradientOrigin;
					
					/**
						\brief	The fill gradient length, in points
					*/
					double fillRelativeGradientLength;
					
					/**
						\brief	The fill gradient aspect ratio, for radial gradients
					*/
					double fillGradientAspectRatio;

					/**
						\brief	The stroke gradient origin, relative to the bounding box of the object to which the
								style would be applied
					*/
					ArtboardPoint strokeRelativeGradientOrigin;
					
					/**
						\brief	The stroke gradient length, in points
					*/
					double strokeRelativeGradientLength;
					
					/**
						\brief	The stroke gradient aspect ratio, for radial gradients
					*/
					double strokeGradientAspectRatio;
					
					/**
						\brief	Constructs a new PaintData object with default values
						\author	GW
						\date	08/2014
					*/
					PaintData();
					
					/**
						\brief	Constructs a new PaintData object from another existing object
						\author	GW
						\date	08/2014
							
						\param	pd_		Existing object to copy
					*/
					PaintData(const PaintData& pd_);
					
					/**
						\brief	Destructs a PaintData object
						\author	GW
						\date	08/2014
					*/
					virtual ~PaintData();
					
					/**
						\brief		Overloaded assignment operator to copy data from the given PaintData object into the
									target
						\author		GW
						\date		08/2014
						
						\param		rhs_	Righthand side of the assignment operator
						\returns	The lefthand side of the assignment operator, but with its values updated
					*/
					PaintData& operator=(const PaintData& rhs_);
				};
		
				typedef std::unique_ptr<PaintData> PaintDataUP;
				typedef std::shared_ptr<PaintData> PaintDataSP;
				typedef std::weak_ptr<PaintData> PaintDataWP;
				
				/**
					\brief	Describes blending info for an art style or paint field of an art style
				*/
				class BlendField
				{
					public:
						/**
							\brief	Constructs an empty BlendField object
							\author	GW
							\date	08/2014
							
							\note	To test if a BlendField object is empty, call isEmpty() on it
							\note	Empty BlendField objects do not relate to any actual art style blend field; they
									are designed to be "receivers" of some other BlendField object via the overloaded
									assignment operator. Empty BlendField objects are useless until such time (though it
									is safe to call any of their methods).
						*/
						BlendField();
					
						/**
							\brief	Constructs a new BlendField object from another existing object
							\author	GW
							\date	08/2014
							
							\param	bf_		Existing object to copy
						*/
						BlendField(const BlendField& bf_);
						
						/**
							\brief	Destructs a BlendField object
							\author	GW
							\date	08/2014
						*/
						virtual ~BlendField();
						
						/**
							\brief		Overloaded assignment operator to copy data from the given BlendField object into
										the target
							\author		GW
							\date		08/2014
							
							\param		rhs_	Righthand side of the assignment operator
							\returns	The lefthand side of the assignment operator, but with its values updated
						*/
						BlendField& operator=(const BlendField& rhs_);

						/**
							\brief		Gets whether the target BlendField object is empty (constructed with the default
										ctor)
							\author		GW
							\date		08/2013
							
							\returns	true if the target BlendField object is empty, false otherwise
						*/
						bool isEmpty() const;

						/**
							\brief		Gets whether a blend field is a default blend; that is, normal 100% opacity with
										non-isolated blending and the knockout attribute set to either off or inherit
							\author		GW
							\date		08/2014
							
							\returns	true if the blend is "default", false otherwise
						*/
						bool isDefault() const;
						
						/**
							\brief	Sets a blend field to a default blend; that is, normal 100% opacity with non-isolated
									blending and the knockout attribute set to off
							\author	GW
							\date	08/2014
						*/
						void setDefault();
					
						/**
							\brief		Gets the visibility status of a blend field, which affects whether it is drawn
							\author		GW
							\date		08/2014
							
							\returns	true if visible, false otherwise
						*/
						bool visible() const;
						
						/**
							\brief	Sets the visibility status of a blend field, which affects whether it is drawn. This
									is applied to all the art objects currently targeted in the document.
							\author	GW
							\date	08/2014
							
							\param	visible_	true to show, false to hide the blend field
						*/
						void setVisible(const bool visible_);
						
						/**
							\brief		Retrieves the blending mode of a blend field
							\author		GW
							\date		08/2014
							
							\returns	The blending mode enum value
						*/
						BlendMode mode() const;

						/**
							\brief	Sets the blending mode of a blend field
							\author	GW
							\date	08/2014
							
							\param	mode_	New blending mode enum value
						*/
						void setMode(const BlendMode mode_);

						/**
							\brief		Retrieves the opacity value of a blend field
							\author		GW
							\date		08/2014
							
							\returns	The opacity value, in the range [0,1] where 0 is completely transparent and 1 is
										completely opaque
						*/
						double opacity() const;
						
						/**
							\brief	Sets the opacity value of a blend field
							\author	GW
							\date	08/2014
							
							\param	o_	New opacity value, in the range [0,1]
						*/
						void setOpacity(const double o_);
						
						/**
							\brief		Gets the isolation state of a blend field
							\author		GW
							\date		08/2014
							
							\returns	true if isolated, false otherwise
						*/
						bool isolated() const;
						
						/**
							\brief	Sets the isolation state of a blend field
							\author	GW
							\date	08/2014
							
							\param	iso_	true to isolate, false otherwise
						*/
						void setIsolated(const bool iso_);

						/**
							\brief		Retrieves the knockout setting of a blend field
							\author		GW
							\date		08/2014
							
							\returns	The knockout enum value
						*/
						BlendKnockout knockout() const;

						/**
							\brief	Sets the knockout setting of a blend field
							\author	GW
							\date	08/2014
							
							\param	ko_		New knockout enum value
						*/
						void setKnockout(const BlendKnockout ko_);
						
						/**
							\brief		Gets whether opacity and mask define the knockout shape
							\author		GW
							\date		08/2014
							
							\returns	true if opacity and mask define the knockout shape
						*/
						bool alphaIsShape() const;
						
						/**
							\brief	Sets whether opacity and mask define the knockout shape
							\author	GW
							\date	08/2014
							
							\param	alpha_	true if opacity and mask define the knockout shape
						*/
						void setAlphaIsShape(const bool alpha_);
					

					private:
						friend class ArtStyle;
						friend class PaintField;
					
						/**
							\brief	Internal data
						*/
						void* __data;
				};
		
				typedef std::unique_ptr<BlendField> BlendFieldUP;
				typedef std::shared_ptr<BlendField> BlendFieldSP;
				typedef std::weak_ptr<BlendField> BlendFieldWP;
			
				/**
					\brief	Effects are contained by art styles and are used to replace typical fill/stroke drawing
					
					\note	Pre-effects are live effects that show up before any fills and strokes in the appearance
							panel.
					\note	Post-effects are live effects that show up after all the fills and strokes in the appearance
							panel.
				*/
				class Effect
				{
					public:
						/**
							\brief	Constructs an empty Effect object
							\author	GW
							\date	08/2014
							
							\note	To test if a Effect object is empty, call isEmpty() on it
							\note	Empty Effect objects do not relate to any actual art style effect; they are designed
									to be "receivers" of some other Effect object via the overloaded assignment operator.
									Empty Effect objects are useless until such time (though it is safe to call any of
									their methods).
						*/
						Effect();
						
						/**
							\brief	Constructs a new Effect object from another existing object
							\author	GW
							\date	08/2014
							
							\param	e_	Existing object to copy
						*/
						Effect(const Effect& e_);
						
						/**
							\brief	Destructs an Effect object
							\author	GW
							\date	08/2014
						*/
						virtual ~Effect();
						
						/**
							\brief		Overloaded assignment operator to copy data from the given Effect object into
										the target
							\author		GW
							\date		08/2014
							
							\param		rhs_	Righthand side of the assignment operator
							\returns	The lefthand side of the assignment operator, but with its values updated
						*/
						Effect& operator=(const Effect& rhs_);

						/**
							\brief		Gets whether the target Effect object is empty (constructed with the default ctor)
							\author		GW
							\date		08/2013
							
							\returns	true if the target Effect object is empty, false otherwise
						*/
						bool isEmpty() const;
					
						/**
							\brief		Gets the visibility status of an effect, which affects whether it is drawn
							\author		GW
							\date		08/2014
							
							\returns	true if the effect is visible
						*/
						bool visible() const;
						
						/**
							\brief	Sets the visibility status of an effect, which affects whether it is drawn
							\author	GW
							\date	08/2014
							
							\param	visible_	true if the effect should be visible
						*/
						void setVisible(const bool visible_);
						
						/**
							\brief		Gets the live effect object for the art style effect
							\author		GW
							\date		08/2014
							
							\returns	The live effect object, or NULL for error
						*/
						std::unique_ptr<LiveEffect> liveEffect() const;

						/**
							\brief	Sets the live effect object for the art style effect
							\author	GW
							\date	08/2014
							
							\param	le_		The new live effect object
						*/
						void setLiveEffect(const LiveEffect& le_);
						
						/**
							\brief		Gets the live effect parameters object for the art style effect
							\author		GW
							\date		08/2014
							
							\returns	The live effect parameters object, or NULL for error
						*/
						std::unique_ptr<LiveEffect::Parameters> parameters() const;
						
						/**
							\brief	Sets the live effect parameters object for the art style effect
							\author	GW
							\date	08/2014
							
							\param	p_	The new live effect parameters object
						*/
						void setParameters(const LiveEffect::Parameters& p_);
					

					private:
						friend class ArtStyle;
						friend class PaintField;
					
						/**
							\brief	Internal data
						*/
						void* __data;
				};
		
				typedef std::unique_ptr<Effect> EffectUP;
				typedef std::shared_ptr<Effect> EffectSP;
				typedef std::weak_ptr<Effect> EffectWP;
				
				/**
					\brief	Corresponds to either a fill or a stroke in an art style. Each paint field can also contain
							live effects and transparency information.
				*/
				class PaintField
				{
					public:
						/**
							\brief	Constructs an empty PaintField object
							\author	GW
							\date	08/2014
							
							\note	To test if a PaintField object is empty, call isEmpty() on it
							\note	Empty PaintField objects do not relate to any actual art style paint field; they are
									designed to be "receivers" of some other PaintField object via the overloaded
									assignment operator. Empty PaintField objects are useless until such time (though it
									is safe to call any of their methods).
						*/
						PaintField();
						
						/**
							\brief	Constructs a new PaintField object from another existing object
							\author	GW
							\date	08/2014
							
							\param	pf_		Existing object to copy
						*/
						PaintField(const PaintField& pf_);
						
						/**
							\brief	Destructs a PaintField object
							\author	GW
							\date	08/2014
						*/
						virtual ~PaintField();
						
						/**
							\brief		Overloaded assignment operator to copy data from the given PaintField object into
										the target
							\author		GW
							\date		08/2014
							
							\param		rhs_	Righthand side of the assignment operator
							\returns	The lefthand side of the assignment operator, but with its values updated
						*/
						PaintField& operator=(const PaintField& rhs_);

						/**
							\brief		Gets whether the target PaintField object is empty (constructed with the default
										ctor)
							\author		GW
							\date		08/2013
							
							\returns	true if the target PaintField object is empty, false otherwise
						*/
						bool isEmpty() const;
						
						/**
							\brief		Gets the visibility status of blend of a paint field (fill/stroke), which
										affects whether the blend is drawn
							\author		GW
							\date		08/2014
							
							\returns	true if the blend field is visible, false otherwise
						*/
						bool visible() const;
						
						/**
							\brief	Sets the visibility status of blend of a paint field (fill/stroke), which affects
									whether the blend is drawn. This is applied to all the art objects currently
									targeted in the document.
							\author		GW
							\date		08/2014
							
							\param	visible_	true if the blend field should be visible, false otherwise
						*/
						void setVisible(const bool visible_);

						/**
							\brief		Reports whether a paint field corresponds to a fill
							\author		GW
							\date		08/2014
							
							\returns	true if the paint field is for a fill
						*/
						bool isFill() const;
						
						/**
							\brief		Gets fill information from a paint field
							\author		GW
							\date		08/2014
							
							\param		fs__	Return-by-reference for the fill style
							\param		pd__	Return-by-reference for the paint data
							\returns	true if the fill style and paint data could be acquired, false otherwise
						*/
						bool fill(FillStyle& fs__, PaintData& pd__);
						
						/**
							\brief	Sets fill information for a paint field
							\author	GW
							\date	08/2014
							
							\param	fs_		New fill style
							\param	pd_		New paint data
						*/
						void setFill(const FillStyle& fs_, const PaintData& pd_);
						
						/**
							\brief		Reports whether a paint field corresponds to a stroke
							\author		GW
							\date		08/2014
							
							\returns	true if the paint field is for a stroke
						*/
						bool isStroke() const;
						
						/**
							\brief		Gets stroke information from a paint field
							\author		GW
							\date		08/2014
							
							\param		ss__	Return-by-reference for the stroke style
							\param		pd__	Return-by-reference for the paint data
							\returns	true if the stroke style and paint data could be acquired, false otherwise
						*/
						bool stroke(StrokeStyle& ss__, PaintData& pd__);
						
						/**
							\brief	Sets stroke information for a paint field
							\author	GW
							\date	08/2014
							
							\param	ss_		New stroke style
							\param	pd_		New paint data
						*/
						void setStroke(const StrokeStyle& ss_, const PaintData& pd_);
						
						/**
							\brief		Gets the color position of a paint field (the point in effect order at which
										painting occurs)
							\author		GW
							\date		08/2014
							
							\returns	The paint field color position
						*/
						int32_t colorPosition() const;
						
						/**
							\brief	Sets the color position of a paint field (the point in effect order at which
									painting occurs)
							\author	GW
							\date	08/2014
							
							\param	pos_	The new paint field color position
						*/
						void setColorPosition(const int32_t pos_);
						
						/**
							\brief		Gets the even-odd fill rule for a paint field
							\author		GW
							\date		08/2014
							
							\returns	true if the even-odd rule is used to determine path insideness
						*/
						bool evenOdd() const;
						
						/**
							\brief	Sets the even-odd fill rule for a paint field
							\author	GW
							\date	08/2014
							
							\param	eo_		true if the even-odd rule should be used to determine path insideness
						*/
						void setEvenOdd(const bool eo_);
						
						/**
							\brief		Gets the number of effects in a paint field
							\author		GW
							\date		08/2014
							
							\returns	The number of effects
						*/
						int32_t effectCount() const;
					
						/**
							\brief		Gets a effect from a paint field by position index
							\author		GW
							\date		08/2014
							
							\param		index_	The position index
							\returns	The effect object, or NULL for error
						*/
						std::unique_ptr<Effect> effectAtIndex(const int32_t index_) const;
						
						/**
							\brief		Retrieves live effect and/or parameter objects for effects in a paint field that
										substitute for the built-in fill or stroke behavior
							\author		GW
							\date		08/2014
							
							\param		le__	Return-by-reference for the live effect
							\param		p__		Return-by-reference for the effect params
							\returns	true if the live effect and/or parameter objects were acquired, false otherwise
							
							\note		Current examples of strokes with an effect are brushes and inside/outside aligned
										strokes, which are implemented as live effects that replace the default stroking
										behavior. The only current example of a paint field that returns an empty live
										effect object but non-empty parameters are strokes with dash adjustment,
										arrowheads, or other beautiful strokes options.
						*/
						bool liveEffectInfo(LiveEffect& le__, LiveEffect::Parameters& p__);
						
						/**
							\brief	Sets the live effect and/or parameter references for effects in a paint field that
									substitute for the built-in fill or stroke behavior
							\author	GW
							\date	08/2014
							
							\param	le_		The new live effect; pass an empty one to remove the existing live effect
							\param	p_		The new effect params; pass an empty one to remove the existing effect params
						*/
						void setLiveEffectInfo(const LiveEffect& le_, const LiveEffect::Parameters& p_);

						/**
							\brief		Gets the blend field for the paint field
							\author		GW
							\date		08/2014
							
							\returns	The blend field object for the paint field, or NULL for error
						*/
						std::unique_ptr<BlendField> blend() const;
					

					private:
						friend class ArtStyle;
					
						/**
							\brief	Internal data
						*/
						void* __data;
				};
		
				typedef std::unique_ptr<PaintField> PaintFieldUP;
				typedef std::shared_ptr<PaintField> PaintFieldSP;
				typedef std::weak_ptr<PaintField> PaintFieldWP;
			
				/**
					\brief	Indicates the type of an art style
				*/
				enum Type
				{
					UnknownType	= 0,
					SimpleType	= 10,	// Has only one fill and one stroke
					ActiveType	= 20	// Can have multiple fills and strokes
				};
				
				/**
					\brief	Indicates the various attributes of an art style
				*/
				enum Attributes
				{
					NoAttributes					= 0,
					HasFillAttribute				= 1 << 0,
					HasStrokeAttribute				= 1 << 1,
					PatternsAttribute				= 1 << 2,
					CustomColorsAttribute			= 1 << 3,
					FillGradientsAttribute			= 1 << 4,
					FillRadialGradientsAttribute	= 1 << 5,
					StrokeGradientsAttribute		= 1 << 6,
					StrokeRadialGradientsAttribute	= 1 << 7,
					TransparencyAttribute			= 1 << 8,
					BrushesAttribute				= 1 << 9,
					EffectsAttribute				= 1 << 10,
					ScalableEffectsAttribute		= 1 << 11
				};

				/**
					\brief	Constructs an empty ArtStyle object
					\author	GW
					\date	08/2013
					
					\note	Empty ArtStyle objects do not relate to any art style in a document; they are designed to be
							"receivers" of some other ArtStyle object via the overloaded assignment operator. Empty
							ArtStyle objects are useless until such time (though it is safe to call any of their methods)
				*/
				ArtStyle();

				/**
					\brief	ArtStyle copy constructor
					\author	GW
					\date	08/2013
					
					\param	a_	Existing ArtStyle object to copy
				*/
				ArtStyle(const ArtStyle& a_);
				
				/**
					\brief	Constructs a new ArtStyle object from an AIArtStyleHandle
					\author	GW
					\date	08/2015
					
					\param	aiArtStyle_		Raw Illustrator AIArtStyleHandle
				*/
				ArtStyle(const AIArtStyleHandle aiArtStyle_);

				/**
					\brief	Destructs an ArtStyle object
					\author	GW
					\date	08/2013
				*/
				virtual ~ArtStyle();
			
				/**
					\brief		Overloaded ArtStyle assignment operator
					\author		GW
					\date		08/2013
					
					\param		rhs_	Exists ArtStyle object to copy values from
					\returns	The target ArtStyle object, but with its values updated
				*/
				virtual ArtStyle& operator=(const ArtStyle& rhs_);
				
				/**
					\brief		Tests whether a given ArtStyle object is the same as another
					\author		GW
					\date		08/2013

					\param		rhs_	ArtStyle to compare against (righthand side of equality operator)
					\returns	true for the target and rhs_ being the same style, false otherwise
				*/
				virtual bool operator==(const ArtStyle& rhs_) const;
				
				/**
					\brief		Tests whether a given ArtStyle object is not the same as another
					\author		GW
					\date		08/2013

					\param		rhs_	ArtStyle to compare against (righthand side of inequality operator)
					\returns	true for the target and rhs_ being different styles, false otherwise
				*/
				virtual bool operator!=(const ArtStyle& rhs_) const;

				/**
					\brief		Gets the art style handle around which the target object is wrapped
					\author		GW
					\date		11/2013
					
					\returns	AIArtStyleHandle for the target object

					\note		Generally, you shouldn't need to call this method. Only use it if you know what you're
								doing. If a specific piece of functionality provided by Illustrator is not handled by
								this class (or related classes), then it should probably be added to the hdi_core library.
				*/
				virtual AIArtStyleHandle aiArtStyleHandle() const;
				
				/**
					\brief		Tests the validity of the target style by searching for it in the document
					\author		GW
					\date		11/2013
					
					\returns	true for a valid art style, false otherwise
				*/
				virtual bool valid();

				/**
					\brief		Gets the name of the art style
					\author		GW
					\date		08/2013
					
					\returns	The UTF-8 name of the art style, or "" (empty string) if none
				*/
				virtual std::string name() const;

				/**
					\brief	Sets the name of the art style
					\author	GW
					\date	08/2013

					\param	name_	New name for the art style, as UTF-8
				*/
				virtual void setName(const std::string& name_);

				/**
					\brief		Gets whether an art style is anonymous (named art styles appear in the styles panel)
					\author		GW
					\date		11/2016

					\returns	true if anonymous, false if named
				*/
				virtual bool isAnonymous() const;

				/**
					\brief		Gets the type of the art style
					\author		GW
					\date		11/2013
					
					\returns	A enum value for the style type
				*/
				virtual Type styleType() const;

				/**
					\brief		Gets the attributes of the art style
					\author		GW
					\date		11/2013
					
					\returns	A bitwise OR of the attributes present in the target style
				*/
				virtual Attributes attributes() const;
				
				/**
					\brief		Gets whether the style is visible
					\author		GW
					\date		08/2014
					
					\returns	true if visible, false otherwise
					
					\note		A style is considered invisible if there are no effects, all the paint fields have a
								color of none, and the overall transparency contains default values.
				*/
				virtual bool visible() const;
				
				/**
					\brief		Gets whether the style has any fill or stroke
					\author		GW
					\date		08/2014
					
					\returns	true if a fill or stroke is present, false otherwise
				*/
				virtual bool hasFillOrStroke() const;
				
				/**
					\brief		Gets whether the style has any live effects
					\author		GW
					\date		08/2014
					
					\returns	true if any live effects are present, false otherwise
				*/
				virtual bool hasEffects() const;
				
				/**
					\brief		Gets whether the style has any transparency
					\author		GW
					\date		08/2014
					
					\returns	true if any non-default transparency info is present, false otherwise
				*/
				virtual bool hasTransparency() const;

				/**
					\brief		Gets the number of paint fields in the target style
					\author		GW
					\date		08/2014
					
					\returns	The number of paint fields, or -1 for error
				*/
				virtual int32_t paintFieldCount() const;
				
				/**
					\brief		Gets a paint field from the target style by position index
					\author		GW
					\date		08/2014
					
					\param		index_	The position index
					\returns	The paint field object, or NULL for error
				*/
				virtual std::unique_ptr<PaintField> paintFieldAtIndex(const int32_t index_) const;
				
				/**
					\brief		Gets the focus fill from the target style
					\author		GW
					\date		08/2014
					
					\returns	The focus fill paint field object, or NULL for error
					
					\note		If the art style has multiple fills, the user can only edit one at a time. The one
								currently being edited is the focus fill, and it is shown in the fill/stroke proxies on
								the Tool and Color panels.
				*/
				virtual std::unique_ptr<PaintField> focusFill() const;
				
				/**
					\brief		Gets the focus stroke from the target style
					\author		GW
					\date		08/2014
					
					\returns	The focus stroke paint field object, or NULL for error
					
					\note		If the art style has multiple stroke, the user can only edit one at a time. The one
								currently being edited is the focus stroke, and it is shown in the fill/stroke proxies
								on the Tool and Color panels.
				*/
				virtual std::unique_ptr<PaintField> focusStroke() const;

				/**
					\brief		Gets the number of pre-effects in the target style
					\author		GW
					\date		08/2014
					
					\returns	The number of pre-effects, or -1 for error
				*/
				virtual int32_t preEffectCount() const;
				
				/**
					\brief		Gets a pre-effect from the target style by position index
					\author		GW
					\date		08/2014
					
					\param		index_	The position index
					\returns	The effect object, or NULL for error
				*/
				virtual std::unique_ptr<Effect> preEffectAtIndex(const int32_t index_) const;
				
				/**
					\brief		Gets the number of post-effects in the target style
					\author		GW
					\date		08/2014
					
					\returns	The number of post-effects, or -1 for error
				*/
				virtual int32_t postEffectCount() const;
				
				/**
					\brief		Gets a post-effect from the target style by position index
					\author		GW
					\date		08/2014
					
					\param		index_	The position index
					\returns	The effect object, or NULL for error
				*/
				virtual std::unique_ptr<Effect> postEffectAtIndex(const int32_t index_) const;

				/**
					\brief		Gets the overall art style transparency in the form of a blend field
					\author		GW
					\date		08/2014
					
					\returns	The blend field object, or NULL for error
				*/
				virtual std::unique_ptr<BlendField> blendField() const;


			private:
				friend aip::ArtStyle* __accessImpl(const ArtStyle&);
				friend ArtStyle __accessCtor(aip::ArtStyle*&);

				/**
					\brief	Private implementation data
				*/
				void* __data;
				
				/**
					\brief	Internal use only
					\author	GW
					\date	08/2013
				*/
				void* __impl() const;

				/**
					\brief	Internal use only
					\author	GW
					\date	08/2013
				*/
				ArtStyle(aip::ArtStyle*&);
		};
		
		typedef std::unique_ptr<ArtStyle> ArtStyleUP;
		typedef std::shared_ptr<ArtStyle> ArtStyleSP;
		typedef std::weak_ptr<ArtStyle> ArtStyleWP;
		
		extern aip::ArtStyle* __accessImpl(const ArtStyle&);
		extern ArtStyle __accessCtor(aip::ArtStyle*&);
	}
}

hdi::core::ArtStyle::Attributes operator|(
	const hdi::core::ArtStyle::Attributes lhs_,
	const hdi::core::ArtStyle::Attributes rhs_
);

hdi::core::ArtStyle::Attributes& operator|=(
	hdi::core::ArtStyle::Attributes& lhs__,
	const hdi::core::ArtStyle::Attributes rhs_
);

#endif
// HDI_CORE_AIP_MODE

#endif
// __HDI_CORE_ART_STYLE__
