/**
	\file
	\brief		Header file for Illustrator artboard manipulation
	\copyright	Hot Door, Inc. 2010-2025
*/

#ifndef __HDI_CORE_ARTBOARD__
#define __HDI_CORE_ARTBOARD__

#if defined(HDI_CORE_AIP_MODE)

#include <vector>

#include "hdicoreMacros.h"

#include "hdicoreArtboardPoint.h"
#include "hdicoreArtboardRect.h"
#include "hdicoreArtColor.h"

namespace hdi
{
	namespace aip
	{
		class Artboard;
	}

	namespace core
	{
		class Art;
		class Layer;
	
		/**
			\brief	Represents an individual artboard in the current document
		*/
		class Artboard
		{
			public:
				typedef std::vector< std::shared_ptr<Art> > ArtVector;
			
				/**
					\brief	Constructs an empty Artboard object
					\author	GW
					\date	08/2013
					
					\note	To test if an Artboard object is empty, call isEmpty() on it
					\note	Empty Artboard objects do not relate to any artboard in an Illustrator document; they are
							designed to be "receivers" of some other Artboard object via the overloaded assignment
							operator. Empty Artboard objects are useless until such time (though it is safe to call any
							of their methods)
				*/
				Artboard();
			
				/**
					\brief	Constructs a new Artboard object from an existing Artboard object (copy constructor)
					\author	GW
					\date	08/2013

					\param	art_	Existing Artboard object
				*/
				Artboard(const Artboard& a_);

				/**
					\brief	Destructs an Artboard object
					\author	GW
					\date	08/2013
				*/
				virtual ~Artboard();
			
				/**
					\brief		Assigns one Artboard object to another
					\author		GW
					\date		08/2013

					\param		rhs_	Existing Artboard object to copy values from
					\returns	The target Artboard object, but with its values updated to match that of the rhs_ argument
				*/
				virtual Artboard& operator=(const Artboard& rhs_);

				/**
					\brief		Gets whether the target Artboard object is empty (constructed with the default ctor)
					\author		GW
					\date		08/2013
					
					\returns	true if the target Artboard object is empty, false otherwise
				*/
				virtual bool isEmpty() const;

				/**
					\brief		Tests the validity of the target artboard
					\author		GW
					\date		08/2013
					
					\returns	true for a valid artboard, false otherwise
				*/
				virtual bool valid() const;

				/**
					\brief		Removes the artboard from the document
					\author		GW
					\date		08/2013

					\returns	true for successful disposal, false otherwise

					\note		After calling dispose(), the target object is useless! The actual artboard is gone, so
								obviously none of these methods will work after that.
				*/
				virtual bool dispose();

				/**
					\brief		Tests whether a given Artboard object is the same as another
					\author		GW
					\date		08/2013

					\param		rhs_	Artboard to compare against
					\returns	true for the target and rhs_ being the same artboard, false otherwise
				*/
				virtual bool operator==(const Artboard& rhs_) const;

				/**
					\brief		Tests whether a given Artboard object is not the same as another
					\author		GW
					\date		08/2013

					\param		rhs_	Artboard to compare against
					\returns	true for the target and rhs_ being different artboards, false otherwise
				*/
				virtual bool operator!=(const Artboard& rhs_) const;

				/**
					\brief		Gets the position (bounds) of the artboard
					\author		GW
					\date		08/2013
					
					\returns	Bounds rect of the target artboard, relative to the document's absolute (0,0) coordinate
				*/
				virtual ArtboardRect bounds() const;

				/**
					\brief	Sets the position (bounds) of the artboard
					\author	GW
					\date	08/2013

					\param	bounds_		New artboard bounds, relative to the document's absolute (0,0) coordinate
				*/
				virtual void setBounds(const ArtboardRect& bounds_);

				/**
					\brief		Gets the ruler origin point of the artboard, relative to the artboard itself
					\author		GW
					\date		08/2013

					\returns	Origin point of the artboard, typically the top-left
					
					\note		Artboard ruler origins have been known to be the cause of some confusion. This method
								gives the ruler origin relative to the artboard's position in the document and NOT
								relative to the document's absolute (0,0) coordinate. If you had an artboard located at
								(200,100) relative to the document (0,0) and the artboard had its individual ruler
								origin set to (20,10) relative to the artboard, then the ruler origin relative to the
								document's absolute (0,0) coordinate is (220,110). Got it?
				*/
				virtual ArtboardPoint rulerOrigin() const;

				/**
					\brief	Sets the ruler origin point of the artboard, relative to the artboard itself
					\author	GW
					\date	08/2013

					\param	origin_		New artboard origin
				*/
				virtual void setRulerOrigin(const ArtboardPoint& origin_);

				/**
					\brief		Gets the artboard's name
					\author		GW
					\date		08/2013

					\returns	Name of the target artboard, as UTF-8
				*/
				virtual std::string name() const;

				/**
					\brief	Sets the artboard's name
					\author	GW
					\date	08/2013

					\param	name_	New name for the artboard, as UTF-8
				*/
				virtual void setName(const std::string& name_);

				#if HDI_CORE_COMPILE_AI30_PLUS
					/**
						\brief		Gets the color for the artboard
						\author		GW
						\date		09/2025
						
						\returns	The artboard color
					*/
					virtual ArtColor color() const;
					
					/**
						\brief	Sets the color for the artboard
						\author	GW
						\date	09/2025
						
						\param	color_	The new artboard color
					*/
					virtual void setColor(const ArtColor& color_);
					
					/**
						\brief		Gets whether the artboard is editable
						\author		GW
						\date		09/2025
						
						\returns	true for the artboard being editable (unlocked), false otherwise
						
						\note		Always returns the opposite of locked() method.
					*/
					virtual bool editable() const;

					/**
						\brief	Sets whether the artboard is editable
						\author	GW
						\date	09/2025

						\param	editable_	true to allow editing, false otherwise

						\note	Always sets the artboard to be locked.
					*/
					virtual void setEditable(const bool editable_);
					
					/**
						\brief		Gets whether the artboard is locked
						\author		GW
						\date		09/2025
						
						\returns	true for the artboard being locked (uneditable), false otherwise
						
						\note		Always returns the opposite of editable() method.
					*/
					virtual bool locked() const;

					/**
						\brief	Sets whether the artboard is locked
						\author	GW
						\date	09/2025

						\param	lock_	true to lock, false otherwise

						\note	Always sets the artboard to be uneditable.
					*/
					virtual void setLocked(const bool lock_);
					
					/**
						\brief		Gets whether the artboard is hidden
						\author		GW
						\date		09/2025
						
						\returns	true for artboard being hidden, false otherwise
					*/
					virtual bool hidden() const;

					/**
						\brief	Sets whether the artboard is hidden
						\author	GW
						\date	09/2025

						\param	hidden_		true to hide, false otherwise
					*/
					virtual void setHidden(const bool hidden_);
				#endif
				// HDI_CORE_COMPILE_AI30_PLUS

				/**
					\brief		Gets whether the target artboard is the current one in the current document
					\author		GW
					\date		08/2013

					\returns	true if the target artboard is currently selected
				*/
				virtual bool active() const;
			
				/**
					\brief		Gets all the art objects that "touch" the target artboard
					\author		GW
					\date		01/2015
				 
					\returns	A vector of art objects whose bounds intersect the bounds of the target artboard in any
								way
				*/
				virtual ArtVector allArtObjects() const;


			private:
				friend aip::Artboard* __accessImpl(const Artboard&);
				friend Artboard __accessCtor(aip::Artboard*&);

				/**
					\brief	Private implementation data
				*/
				void* __data;
				
				/**
					\brief	Internal use only
					\author	GW
					\date	08/2013
				*/
				void* __impl() const;
				
				/**
					\brief	Internal use only
					\author	GW
					\date	08/2013
				*/
				Artboard(aip::Artboard*&);
		};
		
		typedef std::unique_ptr<Artboard> ArtboardUP;
		typedef std::shared_ptr<Artboard> ArtboardSP;
		typedef std::weak_ptr<Artboard> ArtboardWP;
		
		extern aip::Artboard* __accessImpl(const Artboard&);
		extern Artboard __accessCtor(aip::Artboard*&);
	}
}

#endif
// HDI_CORE_AIP_MODE

#endif
// __HDI_CORE_ARTBOARD__
