/**
	\file
	\brief		Header file for callback-related utilities
	\copyright	Hot Door, Inc. 2010-2025
*/

#ifndef __HDI_CORE_CALLBACK__
#define __HDI_CORE_CALLBACK__

#include <assert.h>

#include "hdicoreTypes.h"

/**
	\brief	Inserts a properly templated type name for a Callback of a particular class
	\author	GW
	\date	10/2013
	
	\param	__HDI_C__	Class name (scope-resolved) within which a Callback instantiation will call a method
	
	\code
		// Here's an example:
		typedef HDI_CORE_CALLBACK_T(myplugin::MyPanel) MyPanelCB;
		const MyPanelCB someCommonCB = HDI_CORE_CALLBACK(myplugin::MyPanel, this, commonMethod);
		// Now the someCommonCB object can be assigned to a variety of e.g. widgets
	\endcode
	
	\note	See the documentation for the HDI_CORE_CALLBACK macro for more info.
*/
#define HDI_CORE_CALLBACK_T(__HDI_C__)	hdi::core::__Callback<__HDI_C__>

/**
	\brief	Creates a properly templated Callback object for firing a particular method of a particular object
	\author	GW
	\date	08/2013
	
	\param	__HDI_C__	Class name (scope-resolved) of which __HDI_O__ is an instantiation (e.g. myplugin::MyPanel)
	\param	__HDI_O__	Object pointer that will act as the target for the callback
	\param	__HDI_M__	Method name that will act as the target's action for the callback
	
	\code
		// Here's an example:
		hdi::core::Button* transformBtn = new hdi::core::Button(<args>);
		transformBtn->setClickCallback( HDI_CORE_CALLBACK(myplugin::MyPanel, this, btnWasClicked) );
		// When the above button is clicked by the user, myplugin::MyPanel::btnWasClicked() will be called with the
		// myplugin::MyPanel object ("this") as the target
	\endcode
*/
#define HDI_CORE_CALLBACK(__HDI_C__, __HDI_O__, __HDI_M__) \
	hdi::core::__Callback<__HDI_C__>(__HDI_O__, &__HDI_C__::__HDI_M__)

/**
	\brief	Creates an "empty" callback object for when no callback is desired
	\author	GW
	\date	08/2013
*/
#define HDI_CORE_CALLBACK_NONE	HDI_CORE_CALLBACK(hdi::core::__Dummy, NULL, nothing)

namespace hdi
{
	namespace core
	{
		/**
			\brief		Base class for templated __Callback class
			\details	This class mostly exists to allow for storage of __Callback objects in containers (as Callback*)
		*/
		class Callback
		{
			public:
				/**
					\brief Pure virtual callback execution method, so __Callback objects can be stored as Callback*
				*/
				virtual void execute() const = 0;
				
				/**
					\brief Pure virtual callback cloning method, so __Callback objects can be stored as Callback*
				*/
				virtual Callback* clone() const = 0;

				/**
					\brief	Callback destructor, no-op at this time
					\author	GW
					\date	08/2013
				*/
				virtual ~Callback()
				{
					// no-op
				}


			protected:
				/**
					\brief	Callback constructor, no-op at this time
					\author	GW
					\date	08/2013
					
					\note	Callback objects can currently only be constructed by the __Callback subclass (or a subclass
							of your own)
				*/
				Callback()
				{
					// no-op
				}
		};
		
		typedef std::unique_ptr<Callback> CallbackUP;
		typedef std::shared_ptr<Callback> CallbackSP;
		typedef std::weak_ptr<Callback> CallbackWP;



		/**
			\brief		Class to describe a target object and a method of it, to be called at a later time
			\details	C++ has no callbacks (in the sense of an action [method] of a target [object]), so we had to
						roll our own. You can use this class directly, but the HDI_CORE_CALLBACK macro is easier.
		*/
		template <class hdiTargetType>
		class __Callback : public Callback
		{
			private:
				/**
					\brief	Convenience typedef for the type of the target (object)
				*/
				typedef hdiTargetType TargetType;

				/**
					\brief	Convenience typedef for the type of the action (method)
				*/
				typedef void (TargetType::*ActionType)(void);

				/**
					\brief	Convenience typedef for the type of the action (method)
				*/
				typedef void (TargetType::*ConstActionType)(void) const;

				/**
					\brief	Pointer to target (object) for the callback
				*/
				TargetType* __target;

				/**
					\brief	Function pointer to the action (method) on the target that will be called at a later time
				*/
				ActionType __action;

				/**
					\brief	Function pointer to the action (method) on the target that will be called at a later time
				*/
				ConstActionType __constAction;


			public:
				/**
					\brief		__Callback constructor
					\details	Constructs a __Callback object from a target (object) and action (method) for the target
					\author		GW
					\date		08/2013
					
					\param		target_		Object pointer, whose method (action_) will be called, once executed
					\param		action_		Function/method pointer, which will be called on target_, once executed
					
					\warning	The resulting __Callback object does not assume responsibility for the memory of target_
				*/
				explicit __Callback(TargetType* target_, ActionType action_) :
					Callback(),
					__target(target_),
					__action(action_),
					__constAction(NULL)
				{
//					#if defined(HDI_CORE_DEBUG)
//						assert(target_ != NULL);
//						assert(action_ != NULL);
//					#endif
				}

				explicit __Callback(TargetType* target_, ConstActionType action_) :
					Callback(),
					__target(target_),
					__action(NULL),
					__constAction(action_)
				{
//					#if defined(HDI_CORE_DEBUG)
//						assert(target_ != NULL);
//						assert(action_ != NULL);
//					#endif
				}

				/**
					\brief		__Callback copy constructor
					\details	Use to create a new __Callback object that points to the same target (object) and action
								(method) as the original
					\author		GW
					\date		08/2013
					
					\param		cb_		Existing __Callback object to be copied
				*/
				__Callback(const __Callback<TargetType>& cb_) :
					Callback(),
					__target(cb_.__target),
					__action(cb_.__action),
					__constAction(cb_.__constAction)
				{
					// no-op
				}

				/**
					\brief	__Callback destructor; no-op at this time
					\author	GW
					\date	08/2013
				*/
				virtual ~__Callback()
				{
					// no-op
				}

				/**
					\brief		Convenience method to clone a __Callback object
					\details	Clones the __Callback object, such that a new duplicated object is created that points
								to the same target (object) and action (method)
					\author		GW
					\date		08/2013
					
					\returns	Pointer to __Callback<TargetType>, a clone of the original
					
					\warning	The caller assumes responsibility of the memory for the returned object
				*/
				virtual __Callback<TargetType>* clone() const
				{
					return new __Callback<TargetType>(*this);
				}
				
				/**
					\brief		Executes the callback
					\details	Calls the specified action (method) of the target (object)
					\author		GW
					\date		08/2013
				*/
				virtual void execute() const
				{
					if(!this->__target)
						return;

					if(this->__action)
					{
						(this->__target->*this->__action)();
					}
					else if(this->__constAction)
					{
						(this->__target->*this->__constAction)();
					}
				}


			private:
				/**
					\brief	Default __Callback constructor, not to be used!
					\author	GW
					\date	08/2013
				*/
				__Callback() : Callback(), __target(NULL), __action(NULL), __constAction(NULL)
				{
					// no-op
				}
		};

		/**
			\brief	Just a simple "dummy" class to allow for creation of "empty" callbacks
		*/
		class __Dummy
		{
			public:
				/**
					\brief	__Dummy no-op
					\author	GW
					\date	08/2013
				*/
				void nothing();


			private:
				/**
					\brief	__Dummy constructor
					\author	GW
					\date	08/2013
				*/
				__Dummy();
				
				/**
					\brief	__Dummy copy constructor
					\author	GW
					\date	08/2013
				*/
				__Dummy(const __Dummy&);

				/**
					\brief	__Dummy destructor
					\author	GW
					\date	08/2013
				*/
				~__Dummy();

				/**
					\brief	__Dummy assignment operator
					\author	GW
					\date	08/2013
				*/
				__Dummy& operator=(const __Dummy&);
		};
	}
}

#endif
// __HDI_CORE_CALLBACK__
