/**
	\file
	\brief		Header file for plugin user interface column list widgets
	\copyright	Hot Door, Inc. 2010-2025
*/

#ifndef __HDI_CORE_COLUMN_VIEW__
#define __HDI_CORE_COLUMN_VIEW__

#include <vector>

#include "hdicoreMacros.h"

#include "hdicoreWidget.h"

namespace hdi
{
	namespace core
	{
		class Callback;
		class ColumnEntry;

		/**
			\brief	Column view widget, much like a list view but allowing for entries with multiple columns
		*/
		class ColumnView : public Widget
		{
			public:
				typedef std::vector<ColumnEntry> EntryVector;
				typedef std::vector<std::string> ColumnNameVector;
				typedef std::vector<double> ColumnWidthVector;
			
				/**
					\brief	Constructs an empty ColumnView object
					\author	GW
					\date	01/2015
					
					\note	To test if a ColumnView object is empty, call isEmpty() on it
					\note	Empty ColumnView objects do not relate to any actual UI widget; they are designed to be
							"receivers" of some other ColumnView object via the overloaded assignment operator. Empty
							ColumnView objects are useless until such time (though it is safe to call any of their
							methods).
				*/
				ColumnView();

				/**
					\brief	Constructs a new ColumnView object from an existing ColumnView object (copy constructor)
					\author	GW
					\date	01/2015

					\param	colView_	Existing ColumnView object
				*/
				ColumnView(const ColumnView& colView_);

				/**
					\brief	Constructs a column view with a frame only, no contents
					\author	GW
					\date	01/2015

					\param	frame_		Position and size of the list, in 1x resolution coordinates
					\param	numCols_	The number of columns in the list
				*/
				ColumnView(const Rect& frame_, const uint16_t numCols_);

				/**
					\brief	Constructs a column view with a frame only, no contents
					\author	GW
					\date	01/2015

					\param	frame_		Position and size of the list, in 1x resolution coordinates
					\param	colNames_	Names of each of the columns - also controls the number of columns in the list
				*/
				ColumnView(const Rect& frame_, const ColumnNameVector& colNames_);

				/**
					\brief	Constructs a column view
					\author	GW
					\date	01/2015

					\param	frame_		Position and size of the list, in 1x resolution coordinates
					\param	numCols_	The number of columns in the list
					\param	entries_	Vector of entries
					\param	index_		Initial index to select (must exist in entries_)
				*/
				ColumnView(
					const Rect& frame_,
					const uint16_t numCols_,
					const EntryVector& entries_,
					const int32_t index_
				);

				/**
					\brief	Constructs a column view
					\author	GW
					\date	01/2015

					\param	frame_		Position and size of the list, in 1x resolution coordinates
					\param	colNames_	Names of each of the columns - also controls the number of columns in the list
					\param	entries_	Vector of entries
					\param	index_		Initial index to select (must exist in entries_)
				*/
				ColumnView(
					const Rect& frame_,
					const ColumnNameVector& colNames_,
					const EntryVector& entries_,
					const int32_t index_
				);

				/**
					\brief	Constructor for a column view with an initial ID selected
					\author	GW
					\date	01/2015

					\param	frame_		Position and size of the list, in 1x resolution coordinates
					\param	numCols_	The number of columns in the list
					\param	entries_	Vector of entries
					\param	initialID_	Initial ID to select (must exist in entries_, and must not be a separator), as
										UTF-8
				*/
				ColumnView(
					const Rect& frame_,
					const uint16_t numCols_,
					const EntryVector& entries_,
					const std::string& initialID_
				);

				/**
					\brief	Constructor for a column view with an initial ID selected
					\author	GW
					\date	01/2015

					\param	frame_		Position and size of the list, in 1x resolution coordinates
					\param	colNames_	Names of each of the columns - also controls the number of columns in the list
					\param	entries_	Vector of entries
					\param	initialID_	Initial ID to select (must exist in entries_, and must not be a separator), as
										UTF-8
				*/
				ColumnView(
					const Rect& frame_,
					const ColumnNameVector& colNames_,
					const EntryVector& entries_,
					const std::string& initialID_
				);

				/**
					\brief		Allows one ColumnView object to be assigned from another
					\author		GW
					\date		01/2015

					\param		rhs_	Righthand side of the = operator; the object to copy values from
					\returns	The target ColumnView object, but with its value updated to match that of rhs_
				*/
				virtual ColumnView& operator=(const ColumnView& rhs_);
				
				/**
					\brief	ColumnView destructor
					\author	GW
					\date	01/2015
				*/
				virtual ~ColumnView();
				
				/**
					\brief		Convenience method to clone a ColumnView object on the heap
					\author		GW
					\date		01/2015

					\returns	A pointer to the new ColumnView object
					
					\note		If you subclass ColumnView, you MUST overload this method yourself! If you don't and/or
								your clone() method does not return an instance of your ColumnView subclass, you will
								experience "object slicing" when adding the widget to a window.

					\warning	The caller must manage the memory for the returned ColumnView object.
				*/
				virtual ColumnView* clone() const;
				
				/**
					\brief		Convenience method to duplicate a ColumnView object, creating a new and identical UI
								element to the target (but not belonging to the same containing window)
					\author		GW
					\date		01/2015
					
					\returns	A pointer to the new ColumnView object (and new UI element)
					
					\note		The new column view will have copies of all entries present in the target, but be aware
								that copies of the user data in each entry cannot be made (because they are simply
								void* variables). As such, the user data in the copies will simply be set to NULL by
								this method.

					\warning	The caller must manage the memory for the returned ColumnView object.
				*/
				virtual ColumnView* duplicate() const;

				/**
					\brief	Deselects all currently selected entries in the list
					\author	GW
					\date	01/2015
				*/
				virtual void deselectAll();

				/**
					\brief		Gets the current text value of the list
					\author		GW
					\date		01/2015
					
					\returns	"", as a multi-column list has no specific text value
				*/
				virtual std::string text() const;

				/**
					\brief	Does nothing
					\author	GW
					\date	01/2015

					\note	Since a ColumnView inherently stores multiple text values per row, setting the text value
							has potentially no meaning or an ambiguous meaning. Therefore, this method performs no
							action.
				*/
				virtual void setText(const std::string&);

				/**
					\brief		Gets the index of the currently selected entry
					\author		GW
					\date		01/2015

					\returns	The index number of the currently selected entry, or -1 if no entry is selected
				*/
				virtual int32_t currentIndex() const;
				
				/**
					\brief		Sets the currently selected entry by index
					\author		GW
					\date		01/2015

					\param		index_	New index to select
					\returns	true if the selection occurred successfully, or false if e.g. out of range
				*/
				virtual bool setCurrentIndex(const int32_t index_);
				
				/**
					\brief		Gets the ID of the currently selected entry
					\author		GW
					\date		01/2015

					\returns	UTF-8 ID string of the currently selected entry, or "" if no entry is currently selected
				*/
				virtual std::string currentID() const;
				
				/**
					\brief		Sets the currently selected entry by ID
					\author		GW
					\date		01/2015

					\param		id_		ID of new value to select in the column view, as UTF-8
					\returns	true if the entry with the given ID was selected, or false if invalid
				*/
				virtual bool setCurrentID(const std::string& id_);

				/**
					\brief		Gets the current entry object
					\author		GW
					\date		01/2015

					\returns	The ColumnEntry object for the currently selected item
				*/
				virtual std::unique_ptr<ColumnEntry> currentEntry() const;

				/**
					\brief		Gets the entries vector
					\author		GW
					\date		01/2015

					\returns	The entries vector for the target widget, containing all currently known entries
					
					\note		Modifying any value in an entry will not cause the widget to update automatically; call
								update() to update the UI for changes made to entry data.
				*/
				virtual const std::vector< std::shared_ptr<ColumnEntry> >* const entries() const;
				
				/**
					\brief		Gets whether the entries vector already contains an entry, based on its ID
					\author		GW
					\date		01/2015

					\param		id_		ID of the entry to search for, as UTF-8
					\returns	true if the widget already contains an entry with the given ID, false otherwise
				*/
				virtual bool hasEntry(const std::string& id_) const;
				
				/**
					\brief		Gets the entry with the given ID, provided it exists within the widget
					\author		GW
					\date		12/2022

					\param		id_		ID of the entry to search for
					\returns	The entry with the given ID, or NULL if not found
				*/
				virtual std::shared_ptr<ColumnEntry> entryWithID(const std::string& id_) const;
				
				/**
					\brief	Allows the entries in a column view to be changed out entirely
					\author	GW
					\date	01/2015

					\param	entries_	New set of entries that the widget will contain
				*/
				virtual void setEntries(const EntryVector& entries_);

				/**
					\brief		Pushes an entry onto the list (i.e. at the end)
					\author		GW
					\date		12/2022

					\param		entry_	New entry object to insert into the list of entries
					\returns	true if the entry is pushed successfully, false otherwise

					\note		If the entry is already present in the list (based on its ID), it will not be inserted.
				*/
				virtual bool pushEntry(const ColumnEntry& entry_);

				/**
					\brief		Pops an entry off the list (i.e. from the end)
					\author		GW
					\date		12/2022

					\param		entry__		Return-by-reference for the popped entry, so the caller can inspect which
											one was removed
					\returns	true if an entry was popped and acquired, false otherwise
				*/
				virtual bool popEntry(ColumnEntry& entry__);

				/**
					\brief		Inserts an entry into the list of entries at a given position
					\author		GW
					\date		12/2022

					\param		id_		Existing entry immediately before the position at which the new entry will be
										inserted, as UTF-8
					\param		entry_	New entry object to insert into the list of entries
					\returns	true if the entry is inserted successfully, false otherwise

					\note		If the entry for id_ doesn't exist in the list, the new entry will not be inserted.
					\note		If the passed entry is already present in the list (based on its ID), it will not be
								inserted.
				*/
				virtual bool insertEntry(const std::string& id_, const ColumnEntry& entry_);

				/**
					\brief		Removes a specific entry from the list of entries
					\author		GW
					\date		12/2022

					\param		id_			ID of the entry to be removed, as UTF-8
					\param		entry__		Return-by-reference for the removed entry, so the caller can inspect which
											one was removed
					\returns	true if an entry was removed and acquired, false otherwise
				*/
				virtual bool removeEntry(const std::string& id_, ColumnEntry& entry__);

				/**
					\brief		Reorders a target entry (based on ID) to be newly located just after another existing
								entry in the list
					\author		GW
					\date		12/2022

					\param		targetID_	ID of an existing entry in the list to be relocated
					\param		positionID_	ID of an existing entry immediately before the position at which targetID_
											should be moved; if the passed value is an empty string, targetID_ will be
											repositioned at the beginning
					\returns	true if the entry is repositioned successfully, false otherwise

					\note		If the entry for targetID_ or for positionID_ doesn't exist in the list, this method
								will bail.
				*/
				virtual bool reorderEntry(const std::string& targetID_, const std::string& positionID_);

				/**
					\brief	Clears all entries contained by the widget
					\author	GW
					\date	01/2015
				*/
				virtual void clearEntries();

				/**
					\brief		Gets whether the widget has a header component to show the names of the columns
					\author		GW
					\date		02/2015
					
					\returns	true if the widget was constructed with a header, false otherwise
				*/
				virtual bool hasHeader() const;
				
				/**
					\brief		Gets the number of columns in the widget
					\author		GW
					\date		02/2015
					
					\returns	The column count for the target widget, as defined when the widget was created
				*/
				virtual uint16_t columnCount() const;

				/**
					\brief		Gets the column names vector
					\author		GW
					\date		12/2014
					
					\returns	The column names vector for the target widget, containing all column names as defined
								when the widget was created

					\note		If the hasHeader() method returns false, this method will return an empty vector.
				*/
				virtual ColumnNameVector columnNames() const;
				
				/**
					\brief		Gets the widths of each column in the view
					\author		GW
					\date		01/2015
					
					\returns	A vector populated with the pixels widths of each column, at 1x resolution
				*/
				virtual ColumnWidthVector columnWidths() const;
				
				/**
					\brief	Sets the widths of each column in the view
					\author	GW
					\date	01/2015
					
					\param	widths_		Vector of new column widths to set, at 1x resolution
				*/
				virtual void setColumnWidths(const ColumnWidthVector& widths_);
				
				/**
					\brief		Gets whether the column widths are locked (i.e. the user cannot change them)
					\author		GW
					\date		02/2015
					
					\returns	true for locked widths, false otherwise
				*/
				virtual bool columnWidthsLocked() const;
				
				/**
					\brief	Sets whether the column widths are locked (i.e. the user cannot change them)
					\author	GW
					\date	02/2015
					
					\param	locked_		true to lock widths, false otherwise
				*/
				virtual void setColumnWidthsLocked(const bool locked_);

				/**
					\brief	Simulates the column view having its value changed
					\author	GW
					\date	01/2015
				*/
				virtual void valueChanged();
				
				/**
					\brief		Gets the value changed callback for the column view
					\author		GW
					\date		01/2015

					\returns	A pointer to the currently registered value changed callback

					\note		In this context, "value" refers to the selected index/ID changing.
				*/
				virtual Callback* const valueChangedCallback() const;
				
				/**
					\brief	Sets the value changed callback
					\author	GW
					\date	01/2015

					\param	callback_	New callback for when the column view's value has changed

					\note	In this context, "value" refers to the selected index/ID changing.
				*/
				virtual void setValueChangedCallback(const Callback& callback_);
				
				/**
					\brief	Force the column view to update its entries
					\author	GW
					\date	01/2015
				*/
				virtual void update();
				
				/**
					\brief		Gets the callback set for when a cell is clicked
					\author		GW
					\date		02/2015

					\returns	A pointer to the currently registered cell click callback
				*/
				virtual Callback* const cellClickCallback() const;
				
				/**
					\brief	Sets/updates the callback for when a cell is clicked
					\author	GW
					\date	01/2015

					\param	callback_		New callback for a user clicking a cell

					\note	When your callback is executed, you can determine which entry was clicked by calling the 
							clickedEntry() method and which column by calling the clickedColumnIndex() method.
				*/
				virtual void setCellClickCallback(const Callback& callback_);
				
				/**
					\brief		Gets the callback for when a cell is double clicked
					\author		GW
					\date		11/2024

					\returns	A pointer to the currently registered double-click callback
				*/
				virtual Callback* const cellDoubleClickCallback() const;
				
				/**
					\brief	Sets the callback for when a cell is double-clicked
					\author	GW
					\date	11/2024

					\param	callback_	New callback for when the column view's current entry has a cell double clicked

					\note	This callback will not be fired when editable cells are double clicked.
				*/
				virtual void setCellDoubleClickCallback(const Callback& callback_);
				
				/**
					\brief		Gets the entry of the cell that was last clicked
					\author		GW
					\date		01/2015
					
					\returns	The clicked entry object
					
					\note		The return value of this method is only guaranteed to be valid during the execution of
								the cell click callback.
				*/
				virtual std::unique_ptr<ColumnEntry> clickedEntry() const;
				
				/**
					\brief		Gets the column index of the cell in the entry that was last clicked
					\author		GW
					\date		01/2015
					
					\returns	The column index of the clicked cell, or -1 for none/error
					
					\note		The return value of this method is only guaranteed to be valid during the execution of
								the cell click callback.
				*/
				virtual int32_t clickedColumnIndex() const;
				
				/**
					\brief		Gets the callback for when a cell is modified via double-click inline editing
					\author		GW
					\date		07/2022

					\returns	A pointer to the currently registered cell modification callback
				*/
				virtual Callback* const cellChangedCallback() const;
				
				/**
					\brief	Sets the callback for when a cell is modified via double-click inline editing
					\author	GW
					\date	07/2022

					\param	callback_	New callback for a user modifying a cell via inline editing

					\note	When your callback is executed, you can determine which entry was clicked by calling the
							changedEntry() method.
				*/
				virtual void setCellChangedCallback(const Callback& callback_);
				
				/**
					\brief		Gets the entry that was last modified via double-click inline editing
					\author		GW
					\date		07/2022
					
					\returns	The changed entry object
					
					\note		The return value of this method is only guaranteed to be valid during the execution of
								the cell changed callback.
				*/
				virtual std::unique_ptr<ColumnEntry> changedEntry() const;
				
				/**
					\brief		Gets the column index of the cell in the entry that was last modified
					\author		GW
					\date		07/2022
				 
					\returns	The column index of the changed cell, or -1 for none/error
					
					\note		The return value of this method is only guaranteed to be valid during the execution of
								the cell changed callback.
				*/
				virtual int32_t changedColumnIndex() const;
				
				/**
					\brief		Gets whether reordering support for the list is enabled
					\author		GW
					\date		12/2022
				 
					\returns	true if drag-and-drop reordering support is enabled, false otherwise
				*/
				virtual bool reorderable() const;
			
				/**
					\brief	Sets whether reordering support for the list is enabled
					\author	GW
					\date	12/2022
				 
					\param	enable_		true to enable drag-and-drop reordering support, false otherwise
				*/
				virtual void setReorderable(const bool enable_);

				/**
					\brief		Gets the current potential-reorder callback for the list
					\author		GW
					\date		12/2022

					\returns	A pointer to the currently registered potential-reorder callback
				*/
				virtual Callback* const potentialReorderCallback() const;
			
				/**
					\brief	Sets the potential-reorder callback for the widget
					\author	GW
					\date	12/2022

					\param	callback_	New callback (target and action) for each time the potential reorder target changes
					
					\note	When your callback is executed, you can determine which entry is being proposed for reorder
							and its proposed new position by calling the potentialReorderSource() and
							potentialReorderPosition() methods, respectively.
				*/
				virtual void setPotentialReorderCallback(const Callback& callback_);
			
				/**
					\brief		Gets the source item involved in the potential reorder, i.e. the entry being dragged
					\author		GW
					\date		12/2022
				 
					\returns	The ID path of the entry that might be reordered
				 
					\note		The return value of this method is only guaranteed to be valid during the execution of
								the potential-reorder callback.
				*/
				virtual std::vector<std::string> potentialReorderSource() const;

				/**
					\brief		Gets the position for the reorder, i.e. the entry just before the proposed reposition
					\author		GW
					\date		12/2022
				 
					\returns	The ID path of the entry just before the potential drop point for a reposition; if the
								returned vector contains only an empty string, this refers to the proposed drop position
								being the first at the root level.
				 
					\note		The return value of this method is only guaranteed to be valid during the execution of
								the potential-reorder callback.
				*/
				virtual std::vector<std::string> potentialReorderPosition() const;

				/**
					\brief		Gets whether the potential reorder is acceptable; if not, the drag cursor will indicate
								the proposed drop position is not allowed
					\author		GW
					\date		12/2022
				
					\returns	true if the specific reorder drop position is allowed, false otherwise
					
					\note		The return value of this method is only guaranteed to be valid during the execution of
								the potential-reorder callback.
				*/
				virtual bool potentialReorderGood() const;

				/**
					\brief	Sets whether the potential reorder is acceptable; if not, the drag cursor will indicate the
							proposed drop position is not allowed
					\author	GW
					\date	12/2022
				
					\param	good_	Whether the potential reorder position is acceptable
					
					\note	This method will have no effect if it is called any other time than from within a potential-
							reorder callback.
					\note	A potential reorder is always assumed to be good, so the true/false nature of a potential
							reorder will always be true unless you have called this method within your callback and
							passed false.
					\note	Calling this method and passing false will not stop a potential reorder from occurring
							altogether; rather, it merely indicates whether a specific reorder proposed drop position is
							acceptable. It is still up to you whether the entries, and any underlying associated data,
							is actually reordered once your reorder callback is executed.
				*/
				virtual void setPotentialReorderGood(const bool good_);

				/**
					\brief		Gets the current reorder callback for the widget
					\author		GW
					\date		12/2022

					\returns	A pointer to the currently registered reorder callback
				*/
				virtual Callback* const reorderCallback() const;
			
				/**
					\brief	Sets the reorder callback for the widget
					\author	GW
					\date	12/2022

					\param	callback_	New callback (target and action) for when a reorder is finalized (via dropping)
					
					\note	When your callback is executed, you can determine which entry is being reordered and its new
							position by calling the reorderSource() and reorderPosition() methods, respectively.
				*/
				virtual void setReorderCallback(const Callback& callback_);
			
				/**
					\brief		Gets the source item involved in the reorder, i.e. the entry being dropped
					\author		GW
					\date		12/2022
				 
					\returns	The ID path of the entry that was reordered
				 
					\note		The return value of this method is only guaranteed to be valid during the execution of
								the reorder callback.
				*/
				virtual std::vector<std::string> reorderSource() const;

				/**
					\brief		Gets the position for the reorder, i.e. the entry just before the reposition
					\author		GW
					\date		12/2022
				 
					\returns	The ID path of the entry just before the drop point for a reposition; if the returned
								vector contains only an empty string, this refers to the drop position being the first
								at the root level.
				 
					\note		The return value of this method is only guaranteed to be valid during the execution of
								the reorder callback.
				*/
				virtual std::vector<std::string> reorderPosition() const;
		};
		
		typedef std::unique_ptr<ColumnView> ColumnViewUP;
		typedef std::shared_ptr<ColumnView> ColumnViewSP;
		typedef std::weak_ptr<ColumnView> ColumnViewWP;
	}
}

#endif
// __HDI_CORE_COLUMN_VIEW__
