/**
	\file
	\brief		Header file for plugin user interface combo box widgets
	\copyright	Hot Door, Inc. 2010-2025
*/

#ifndef __HDI_CORE_COMBO_BOX__
#define __HDI_CORE_COMBO_BOX__

#include <vector>

#include "hdicoreWidget.h"

namespace hdi
{
	namespace core
	{
		class Callback;
		class PopupEntry;

		/**
			\brief	Combo box widget, which consists of a text-field for custom input and a popup menu for preset values
		*/
		class ComboBox : public Widget
		{
			public:
				typedef std::vector<PopupEntry> EntryVector;
			
				/**
					\brief	Text alignment options
				*/
				enum Alignment
				{
					UnknownAlignment		= 0,
					LeftAlignment			= 10,
					CenterAlignment			= 20,
					RightAlignment			= 30
				};

				/**
					\brief	Constructs an empty ComboBox object
					\author	GW
					\date	09/2013
					
					\note	To test if a ComboBox object is empty, call isEmpty() on it
					\note	Empty ComboBox objects do not relate to any actual UI widget; they are designed to be
							"receivers" of some other ComboBox object via the overloaded assignment operator. Empty
							ComboBox objects are useless until such time (though it is safe to call any of their methods).
				*/
				ComboBox();

				/**
					\brief	Constructs a new ComboBox object from an existing ComboBox object (copy constructor)
					\author	GW
					\date	09/2013

					\param	combobox_	Existing ComboBox object
				*/
				ComboBox(const ComboBox& combobox_);

				/**
					\brief	Constructs a combo box
					\author	GW
					\date	09/2013

					\param	loc_	Top-left location of the combo box, in 1x resolution coordinates
					\param	width_	Width of the combo box
				*/
				ComboBox(const Point& loc_, const double width_);

				/**
					\brief	Constructs a combo box
					\author	GW
					\date	09/2013

					\param	loc_			Top-left location of the combo box, in 1x resolution coordinates
					\param	width_			Width of the combo box, at 1x resolution
					\param	entries_		Vector of string entries
					\param	initialIndex_	Initial index to select (must exist in entries_, and must not be a separator)
				*/
				ComboBox(
					const Point& loc_,
					const double width_,
					const EntryVector& entries_,
					const int32_t initialIndex_
				);

				/**
					\brief		Named constructor for a combo box with an initial ID selected
					\author		GW
					\date		09/2013

					\param		loc_		Top-left location of the combo box, in 1x resolution coordinates
					\param		width_		Width of the combo box, at 1x resolution
					\param		entries_	Vector of string entries
					\param		initialID_	Initial ID to select (must exist in entries_, and must not be a separator),
											as UTF-8
					\returns	A ComboBox object with the specified entries and initially selected entry
				*/
				static ComboBox ComboBoxWithID(
					const Point& loc_,
					const double width_,
					const EntryVector& entries_,
					const std::string& initialID_
				);

				/**
					\brief		Named constructor for a combo box with an initial string value (whether it's in the
								entries_ argument or not)
					\author		GW
					\date		09/2013

					\param		loc_		Top-left location of the combo box, in 1x resolution coordinates
					\param		width_		Width of the combo box, at 1x resolution
					\param		entries_	Vector of string entries
					\param		value_		Initial value of combo box, as UTF-8
					\returns	A ComboBox object with the specified entries and initially selected entry
				*/
				static ComboBox ComboBoxWithValue(
					const Point& loc_,
					const double width_,
					const EntryVector& entries_,
					const std::string& value_
				);

				/**
					\brief	ComboBox destructor
					\author	GW
					\date	09/2013
				*/
				virtual ~ComboBox();

				/**
					\brief		Allows one ComboBox object to be assigned from another
					\author		GW
					\date		10/2013

					\param		rhs_	Righthand side of the = operator; the object to copy values from
					\returns	The target ComboBox object, but with its value updated to match that of rhs_
				*/
				virtual ComboBox& operator=(const ComboBox& rhs_);
				
				/**
					\brief		Convenience method to clone a ComboBox object on the heap
					\author		GW
					\date		10/2013

					\returns	A pointer to the new ComboBox object
					
					\note		If you subclass ComboBox, you MUST overload this method yourself! If you don't and/or
								your clone() method does not return an instance of your ComboBox subclass, you will
								experience "object slicing" when adding the widget to a window.
								
					\warning	The caller must manage the memory for the returned ComboBox object.
				*/
				virtual ComboBox* clone() const;
				
				/**
					\brief		Convenience method to duplicate a ComboBox object, creating a new and identical UI
								element to the target (but not belonging to the same containing window)
					\author		GW
					\date		11/2013
					
					\returns	A pointer to the new ComboBox object (and new UI element)
					
					\note		The new combo box will have copies of all entries present in the target, but be aware
								that copies of the user data in each entry cannot be made (because they are simply void*
								variables). As such, the user data in the copies will simply be set to NULL by this
								method.

					\warning	The caller must manage the memory for the returned ComboBox object.
				*/
				virtual ComboBox* duplicate() const;

				/**
					\brief		Gets the current alignment of the text field
					\author		GW
					\date		09/2013

					\returns	Current text alignment
				*/
				virtual Alignment alignment() const;

				/**
					\brief	Sets the current alignment of the text field
					\author	GW
					\date		09/2013

					\param	alignment_	New text alignment
				*/
				virtual void setAlignment(const Alignment alignment_);
			
				/**
					\brief		Gets whether the text is bold
					\author		GW
					\date		09/2013

					\returns	true if the text is bold, false otherwise
				*/
				virtual bool bold() const;

				/**
					\brief	Sets the font weight
					\author	GW
					\date	09/2013

					\param	bold_	true for bold, false otherwise
				*/
				virtual void setBold(const bool bold_);

				/**
					\brief		Gets whether the text is italicized
					\author		GW
					\date		09/2013

					\returns	true if the text is italicized, false otherwise
				*/
				virtual bool italic() const;

				/**
					\brief	Sets the font style
					\author	GW
					\date	09/2013

					\param	italic_		true for italics, false otherwise
				*/
				virtual void setItalic(const bool italic_);

				/**
					\brief		Gets the index of the currently selected entry
					\author		GW
					\date		09/2013

					\returns	The index number of the currently selected entry, or -1 if no entry is selected (the
								combo box has a custom value)
				*/
				virtual int32_t currentIndex() const;
				
				/**
					\brief		Sets the currently selected entry by index
					\author		GW
					\date		09/2013

					\param		index_	New index to select
					\returns	true if the selection occurred successfully, or false if e.g. out of range
				*/
				virtual bool setCurrentIndex(const int32_t index_);
				
				/**
					\brief		Gets the ID of the currently selected entry
					\author		GW
					\date		09/2013

					\returns	UTF-8 ID string of the currently selected entry, or "" if no entry is currently selected
				*/
				virtual std::string currentID() const;
				
				/**
					\brief		Sets the currently selected entry by ID
					\author		GW
					\date		09/2013

					\param		id_		ID of new value to select in the combo box, as UTF-8
					\returns	true if the entry with the given ID was selected, or false if invalid
				*/
				virtual bool setCurrentID(const std::string& id_);
				
				/**
					\brief		Gets the string value of the currently selected entry (or the custom value, if the user
								has entered one)
					\author		GW
					\date		09/2013

					\returns	Value string of the combo box, as UTF-8
				*/
				virtual std::string currentValue() const;
				
				/**
					\brief		Sets the currently selected entry by string value
					\author		GW
					\date		09/2013

					\param		value_	New UTF-8 value to set for the combo box, and (if the same as a predefined value
										from the popup portion of the combo box) the entry to select
					\returns	true if the value was set, false otherwise
				*/
				virtual bool setCurrentValue(const std::string& value_);

				/**
					\brief		Gets the current entry object
					\author		GW
					\date		09/2013

					\returns	The PopupEntry object for the currently selected item, or NULL for none (or a custom value)
				*/
				virtual std::unique_ptr<PopupEntry> currentEntry() const;
				
				/**
					\brief		Gets the user data for the current entry
					\author		GW
					\date		09/2013

					\param		data__	Return-by-reference for the current user data; ignore if false is returned
					\returns	true if the method succeeds in acquiring the contained entry data, false otherwise
				*/
				virtual bool currentUserData(void*& data__) const;

				/**
					\brief		Gets the entries vector
					\author		GW
					\date		09/2013

					\returns	The entries vector for the target widget, containing all currently known entries
					
					\note		Modifying any value in an entry will not cause the widget to update automatically; call
								update() to update the UI for changes made to entry data.
				*/
				virtual const std::vector< std::shared_ptr<PopupEntry> >* const entries() const;
				
				/**
					\brief		Gets whether the entries vector already contains an entry, based on its ID
					\author		GW
					\date		09/2013

					\param		id_		ID of the entry to search for, as UTF-8
					\returns	true if the widget already contains an entry with the given ID, false otherwise
				*/
				virtual bool hasEntry(const std::string& id_) const;
				
				/**
					\brief		Gets the entry with the given ID, provided it exists within the widget
					\author		GW
					\date		02/2023

					\param		id_		ID of the entry to search for
					\returns	The entry with the given ID, or NULL if not found
				*/
				virtual std::shared_ptr<PopupEntry> entryWithID(const std::string& id_) const;
				
				/**
					\brief	Allows the entries in an combo box to be changed out entirely
					\author	GW
					\date	09/2013

					\param	entries_	New set of entries that the widget will contain
				*/
				virtual void setEntries(const EntryVector& entries_);

				/**
					\brief		Pushes an entry onto the list (i.e. at the end)
					\author		GW
					\date		09/2013

					\param		entry_	New entry object to insert into the list of entries
					\returns	true if the entry is pushed successfully, false otherwise

					\note		If the entry is already present in the list (based on its ID), it will not be inserted.
				*/
				virtual bool pushEntry(const PopupEntry& entry_);

				/**
					\brief		Pops an entry off the list (i.e. from the end)
					\author		GW
					\date		09/2013

					\param		entry__		Return-by-reference for the popped entry, so the caller can inspect which
											one was removed
					\returns	true if an entry was popped and acquired, false otherwise
				*/
				virtual bool popEntry(PopupEntry& entry__);

				/**
					\brief		Inserts an entry into the list of entries at a given position
					\author		GW
					\date		09/2013

					\param		id_		Existing entry immediately before the position at which the new entry will be
										inserted, as UTF-8
					\param		entry_	New entry object to insert into the list of entries
					\returns	true if the entry is inserted successfully, false otherwise

					\note		If the entry for id_ doesn't exist in the list, the new entry will not be inserted.
					\note		If the passed entry is already present in the list (based on its ID), it will not be
								inserted.
				*/
				virtual bool insertEntry(const std::string& id_, const PopupEntry& entry_);

				/**
					\brief		Removes a specific entry from the list of entries
					\author		GW
					\date		09/2013

					\param		id_			ID of the entry to be removed, as UTF-8
					\param		entry__		Return-by-reference for the removed entry, so the caller can inspect which
											one was removed
					\returns	true if an entry was removed and acquired, false otherwise
				*/
				virtual bool removeEntry(const std::string& id_, PopupEntry& entry__);

				/**
					\brief	Clears all entries contained by the widget
					\author	GW
					\date	09/2013
				*/
				virtual void clearEntries();

				/**
					\brief		Gets the selection range of the text in the field, if any, in UTF-8
					\author		GW
					\date		05/2023
				 
					\param		pos__	Return-by-reference for the starting position of the selection range
					\param		len__	Return-by-reference for the length of the selection range
					\returns	true if the field is focused and the selection range could be acquired, false otherwise
				 
				 	\warning	This method utilizes UTF-8 byte ranges, which can be useful in certain circumstances. If
				 				you require character ranges instead, use the UTF-32 version of this method.
				 
					\note		If len__ is zero then no text is selected, but pos__ indicates the position of the
								insertion point (text caret).
				*/
				virtual bool selectionRange(uint32_t& pos__, uint32_t& len__) const;
			
				/**
					\brief		Sets the selection range of the text in the field, in UTF-8 (and focuses it)
					\author		GW
					\date		05/2023
				 
					\param		pos_	The starting position of the selection range
					\param		len_	The length of the selection range
				 
				 	\warning	This method utilizes UTF-8 byte ranges, which can be useful in certain circumstances. If
				 				you require character ranges instead, use the UTF-32 version of this method.
				 
					\note		If len_ is zero then no text will be selected, but the insertion point (text caret) will be
								moved to the pos_ value.
				*/
				virtual void setSelectionRange(const uint32_t pos_, const uint32_t len_);

				/**
					\brief		Gets the selection range of the text in the field, if any, in UTF-32
					\author		GW
					\date		05/2023
				 
					\param		pos__	Return-by-reference for the starting position of the selection range
					\param		len__	Return-by-reference for the length of the selection range
					\returns	true if the field is focused and the selection range could be acquired, false otherwise
				 
				 	\warning	This method utilizes UTF-32 character ranges, which can be useful in certain circumstances.
				 				If you require byte ranges instead, use the UTF-8 version of this method.
				 
					\note		If len__ is zero then no text is selected, but pos__ indicates the position of the
								insertion point (text caret).
				*/
				virtual bool selectionRangeUTF32(uint32_t& pos__, uint32_t& len__) const;
			
				/**
					\brief		Sets the selection range of the text in the field, in UTF-32 (and focuses it)
					\author		GW
					\date		05/2023
				 
					\param		pos_	The starting position of the selection range
					\param		len_	The length of the selection range
				 
				 	\warning	This method utilizes UTF-32 character ranges, which can be useful in certain circumstances.
				 				If you require byte ranges instead, use the UTF-8 version of this method.
				 
					\note		If len_ is zero then no text will be selected, but the insertion point (text caret) will be
								moved to the pos_ value.
				*/
				virtual void setSelectionRangeUTF32(const uint32_t pos_, const uint32_t len_);

				/**
					\brief		Gets the last character typed into the field
					\author		GW
					\date		05/2023

					\returns	Last character typed, or "" if none

					\note		The return value of this method is only valid when called from inside a value-changing
								or value-changed callback (and it's very useful to determine if the user pressed e.g. the
								enter or return key)
				*/
				virtual std::string lastCharTyped() const;
				
				/**
					\brief		Gets the last set of modifier keys used when typing into the field
					\author		GW
					\date		05/2023

					\returns	Last set of modifier keys, or NoModifierKey for none

					\note		The return value of this method is only valid when called from inside a value-changing
								or value-changed callback.
					\note		Some modifiers are difficult/impossible to detect here, such as ControlModifierKey on
								Windows. The reason is that such modifiers are usually captured by other parts of the UI
								event loop before the widget is even notified of the key strokes (picture e.g. ctrl+a on
								the Windows to "select all").
				*/
				virtual ModifierKey lastModifierKeys() const;

				/**
					\brief	Simulates the value of the combo box changing
					\author	GW
					\date	01/2017
				*/
				virtual void valueChanging();
				
				/**
					\brief		Gets the value changing callback for the combo box
					\author		GW
					\date		01/2017

					\returns	A pointer to the currently registered value changing callback
				*/
				virtual Callback* const valueChangingCallback() const;
				
				/**
					\brief	Sets the value changing callback
					\author	GW
					\date	09/2013

					\param	callback_	New callback for when the combo box's value is changing
				*/
				virtual void setValueChangingCallback(const Callback& callback_);

				/**
					\brief	Simulates the combo box having its value changed
					\author	GW
					\date	09/2013
				*/
				virtual void valueChanged();
				
				/**
					\brief		Gets the value changed callback for the combo box
					\author		GW
					\date		09/2013

					\returns	A pointer to the currently registered value changed callback
				*/
				virtual Callback* const valueChangedCallback() const;
				
				/**
					\brief	Sets the value changed callback
					\author	GW
					\date	09/2013

					\param	callback_	New callback for when the combo box's value has changed
				*/
				virtual void setValueChangedCallback(const Callback& callback_);
				
				/**
					\brief	Force the combo box to update its entries
					\author	GW
					\date	12/2013
				*/
				virtual void update();
				
				/**
					\brief		Gets the expansion state of the combo box's popup menu
					\author		GW
					\date		05/2023
					
					\returns	true if the popup is showing, false otherwise
				*/
				virtual bool expanded() const;
				
				/**
					\brief	Sets the expansion state of the combo box's popup menu
					\author	GW
					\date	05/2023
					
					\param	exp_	true to show the popup, false otherwise
				*/
				virtual void setExpanded(const bool exp_);
		};
		
		typedef std::unique_ptr<ComboBox> ComboBoxUP;
		typedef std::shared_ptr<ComboBox> ComboBoxSP;
		typedef std::weak_ptr<ComboBox> ComboBoxWP;
	}
}

#endif
// __HDI_CORE_COMBO_BOX__
