/**
	\file
	\brief		Header file for a class to create and manipulate Illustrator control bars
	\copyright	Hot Door, Inc. 2010-2025
*/

#ifndef __HDI_CORE_CONTROLBAR__
#define __HDI_CORE_CONTROLBAR__

#if defined(HDI_CORE_AIP_MODE)

#include "hdicoreRect.h"
#include "hdicoreWindow.h"

namespace hdi
{
	namespace core
	{
		class Callback;
		class Widget;

		typedef	AIControlBarRef PlatformControlBarPtr;

		/**
			\brief	Used to add a control bar to Illustrator's UI and contain various UI widgets
		*/
		class ControlBar : public Window
		{
			public:
				/**
					\brief	Most Illustrator control bars are this height, at 1x resolution
				*/
				static const double typicalHeight;

				/**
					\brief	Constructs an empty ControlBar object
					\author	GW
					\date	10/2013
					
					\note	To test if a ControlBar object is empty, call isEmpty() on it
					\note	Empty ControlBar objects do not relate to any actual UI control bar; they are designed to be
							"receivers" of some other ControlBar object via the overloaded assignment operator. Empty
							ControlBar objects are useless until such time (though it is safe to call any of their
							methods).
				*/
				ControlBar();

				/**
					\brief	Constructs a new ControlBar object from an existing ControlBar object (copy constructor)
					\author	GW
					\date	10/2013

					\param	cb_		Existing ControlBar object
				*/
				ControlBar(const ControlBar& cb_);

				/**
					\brief	Constructs a new ControlBar object
					\author	GW
					\date	09/2013

					\param	frame_		Location and size (width and height) of the control bar, at 1x resolution
					\param	minWidth_	Minimum width of the control bar, at 1x resolution
					\param	maxWidth_	Maximum width of the control bar, at 1x resolution
					
					\note	Once the size is set by the constructor, only the width can be changed thereafter
				*/
				ControlBar(const Rect& frame_, const double minWidth_, const double maxWidth_);

				/**
					\brief	Destructs a ControlBar object
					\author	GW
					\date	09/2013
				*/
				virtual ~ControlBar();

				/**
					\brief		Allows one ControlBar object to be assigned from another
					\author		GW
					\date		10/2013

					\param		rhs_	Righthand side of the = operator; the object to copy values from
					\returns	The target ControlBar object, but with its value updated to match that of rhs_
				*/
				virtual ControlBar& operator=(const ControlBar& rhs_);
				
				/**
					\brief		Convenience method to clone a ControlBar object on the heap
					\author		GW
					\date		10/2013

					\returns	A pointer to the new ControlBar object
					
					\note		If you subclass ControlBar, you MUST overload this method yourself! If you don't and/or
								your clone() method does not return an instance of your ControlBar subclass, you will
								experience "object slicing" when e.g. adding the control bar to a window collection.

					\warning	The caller must manage the memory for the returned ControlBar object.
				*/
				virtual ControlBar* clone() const;

				/**
					\brief	Destroys the UI control bar, converting the target object to an empty ControlBar object (see
							the default constructor for more info on empty ControlBar objects)
					\author	GW
					\date	10/2013

					\note	This method must be called for any existing control bars by the time the plugin is shutting
							down. If you no longer retain any of your own references to a control bar then the lib will
							call this method for you as it is shutting down.
				*/
				virtual void destroy();

				/**
					\brief		Gets the platform-specific control bar, around which the target object is wrapped
					\author		GW
					\date		11/2013

					\returns	Pointer to platform-specific control bar. See PlatformControlBarPtr typedef.

					\note		Generally, you shouldn't need to call this method. Only use it if you know what you're
								doing. If a specific piece of UI functionality is not handled by this class (or related
								classes), then it should probably be added to the hdi_core library.
				*/
				virtual PlatformControlBarPtr platformControlBar() const;

				/**
					\brief	Shows the control bar
					\author	GW
					\date	09/2013
				*/
				virtual void show();

				/**
					\brief	Hides the control bar
					\author	GW
					\date	09/2013
				*/
				virtual void hide();

				/**
					\brief		Gets whether the control bar is visible
					\author		GW
					\date		09/2013

					\returns	true if the control bar is visible, false otherwise
				*/
				virtual bool visible() const;

				/**
					\brief	Sets control bar visibility
					\author	GW
					\date	09/2013

					\param	visible_	true to show the control bar, false to hide

					\note	This method simply calls show() or hide() as appropriate
				*/
				virtual void setVisible(const bool visible_);

				/**
					\brief		Gets the current width of the control bar
					\author		GW
					\date		09/2013

					\returns	The current width of the target control bar, at 1x resolution
				*/
				virtual double width() const;

				/**
					\brief	Sets the width of the control bar
					\author	GW
					\date	09/2013

					\param	w_	New width in pixels, at 1x resolution
				*/
				virtual void setWidth(const double w_);

				/**
					\brief	Moves the control bar
					\author	GW
					\date	09/2013

					\param	loc_	New top-left position for the control bar, in 1x resolution coordinates
				*/
				virtual void move(const Point& loc_);

				/**
					\brief		Adds a widget (label, button, text field, etc.) to the control bar
					\author		GW
					\date		09/2013

					\param		widget_		Widget to place in the control bar
					\returns	true if the widget was added successfully
				*/
				virtual bool addWidget(const Widget& widget_);

				/**
					\brief		Removes a widget (label, button, text field, etc.) from the control bar
					\author		GW
					\date		09/2013

					\param		widget_		Widget to remove from the control bar
					\returns	true if the widget was removed successfully
				*/
				virtual bool removeWidget(const Widget& widget_);

				/**
					\brief		Gets the registered callback for the control bar showing, or NULL if none
					\author		GW
					\date		12/2015

					\returns	A pointer to the currently registered "control bar show" callback
				*/
				virtual Callback* const showCallback() const;

				/**
					\brief	Sets/updates the callback for the control bar showing
					\author	GW
					\date	12/2015

					\param	cb_		New callback for a user showing the control bar
				*/
				virtual void setShowCallback(const Callback& cb_);

				/**
					\brief		Gets the registered callback for the control bar hiding, or NULL if none
					\author		GW
					\date		12/2015

					\returns	A pointer to the currently registered "control bar hide" callback
				*/
				virtual Callback* const hideCallback() const;

				/**
					\brief	Sets/updates the callback for the control bar hiding
					\author	GW
					\date	12/2015

					\param	cb_		New callback for a user hiding the control bar
				*/
				virtual void setHideCallback(const Callback& cb_);

				/**
					\brief		Gets the callback set for the control bar resize, or NULL if none
					\author		GW
					\date		12/2015

					\returns	A pointer to the currently registered "control bar resize" callback
				*/
				virtual Callback* const resizeCallback() const;

				/**
					\brief	Sets/updates the callback for the control bar resize
					\author	GW
					\date	12/2015

					\param	cb_		New callback for a user resizing the control bar
				*/
				virtual void setResizeCallback(const Callback& cb_);

				/**
					\brief	Forces the widgets in the control bar to update/redraw
					\author	GW
					\date	09/2013
				*/
				virtual void update() const;
		};
		
		typedef std::unique_ptr<ControlBar> ControlBarUP;
		typedef std::shared_ptr<ControlBar> ControlBarSP;
		typedef std::weak_ptr<ControlBar> ControlBarWP;
	}
}

#endif
// HDI_CORE_AIP_MODE

#endif
// __HDI_CORE_CONTROLBAR__
