/**
	\file
	\brief		Header file for a class to manipulate the current Illustrator document
	\copyright	Hot Door, Inc. 2010-2025
*/

#ifndef __HDI_CORE_CURRENT_DOCUMENT__
#define __HDI_CORE_CURRENT_DOCUMENT__

#if defined(HDI_CORE_AIP_MODE)

#include <vector>

#include "hdicoreTypes.h"
#include "hdicoreArtboardPoint.h"
#include "hdicoreArtboardRect.h"
#include "hdicoreFileFormat.h"
#include "hdicorePathStyle.h"
#include "hdicoreSize.h"

namespace hdi
{
	namespace core
	{
		class Art;
		class Artboard;
		class ArtStyle;
		class DocumentView;
		class Font;
		class Grid;
		class Layer;
		class Pattern;
		class Symbol;
		class TextRangeSet;
		class TextStory;
	
		namespace ai
		{
			class Dictionary;
		}

		/**
			\brief	Current (focused) document class to allow for metadata access, updating, dictionary access, selected
					art access, layer access, etc.
		*/
		class CurrentDocument
		{
			public:
				typedef std::vector< std::shared_ptr<Art> > ArtVector;
				typedef std::vector<MatchArtSpec> MatchArtSpecVector;
			
				/**
					\brief	Describes the setup options of an existing document
				*/
				class Setup
				{
					public:
						/**
							\brief	Document dimensions
						*/
						Size size;

						/**
							\brief	Indicates whether placed images are shown
						*/
						bool showPlacedImages;

						/**
							\brief	Bezier resolution in dpi
						*/
						double resolution;

						/**
							\brief	Controls whether long paths are split. Which paths are affected also depends on the
									bezier resolution.
						*/
						bool splitLongPaths;

						/**
							\brief	Controls whether multiple tiled pages are printed (or just a single page)
						*/
						bool printTiles;
						
						/**
							\brief	Ignored when printTiles is false; otherwise, controls whether many tiles are created
									to print the entire artboard (or just the imageable area of a page)
						*/
						bool tileFullPages;
						
						/**
							\brief	Constructs a new Setup object with default values
							\author	GW
							\date	08/2013
						*/
						Setup();
						
						/**
							\brief	Destructs a Setup object
							\author	GW
							\date	08/2013
						*/
						virtual ~Setup();
				};
		
				typedef std::unique_ptr<Setup> SetupUP;
				typedef std::shared_ptr<Setup> SetupSP;
				typedef std::weak_ptr<Setup> SetupWP;

				/**
					\brief		Gets the built-in document data dictionary object for the current document
					\author		GW
					\date		08/2013

					\returns	The Dictionary object for the current document
				*/
				std::unique_ptr<ai::Dictionary> dictionary();

				/**
					\brief		Gets the document handle around which the target object is wrapped
					\author		GW
					\date		11/2013
					
					\returns	AIDocumentHandle for this object

					\note		Generally, you shouldn't need to call this method. Only use it if you know what you're
								doing. If a specific piece of functionality provided by Illustrator is not handled by
								this class (or related classes), then it should probably be added to the hdi_core library.
				*/
				AIDocumentHandle aiDocumentHandle() const;

				/**
					\brief		Gets the platform-specific window handle for the document
					\author		GW
					\date		02/2015

					\returns	Pointer to platform-specific window. See PlatformDocWindowPtr typedef.

					\warning	On Mac, the caller must perform a release when they are done with the returned object.
				 
					\note		Generally, you shouldn't need to call this method. Only use it if you know what you're
								doing. If a specific piece of UI functionality is not handled by this class (or related
								classes), then it should probably be added to the hdi_core library.
				*/
				PlatformDocWindowPtr platformWindow() const;
				
				/**
					\brief		Gets the complete path to the current document
					\author		GW
					\date		08/2013

					\returns	The URL (location) of the target document, as UTF-8

					\note		In the case of a new (never saved) document, this simply returns the file name
				*/
				std::string filePath() const;
				
				/**
					\brief		Gets the name of the current document
					\author		GW
					\date		08/2013

					\returns	The name of the target document, as UTF-8
				*/
				std::string fileName() const;

				/**
					\brief		Gets the max bounds rect for the current document
					\author		GW
					\date		08/2013

					\returns	Maximum bounds for the target document
				*/
				ArtboardRect maxBounds() const;
			
				/**
					\brief		Gets the scale of the current doc
					\author		GW
					\date		12/2019
				 
					\returns	The current doc scale
				 
					\note		Use this method to determine whether a document is in large canvas mode. As of Illustrator
								24.2, this method will return 10.0 for large canvas mode (and of course 1.0 for "regular"
								canvas mode).
				*/
				double scale() const;

				/**
					\brief		Gets the ruler units for the current document
					\author		GW
					\date		08/2013

					\returns	The current ruler units for the target document
				*/
				RulerUnits rulerUnits() const;

				/**
					\brief		Gets the setup options for the current document
					\author		GW
					\date		08/2013

					\returns	The current setup options for the target document
				*/
				std::unique_ptr<Setup> setup() const;

				/**
					\brief		Gets whether the current document is in text editing mode
					\author		GW
					\date		08/2013

					\returns	true if the target document has text editing focus, false otherwise
				*/
				bool hasTextFocus() const;
				
				/**
					\brief		Gets the story that has editing focus (i.e. has some selected range)
					\author		GW
					\date		10/2015
					
					\returns	The focused story, or NULL for none/error
				*/
				std::unique_ptr<TextStory> focusedText() const;
				
				/**
					\brief		Sets the story that should have editing focus, and the editing caret is placed at the
								start of said story
					\author		GW
					\date		10/2015
					
					\param		story_	Story to put into editing mode
					\returns	true if the story was focused, false otherwise
				*/
				bool setFocusedText(const TextStory& story_);
				
				/**
					\brief	Ends text editing mode for the current document
					\author	GW
					\date	10/2015
				*/
				void loseTextFocus();
				
				/**
					\brief		Gets the set of selected text ranges for the current document
					\author		GW
					\date		10/2015
					
					\returns	The selected text ranges (see notes)
					
					\note		The text selection can describe multiple discontiguous ranges of text across multiple art
								objects. If the document is in text editing mode, the result is a range representing the
								caret or the user selection. If the document is not in text editing mode, there might be
								multiple ranges for the selected text frame object(s).
				*/
				std::unique_ptr<TextRangeSet> selectedText() const;

				/**
					\brief		Gets whether the current document has been modified
					\author		GW
					\date		08/2013

					\returns	true if the target document has been edited since it was last saved, or false otherwise
				*/
				bool modified() const;

				/**
					\brief	Sets whether the current document is in a modified state
					\author	GW
					\date	08/2013

					\param	mod_	true to put the document in a modified state, false otherwise

					\note	This method is useful if you've made some small/internal change to the document that you do
							not wish the user to become aware of
				*/
				void setModified(const bool mod_);

				/**
					\brief	Updates cached artwork properties for the current document
					\author	GW
					\date	08/2013

					\note	In general, when Illustrator behaves in a mildly erroneous (but consistent) manner, e.g.
							artwork not selecting properly under certain repeatable circumstances, calling this method
							(either before or after the call that introduced the erroneous behavior) will sometimes
							resolve the problem. The only people that know why are on the Illustrator team...
				*/
				void sync() const;

				/**
					\brief	Forces the art in the current document to be redrawn
					\author	GW
					\date	08/2013

					\note	With large documents, this method can have a significant performance hit.
				*/
				void redraw() const;
				
				/**
					\brief	Updates cached artwork properties for the current document, synchronizes the current context,
							and forces the art in the current document to be redrawn
					\author	GW
					\date	09/2014
					
					\note	It is well known that after creating text art objects anew and setting their contents, this
							method must be called thereafter if you would like to select the text art objects before
							your plugin's context exits. Otherwise, you would have to select the text after a delay (i.e.
							in a timer) of some sort.
					\note	Do not confuse this with the sync() and redraw() methods, as the underlying Illustrator
							suite call for this method is not the same as the others. It is entirely possible that the
							underlying suite call performs additional tasks over calling sync() and redraw() together,
							but the only people that know for sure are on the Illustrator team...
				*/
				void syncAndRedraw() const;

				/**
					\brief		Saves the current document, prompting the user if it has not been saved before
					\author		GW
					\date		10/2013
					
					\returns	true if the document was saved, false otherwise
				*/
				bool save() const;
				
				/**
					\brief		Saves the current document with the given parameters, even if the document has not been
								saved before (and without affecting the modification status of the document)
					\author		GW
					\date		12/2014
					
					\param		path_		Path, as UTF-8, to which the document should be saved
					\param		ff_			File format to use when writing the document to disk
					\param		prompt_		Whether to prompt the user for configuration of the file before writing
					\returns	true if the document was written to disk successfully
					
					\note		If inadequate formatting parameters are available, then the user will be prompted for
								configuration even if prompt_ is false.
				*/
				bool save(const std::string& path_, const FileFormat& ff_, const bool prompt_ = true) const;
				
				/**
					\brief		Saves the current document with the given parameters and options, even if the document
								has not been saved before (and without affecting the modification status of the document)
					\author		GW
					\date		12/2014
					
					\param		path_		Path, as UTF-8, to which the document should be saved
					\param		ff_			File format to use when writing the document to disk
					\param		opt_		Bitwise OR of write options
					\param		prompt_		Whether to prompt the user for configuration of the file before writing
					\returns	true if the document was written to disk successfully
					
					\note		If inadequate formatting parameters are available, then the user will be prompted for
								configuration even if prompt_ is false.
				*/
				bool save(
					const std::string& path_,
					const FileFormat& ff_,
					const FileFormat::Options opt_,
					const bool prompt_ = true
				) const;

				/**
					\brief		Closes the current document
					\author		GW
					\date		10/2013
					
					\returns	true if the document was closed, false otherwise
				*/
				bool close() const;

				/**
					\brief	Undoes the most recent undoable transaction for the current document (same as the user using
							the undo menu item)
					\author	GW
					\date	11/2015

					\note	Use only when this plug-in is issuing a command and is entirely in control of that command.
							Do not call when responding to messages invoked by other commands, such as save, close,
							update, and so on.
				*/
				void undo() const;
			
				/**
					\brief	Redoes the most recent redoable transaction for the current document (same as the user using
							the redo menu item)
					\author	GW
					\date	11/2015

					\note	Use only when this plug-in is issuing a command and is entirely in control of that command.
							Do not call when responding to messages invoked by other commands, such as save, close,
							update, and so on.
				*/
				void redo() const;

				/**
					\brief	Performs a cut operation, removing any selected art objects from the document and placing
							them in the clipboard
					\author	GW
					\date	04/2021

					\note	Use only when this plug-in is issuing a command and is entirely in control of that command.
							Do not call when responding to messages invoked by other commands, such as save, close,
							update, and so on.
				*/
				void cut();
			
				/**
					\brief	Performs a copy operation, placing any selected art objects from the current document in the
							clipboard
					\author	GW
					\date	04/2021

					\note	Use only when this plug-in is issuing a command and is entirely in control of that command.
							Do not call when responding to messages invoked by other commands, such as save, close,
							update, and so on.
				*/
				void copy();
			
				/**
					\brief	Pastes the contents of the clipboard on the current document
					\author	GW
					\date	04/2021

					\note	Use only when this plug-in is issuing a command and is entirely in control of that command.
							Do not call when responding to messages invoked by other commands, such as save, close,
							update, and so on.
				*/
				void paste();

				/**
					\brief		Gets whether any art is selected at all in the document
					\author		GW
					\date		12/2015
					
					\returns	true for any art being selected, false otherwise
					
					\note		This method is much faster for large documents with large selections than getting all of
								the currently selected art and simply checking if the vector is not empty.
				*/
				bool anyArtSelected() const;

				/**
					\brief		Gets a vector of all currently selected art in the document
					\author		GW
					\date		08/2013

					\returns	An artwork vector of all currently selected art
				*/
				ArtVector selectedArt() const;

				/**
					\brief	Deselects all currently selected art
					\author	GW
					\date	08/2013
				*/
				void deselectAllArt() const;

				/**
					\brief		Gets a vector of art in the document whose specs match those given
					\author		GW
					\date		08/2013

					\param		specs_	Vector of match specifications
					\returns	An artwork vector whose entries match the provided specifications

					\note		This method takes a vector of specs so the caller can specify a multitude of artwork
								properties, thereby acquiring a vector of artwork that (currently) exhibit all of the
								provided properties.
				*/
				ArtVector matchingArt(const MatchArtSpecVector& specs_) const;

				/**
					\brief		Gets a vector of art in the document whose type matches the given type
					\author		GW
					\date		08/2013

					\param		type_	Logical OR of types of art to find
					\returns	An artwork vector whose entries match the provided art type(s)
				*/
				ArtVector artOfType(const MatchArtType type_) const;

				/**
					\brief		Gets the path style of the currently selected art in the document
					\author		GW
					\date		08/2013

					\returns	A path style representing the style of the currently selected art
				*/
				PathStyle selectedArtPathStyle() const;

				/**
					\brief		Gets the path style and map of the currently selected art in the document
					\author		GW
					\date		08/2013

					\param		style__		Return-by-reference for the path style of the currently selected art
					\param		map__		Return-by-reference for a path style map, indicating which path style
											properties are fully known (or unknown)
					\returns	true if the style and map could be acquired, false otherwise

					\note		In general, the purpose of a map is to indicate which values are known/common across all
								target artwork. For example, if two pieces of art were selected, one with a stroke and
								the other with no stroke, then the map would indicate that the stroke is "unknown". If
								the same pieces of both had a fill, then the map would indicate that the fill is "known".
				*/
				bool selectedArtPathStyle(PathStyle& style__, PathStyle::Map& map__) const;

				/**
					\brief	Sets the path style of the currently selected art in the document
					\author	GW
					\date	08/2013

					\param	style_	New path styling for the currently selected art
				*/
				void setSelectedArtPathStyle(const PathStyle& style_);

				/**
					\brief	Sets the path style of the currently selected art in the document, taking a map into account
					\author	GW
					\date	08/2013

					\param	style_	New path styling for the currently selected art
					\param	map_	Style map, indicating which path style properties are fully known (or unknown)
				*/
				void setSelectedArtPathStyle(const PathStyle& style_, const PathStyle::Map& map_);
				
				/**
					\brief		Gets a count of the views in the current document
					\author		GW
					\date		08/2013

					\returns	A count of the number of views in the current document
				*/  
				uint32_t viewCount() const;

				/**
					\brief		Gets a given document view, by its index, in the current document
					\author		GW
					\date		08/2013

					\param		index_	Index number of the document view in question
					\returns	The view in question, or NULL for none
				*/
				std::unique_ptr<DocumentView> viewAtIndex(const uint32_t index_) const;

				/**
					\brief		Gets the current view for the current document
					\author		GW
					\date		08/2013

					\returns	The current document view, or NULL if none/error
				*/
				std::unique_ptr<DocumentView> currentView() const;

				/**
					\brief		Gets a count of the layers in the current document
					\author		GW
					\date		08/2013

					\returns	A count of the number of layers in the current document
				*/   
				uint32_t layerCount() const;

				/**
					\brief		Gets the first layer in the list in the current document
					\author		GW
					\date		08/2013

					\returns	The Layer object for the first layer in the document
				*/
				std::unique_ptr<Layer> firstLayer() const;

				/**
					\brief		Gets the last layer in the list in the current document
					\author		GW
					\date		08/2013

					\returns	The Layer object for the last layer in the document
				*/
				std::unique_ptr<Layer> lastLayer() const;

				/**
					\brief		Gets a given layer, by its name, in the current document (case-insensitive)
					\author		GW
					\date		08/2013

					\param		title_		Title of the layer in question, as UTF-8
					\returns	The layer in question, or NULL for none
				*/
				std::unique_ptr<Layer> layerByTitle(const std::string& title_) const;

				/**
					\brief		Gets a given layer, by its index, in the current document
					\author		GW
					\date		08/2013

					\param		index_		Index number of the layer in question
					\returns	The layer in question, or NULL for none
				*/
				std::unique_ptr<Layer> layerAtIndex(const uint32_t index_) const;

				/**
					\brief		Gets the currently selected layer of the current document
					\author		GW
					\date		08/2013

					\returns	A Layer object for the currently selected layer, or NULL if none
				*/
				std::unique_ptr<Layer> currentLayer() const;

				/**
					\brief	Sets the currently selected layer in the current document
					\author	GW
					\date	08/2013

					\param	layer_	The layer to select
				*/
				void setCurrentLayer(const Layer& layer_);
				
				/**
					\brief		Gets whether the current document is in isolation mode
					\author		GW
					\date		08/2013

					\returns	true if in isolation mode, false otherwise
				*/
				bool inIsolationMode() const;
				
				/**
					\brief		Gets the currently isolated art
					\author		GW
					\date		08/2013

					\returns	Art object for the currently isolated art
				*/
				std::unique_ptr<Art> isolatedArt() const;
			
				/**
					\brief		Enters isolation mode of the given art object or one of its parents
					\author		GW
					\date		11/2015
					
					\param		art_			Art to be isolated
					\param		hideOtherArt_	true to make non-isolated invisible, false to dim it instead
					\returns	true for successful isolation, false otherwise
					
					\note		When isolating a piece of art, this will look up the parent lineage and automatically
								isolate the first parent that can be isolated (as Illustrator does not allow "leaf art",
								such as paths, to be isolated).
				*/
				bool enterIsolationMode(const Art& art_, const bool hideOtherArt_);
				
				/**
					\brief	Ends isolation mode in the current document
					\author	GW
					\date	08/2013

					\note	This method is safe to call if not currently in isolation mode
				*/
				void exitIsolationMode() const;
			
				/**
					\brief		Gets whether the current document is in pattern editing mode (similar to isolation mode,
								but for editing a pattern in-place)
					\author		GW
					\date		11/2015

					\returns	true if in pattern editing mode, false otherwise
					
					\note		When a document is in pattern editing mode, the inIsolationMode() method will also return
								true. To determine if in pattern editing mode, use this method instead.
				*/
				bool inPatternEditingMode() const;
			
				/**
					\brief		Enters pattern editing mode for the current document
					\author		GW
					\date		11/2015
					
					\param		toEdit_		The pattern to be edited
					\returns	true if pattern editing mode was entered successfully, false otherwise
				*/
				bool enterPatternEditingMode(const Pattern& toEdit_);
			
				/**
					\brief		Ends pattern editing mode, applying the changes made for the given pattern
					\author		GW
					\date		11/2015
					
					\param		editPatt_	The pattern currently being edited
					\returns	true if editPatt_ is actually being edited and the changes were applied
					
					\note		This method is safe to call if not currently in isolation mode
					\note		If you want to exit pattern editing mode without saving any of the changes made, call the
								exitPatternEditingMode() method instead.
				*/
				bool applyPatternEditingModeChanges(const Pattern& editPatt_);
			
				/**
					\brief	Ends pattern editing mode in the current document, discarding any changes made to the pattern
					\author	GW
					\date	11/2015

					\note	This method is safe to call if not currently in pattern editing mode
				*/
				void exitPatternEditingMode() const;
			
				/**
					\brief		Gets whether the current document is in symbol editing mode (similar to isolation mode,
								but for editing a prototype of the symbol source art in-place)
					\author		GW
					\date		11/2015

					\returns	true if in symbol editing mode, false otherwise
					
					\note		When a document is in symbol editing mode, the inIsolationMode() method will also return
								true. To determine if in symbol editing mode, use this method instead.
				*/
				bool inSymbolEditingMode() const;
			
				/**
					\brief		Enters symbol editing mode for the current document, creating a prototype (i.e. temporary
								copy) of the symbol source art for in-place editing
					\author		GW
					\date		11/2015
					
					\param		toEdit_		The symbol to be edited
					\param		inst_		An optional instance of the symbol
					\returns	true if symbol editing mode was entered successfully, false otherwise
					
					\note		If inst_ is NULL, then editing will take place in the center of the artboard. Otherwise,
								editing will take place wherever the instance is located.
				*/
				bool enterSymbolEditingMode(const Symbol& toEdit_, const Art* const inst_ = NULL);
			
				/**
					\brief		Ends symbol editing mode, applying the changes made for the given symbol prototype to
								the symbol source art
					\author		GW
					\date		11/2015
					
					\param		editSym_	The symbol currently being edited
					\returns	true if editSym_ is actually being edited and the changes were applied
					
					\note		This method is safe to call if not currently in isolation mode
					\note		If you want to exit symbol editing mode without saving any of the changes made to the
								prototype, call the exitSymbolEditingMode() method instead.
				*/
				bool applySymbolEditingModeChanges(const Symbol& editSym_);
			
				/**
					\brief	Ends symbol editing mode in the current document, discarding any changes made to the symbol
					\author	GW
					\date	11/2015

					\note	This method is safe to call if not currently in symbol editing mode
				*/
				void exitSymbolEditingMode() const;

				/**
					\brief		Gets the number of artboards in the document
					\author		GW
					\date		08/2013

					\returns	A count of the number of artboards in the current document
				*/
				uint32_t artboardCount() const;

				/**
					\brief		Gets a given artboard, by its name, in the current document (case-insensitive)
					\author		GW
					\date		01/2017

					\param		title_		Title of the artboard in question, as UTF-8
					\returns	The artboard in question, or NULL for none
				*/
				std::unique_ptr<Artboard> artboardByTitle(const std::string& title_) const;
				
				/**
					\brief		Gets a given artboard, by its index, in the current document
					\author		GW
					\date		08/2013

					\param		index_	Index number of the artboard in question
					\returns	Artboard object with the specified index
				*/
				std::unique_ptr<Artboard> artboardAtIndex(const uint32_t index_) const;

				/**
					\brief		Inserts a new artboard at a given position in the artboard list in the current
								document
					\author		GW
					\date		08/2013

					\param		index_	Index at which to create the new artboard
					\param		ab__	Return-by-reference for the new Artboard object at the specified index
					\returns	true if the artboard was created and inserted, false otherwise
				*/
				bool insertArtboard(const uint32_t index_, Artboard& ab__) const;

				/**
					\brief		Gets the current (active) artboard in the document
					\author		GW
					\date		08/2013

					\returns	An Artboard object for the currently selected artboard, or NULL if none
				*/
				std::unique_ptr<Artboard> currentArtboard() const;

				/**
					\brief	Sets the currently selected artboard in the current document
					\author	GW
					\date	08/2013

					\param	artboard_	The artboard to select
				*/
				void setCurrentArtboard(const Artboard& artboard_);
				
				/**
					\brief		Gets the number of art styles in the document
					\author		GW
					\date		11/2013

					\returns	A count of the number of art styles in the current document
				*/
				uint32_t artStyleCount() const;
				
				/**
					\brief		Gets a given art style, by its index, in the current doc
					\author		GW
					\date		11/2013

					\param		index_	Index number of the style in question
					\returns	The ArtStyle object with the specified index
				*/
				std::unique_ptr<ArtStyle> artStyleAtIndex(const uint32_t index_) const;
				
				/**
					\brief		Gets the number of patterns in the document
					\author		GW
					\date		11/2013

					\returns	A count of the number of patterns in the current document
				*/
				uint32_t patternCount() const;
				
				/**
					\brief		Gets a given pattern, by its index, in the current doc
					\author		GW
					\date		11/2013

					\param		index_	Index number of the pattern in question
					\returns	The Pattern object with the specified index
				*/
				std::unique_ptr<Pattern> patternAtIndex(const uint32_t index_) const;
				
				/**
					\brief		Gets the number of symbol definitions in the document
					\author		GW
					\date		11/2013

					\returns	A count of the number of symbol definitions in the current document
				*/
				uint32_t symbolDefCount() const;
				
				/**
					\brief		Gets a given symbol definition, by its index, in the current doc
					\author		GW
					\date		11/2013

					\param		index_	Index number of the symbol def in question
					\returns	The Symbol object with the specified index
				*/
				std::unique_ptr<Symbol> symbolDefAtIndex(const uint32_t index_) const;

				/**
					\brief		Gets the grid object for the current document
					\author		GW
					\date		08/2013

					\returns	A Grid object for the grid of the current document
				*/
				Grid* const grid() const;
			
				/**
					\brief		Gets the color model for the current document
					\author		GW
					\date		08/2013

					\returns	Current color model for the target document
				*/
				DocumentColorModel colorModel() const;
				
				/**
					\brief		Retrieves the default path style that tools should use when creating new objects other
								than text objects
					\author		GW
					\date		08/2014
					
					\returns	The default path style
				*/
				PathStyle defaultPathStyle() const;
				
				/**
					\brief	Sets the default path style that tools should use when creating new objects other than text
							objects
					\author	GW
					\date	08/2014
					
					\param	style_	New default path style
				*/
				void setDefaultPathStyle(const PathStyle& style_);
				
				/**
					\brief		Retrieves the default path style that tools should use when creating text objects
					\author		GW
					\date		08/2014
					
					\returns	The default text style
				*/
				PathStyle defaultTextStyle() const;
				
				/**
					\brief	Sets the default path style that tools should use when creating text objects
					\author	GW
					\date	08/2014
					
					\param	style_	New default text style
				*/
				void setDefaultTextStyle(const PathStyle& style_);
			
				/**
					\brief		Gets the current font for the document
					\author		GW
					\date		04/2016
					
					\returns	The current document's current font, or NULL for none/error
				*/
				std::unique_ptr<Font> currentFont() const;
			
				/**
					\brief	Sets the current font for the document
					\author	GW
					\date	04/2016
					
					\param	font_	New font object for the document
				*/
				void setCurrentFont(const Font& font_);

				/**
					\brief		Gets a vector of all art found at a given point on the document
					\author		GW
					\date		08/2013

					\param		pt_				Point at which to check for art
					\param		request_		As art is found with hit requests, this specifies what art to look for
					\param		tolerance_		Point radius around pt_ in which a hit is valid
					\param		adjustToZoom_	Adjust the tolerance_ argument to take document zoom into account
					\returns	An artwork vector whose entries match the provided specs, in order from the top-most art
								found to the bottom-most

					\note		If adjustToZoom_ is true, then the tolerance_ argument specifies the point radius around
								pt_ at 100% document zoom. If the document is zoomed to 200%, tolerance_ will be
								adjusted to 1/2 the given value; if zoomed to 400%, tolerance_ will be adjusted to 1/4
								the given value; etc.
				*/
				ArtVector artAtPoint(
					const ArtboardPoint& pt_,
					const HitRequest request_,
					const double tolerance_ = 2.0,
					const bool adjustToZoom_ = true
				) const;


			private:
				// Only the Illustrator class can construct a CurrentDocument object
				friend class Illustrator;

				/**
					\brief	Private implementation data
				*/
				void* __data;

				/**
					\brief	Internal use only
					\author	GW
					\date	08/2013
				*/
				CurrentDocument();
				
				/**
					\brief	Unused
					\author	GW
					\date	01/2015
				*/
				CurrentDocument(const CurrentDocument&);

				/**
					\brief	Internal use only
					\author	GW
					\date	08/2013
				*/
				~CurrentDocument();
				
				/**
					\brief	Unused
					\author	GW
					\date	01/2015
				*/
				CurrentDocument& operator=(const CurrentDocument&);
		};
	}
}

#endif
// HDI_CORE_AIP_MODE

#endif
// __HDI_CORE_CURRENT_DOCUMENT__
