/**
	\file
	\brief		Header file for manipulating cursors inside of a plugin
	\copyright	Hot Door, Inc. 2010-2025
*/

#ifndef __HDI_CORE_CURSOR__
#define __HDI_CORE_CURSOR__

#include "hdicorePoint.h"
#include "hdicoreTypes.h"

namespace hdi
{
	namespace pui
	{
		class Cursor;
	}

	namespace core
	{
		/**
			\brief	Represents a cursor that can be shown on the screen, as well as metadata about the current cursor
		*/
		class Cursor
		{
			public:
				/**
					\brief		Default image ID to indicate that no image should be used

					\warning	Do not use 0 as your own PNGI image ID!
				*/
				static const int16_t noneImageID = 0;
			
				/**
					\brief	Describes the built-in Adobe cursors that can be acquired
				*/
				enum AdobeCursor
				{
					BrushCursor						= 10,
					CrosshairsCursor				= 20,
					CrosshairsAddCursor				= 30,
					CrosshairsAnchorCursor			= 40,
					CrosshairsCircleBoldCursor		= 50,
					CrosshairsCircleBrokenCursor	= 60,
					CrosshairsCircleWholeCursor		= 70,
					CrosshairsCloseCursor			= 80,
					CrosshairsContinueCursor		= 90,
					CrosshairsDeleteCursor			= 100,
					CrosshairsEllipsisCursor		= 110,
					CrosshairsNoCursor				= 120,
					CrosshairsSquareBrokenCursor	= 130,
					DirectMoveCursor				= 140,
					DirectMoveCopyCursor			= 150,
					DirectSelectCursor				= 160,
					DirectSelectAddCursor			= 170,
					DirectSelectBehindCursor		= 180,
					EyeDropperCursor				= 190,
					EyeDropperFilledCursor			= 200,
					FistCursor						= 210,
					HandCursor						= 220,
					MoveCursor						= 230,
					MoveCopyCursor					= 240,
					PenCursor						= 250,
					PenAddCursor					= 260,
					PencilCursor					= 270,
					PenCloseCursor					= 280,
					PenContinueCursor				= 290,
					PenCornerCursor					= 300,
					PenNewCursor					= 310,
					PenSubstractCursor				= 320,
					PenWeldCursor					= 330,
					RotateBottomCenterCursor		= 340,
					RotateBottomLeftCursor			= 350,
					RotateBottomRightCursor			= 360,
					RotateMidLeftCursor				= 370,
					RotateMidRightCursor			= 380,
					RotateTopCenterCursor			= 390,
					RotateTopLeftCursor				= 400,
					RotateTopRightCursor			= 410,
					ScaleDeclineCursor				= 420,
					ScaleHorizontalCursor			= 430,
					ScaleInclineCursor				= 440,
					ScaleVerticalCursor				= 450,
					SelectCursor					= 460,
					SelectBehindCursor				= 470,
					WatchCursor						= 480,
					ZoomInCursor					= 490,
					ZoomNoneCursor					= 500,
					ZoomOutCursor					= 510
				};
			
				/**
					\brief	Constructs an empty Cursor object
					\author	GW
					\date	09/2014
					
					\note	To test if a Cursor object is empty, call isEmpty() on it
					\note	Empty Cursor objects do not relate to any UI cursor; they are designed to be "receivers" of
							some other Cursor object via the overloaded assignment operator. Empty Cursor objects are
							useless until such time (though it is safe to call any of their methods)
				*/
				Cursor();
			
				/**
					\brief	Constructs a new Cursor object from an existing Cursor object (copy constructor)
					\author	GW
					\date	09/2014

					\param	c_	Existing Cursor object
				*/
				Cursor(const Cursor& c_);
				
				/**
					\brief	Constructs a new Cursor object from a built-in Adobe cursor
					\author	GW
					\date	09/2014

					\param	cursor_		Adobe cursor identifier
				*/
				Cursor(const AdobeCursor cursor_);
			
				/**
					\brief	Constructs a new Cursor object from a PNG image and a hot spot
					\author	GW
					\date	08/2013

					\param	cursorID_		PNGI resource ID for the cursor
					\param	cursorID3to2x_	PNGI resource ID for the cursor at 1.5x resolution
					\param	cursorID2x_		PNGI resource ID for the cursor at 2x resolution
					\param	hotSpot_		Coordinate within the cursor that acts as the hot spot, at 1x resolution
				 
					\note	Use noneImageID to indicate that no 1.5x or 2x variant is available.
					\note	PNGI resource IDs from 0-5000 are reserved for hdi_core.
				*/
				Cursor(
					const int16_t cursorID_,
					const int16_t cursorID3to2x_,
					const int16_t cursorID2x_,
					const Point& hotSpot_
				);
			
				/**
					\brief	Constructs a new Cursor object from a PNG image and a hot spot
					\author	GW
					\date	08/2013

					\param	cursorID_		PNGI resource ID for the cursor
					\param	cursorID3to2x_	PNGI resource ID for the cursor at 1.5x resolution
					\param	cursorID2x_		PNGI resource ID for the cursor at 2x resolution
					\param	hotSpot_		Coordinate within the cursor that acts as the hot spot, at 1x resolution
				 
					\note	Use noneImageID to indicate that no variants above 1x are available.
					\note	PNGI resource IDs from 0-5000 are reserved for hdi_core.
				*/
				Cursor(
					const int16_t cursorID_,
					const int16_t cursorID3to2x_,
					const int16_t cursorID2x_,
					const int16_t cursorID3x_,
					const int16_t cursorID4x_,
					const Point& hotSpot_
				);
			
				/**
					\brief		Named constructor to acquire a Cursor object for whichever cursor image is current
					\author		GW
					\date		11/2015
					
					\returns	A Cursor object for the current UI cursor
				*/
				static Cursor CurrentCursor();
			
				/**
					\brief		Assigns one Cursor object to another
					\author		GW
					\date		09/2014

					\param		rhs_	Existing Cursor object to copy values from
					\returns	The target Cursor object, but with its values updated to match that of the rhs_ argument
				*/
				Cursor& operator=(const Cursor& rhs_);

				/**
					\brief	Destructs a Cursor object
					\author	GW
					\date	08/2013
				*/
				virtual ~Cursor();

				/**
					\brief		Gets whether the target Cursor object is empty (constructed with the default ctor)
					\author		GW
					\date		09/2014
					
					\returns	true if the target Cursor object is empty, false otherwise
				*/
				virtual bool isEmpty() const;

				/**
					\brief	Shows the cursor (sets it as current)
					\author	GW
					\date	08/2013
				*/
				virtual void enable();
				
				/**
					\brief		Gets the cursor hot spot
					\author		GW
					\date		09/2014
					
					\returns	The hot spot for the target cursor
				*/
				virtual Point hotSpot() const;
				
				/**
					\brief		Tests whether a given Cursor object is the same as another
					\author		GW
					\date		09/2014

					\param		rhs_	Cursor to compare against (righthand side of equality operator)
					\returns	true for the target and rhs_ being the same cursor, false otherwise
				*/
				virtual bool operator==(const Cursor& rhs_) const;
				
				/**
					\brief		Tests whether a given Cursor object is not the same as another
					\author		GW
					\date		09/2014

					\param		rhs_	Cursor to compare against (righthand side of inequality operator)
					\returns	true for the target and rhs_ being different cursors, false otherwise
				*/
				virtual bool operator!=(const Cursor& rhs_) const;

				/**
					\brief		Gets the absolute location of the cursor on the screen
					\author		GW
					\date		08/2013

					\returns	The location of the cursor on the screen, with the origin being top-left
				*/
				static Point screenLoc();

				/**
					\brief	Simulates a mouse down (with no modifier keys) at an absolute location on the screen
					\author	GW
					\date	08/2013

					\param	pt_		Point at which to create the mouse down event
				*/
				static void mouseDown(const Point& pt_);

				/**
					\brief	Simulates a mouse up (with no modifier keys) at an absolute location on the screen
					\author	GW
					\date	08/2013

					\param	pt_		Point at which to create the mouse up event
				*/
				static void mouseUp(const Point& pt_);

				/**
					\brief	Simulates a click (with no modifier keys) at an absolute location on the screen
					\author	GW
					\date	08/2013

					\param	pt_		Point to click

					\note	A click event is simply a mouse down immediately followed by a mouse up
				*/
				static void click(const Point& pt_);


			private:
				friend pui::Cursor* __accessImpl(const Cursor&);

				/**
					\brief	Private implementation data
				*/
				void* __data;
				
				/**
					\brief	Internal use only
					\author	GW
					\date	09/2014
				*/
				void* __impl() const;
		};
		
		typedef std::unique_ptr<Cursor> CursorUP;
		typedef std::shared_ptr<Cursor> CursorSP;
		typedef std::weak_ptr<Cursor> CursorWP;
		
		extern pui::Cursor* __accessImpl(const Cursor&);
	}
}

#endif
// __HDI_CORE_CURSOR__
