/**
	\file
	\brief		Header file for handling a day of the year
	\copyright	Hot Door, Inc. 2010-2025
*/

#ifndef __HDI_CORE_DATE__
#define __HDI_CORE_DATE__

#include "hdicoreDuration.h"

namespace hdi
{
	namespace plat
	{
		class Date;
	}

	namespace core
	{
		/**
			\brief	Allows for acquisition and manipulation of a date (day of year)
		*/
		class Date
		{
			public:
				/**
					\brief	Constructs a Date object with a default value (1970-01-01)
					\author	GW
					\date	11/2014
				*/
				Date();
				
				/**
					\brief	Constructs a Date object from another existing Date object
					\author	GW
					\date	11/2014
					
					\param	d_	Existing object to copy values from
				*/
				Date(const Date& d_);
				
				/**
					\brief	Constructs a Date object from a string representation of a date
					\author	GW
					\date	11/2014
					
					\param	date_	String representation of a date
					
					\note	The expected format is e.g. "2014-11-25". A variety of delimiters can be used (i.e.	'-', '.',
							' ', '/', and ','). Strings that do not follow this format have the same result as using the
							Date() ctor.
				*/
				Date(const std::string& date_);
				
				/**
					\brief	Constructs a Date object from a given year, month, and day
					\author	GW
					\date	11/2014
					
					\param	year_	Year, in the range [1970,9999]
					\param	month_	Month of the given year, in the range [1,12]
					\param	day_	Day of the month, in the range [1,31]
					
					\note	Values below or above the expected range will be forced to the minimum or maximum value,
							respectively.
				*/
				Date(const int16_t year_, const int16_t month_, const int16_t day_);
				
				/**
					\brief		Constructs a Date object from the current system date
					\author		GW
					\date		04/2015
					
					\returns	A Date object for the current UTC time
				*/
				static Date Current();
				
				/**
					\brief	Destructs a Date object
					\author	GW
					\date	11/2014
				*/
				virtual ~Date();
				
				/**
					\brief		Overloaded assignment operator to copy values from one Date object to another
					\author		GW
					\date		11/2014
					
					\param		rhs_	Righthand side of the operator; Date object to copy values from
					\returns	The target Date object, but with values updated to match that of rhs_
				*/
				virtual Date& operator=(const Date& rhs_);
				
				/**
					\brief		Overloaded equality operator to determine if two Date objects are equal
					\author		GW
					\date		11/2014
					
					\param		rhs_	Righthand side of the operator; Date object to compare
					\returns	true if the two objects represent the same day
				*/
				virtual bool operator==(const Date& rhs_) const;
				
				/**
					\brief		Overloaded inequality operator to determine if two Date objects differ
					\author		GW
					\date		11/2014
					
					\param		rhs_	Righthand side of the operator; Date object to compare
					\returns	true if the two objects represent different days
				*/
				virtual bool operator!=(const Date& rhs_) const;
				
				/**
					\brief		Overloaded less-than operator to determine if one Date object is earlier in time than
								the other
					\author		GW
					\date		02/2015
					
					\param		rhs_	Righthand side of the operator; Date object that might be later in time than the
										target (i.e. lefthand side)
					\returns	true if the target (i.e. lefthand side) represents a date earlier than rhs_
				*/
				virtual bool operator<(const Date& rhs_) const;
				
				/**
					\brief		Overloaded less-than-or-equal operator to determine if one Date object is earlier in
								time than, or equal to, the other
					\author		GW
					\date		02/2015
					
					\param		rhs_	Righthand side of the operator; Date object that might be later in time than,
										or equal to, the target (i.e. lefthand side)
					\returns	true if the target (i.e. lefthand side) represents a date earlier than, or equal to, rhs_
				*/
				virtual bool operator<=(const Date& rhs_) const;
				
				/**
					\brief		Overloaded greater-than operator to determine if one Date object is later in time than
								the other
					\author		GW
					\date		02/2015
					
					\param		rhs_	Righthand side of the operator; Date object that might be earlier in time than
										the target (i.e. lefthand side)
					\returns	true if the target (i.e. lefthand side) represents a date later than rhs_
				*/
				virtual bool operator>(const Date& rhs_) const;
				
				/**
					\brief		Overloaded greater-than-or-equal operator to determine if one Date object is later in
								time than, or equal to, the other
					\author		GW
					\date		02/2015
					
					\param		rhs_	Righthand side of the operator; Date object that might be earlier in time than,
										or equal to, the target (i.e. lefthand side)
					\returns	true if the target (i.e. lefthand side) represents a date later than, or equal to, rhs_
				*/
				virtual bool operator>=(const Date& rhs_) const;
				
				/**
					\brief		Overloaded subtraction operator to determine the duration between two Date objects
					\author		GW
					\date		11/2014
					
					\param		rhs_	Righthand side of the operator, Date object to "substract" from the lefthand side
					\returns	The duration between the two days
				*/
				virtual Duration operator-(const Date& rhs_) const;
				
				/**
					\brief		Overloaded substraction operator to determine the resultant date before the target
					\author		GW
					\date		11/2014
					
					\param		rhs_	Righthand side of the operator, Duration object to substract from the lefthand
										side
					\returns	The date that would result from subtracting the duration from the target
					
					\note		For partial-day durations, the subtraction behaves as if the time of day is 00:00:00.
								That is, if subtracting less than a whole day the new date will become the previous day,
								and if adding less than a whole day the new date will remain unchanged.
				*/
				virtual Date operator-(const Duration& rhs_) const;
				
				/**
					\brief		Overloaded subtraction-assignment operator to determine and assign the resultant date
								before the initial target
					\author		GW
					\date		11/2014
					
					\param		rhs_	Righthand side of the operator, Duration object to substract from the lefthand
										side
					\returns	The target Date object, but with its values updated to reflect the substraction of the
								given duration
					
					\note		For partial-day durations, the subtraction behaves as if the time of day is 00:00:00.
								That is, if subtracting less than a whole day the new date will become the previous day,
								and if adding less than a whole day the new date will remain unchanged.
				*/
				virtual Date& operator-=(const Duration& rhs_);
				
				/**
					\brief		Overloaded addition operator to determine the resultant date after the target
					\author		GW
					\date		11/2014
					
					\param		rhs_	Righthand side of the operator, Duration object to add to the lefthand side
					\returns	The date that would result from adding the duration to the target
					
					\note		For partial-day durations, the addition behaves as if the time of day is 00:00:00. That
								is, if adding less than a whole day the new date will remain unchanged, and if
								subtracting less than a whole day the new date will become the previous day.
				*/
				virtual Date operator+(const Duration& rhs_) const;
				
				/**
					\brief		Overloaded addition-assignment operator to determine and assign the resultant date
								after the initial target
					\author		GW
					\date		11/2014
					
					\param		rhs_	Righthand side of the operator, Duration object to add to the lefthand side
					\returns	The target Date object, but with its values updated to reflect the addition of the given
								duration
					
					\note		For partial-day durations, the addition behaves as if the time of day is 00:00:00. That
								is, if adding less than a whole day the new date will remain unchanged, and if
								subtracting less than a whole day the new date will become the previous day.
				*/
				virtual Date& operator+=(const Duration& rhs_);
				
				/**
					\brief		Formats the target Date object into a string representation
					\author		GW
					\date		11/2014
					
					\param		format_		Input string representing the format for the output string
					\returns	An output string for the target object's date, formatted as indicated
					
					\note		The following characters are supported as part of the format_ argument: "Y" for a full
								4-digit numeric representation of a year, "y" for a 2-digit representation of a year, "m"
								for a numeric representation of a month with leading zeros, "n" for a numeric
								representation of a month without leading zeros, "M" for a short textual representation
								of a month in three letters (e.g. Jan, Feb), "F" for a full textual representation of a
								month (e.g. January, February), "t" for the number of days in the month, "W" for the
								ISO-8601 week number of year (from 1 to 53, starting on Monday), "d" for a 2-digit
								representation of the day of the month with leading zeros, "j" for a 2-digit
								representation of the day of the month without leading zeros, "z" for the day of the
								year (from 1 to 366), "D" for a short textual representation of a day in three letters
								(e.g. Mon, Tue), "l" for a full textual representation of the day of the week (e.g.
								Monday, Tuesday), and "N" for the ISO-8601 numeric representation of the day of the week
								(1 for Monday, 2 for Tuesday).
					\note		The special characters above will be expanded as described, and any other characters
								will simply be inserted in the output string. A special character can be preceeded by a
								'\\' (backslash) character to prevent expansion (the backslash will not be inserted in
								the output string). If a backslash character is desired in the output string, preceed it
								with another backslash character.
				*/
				virtual std::string format(const std::string& format_ = "Y-m-d") const;
				
				/**
					\brief		Gets the year represented by the target
					\author		GW
					\date		11/2014
					
					\returns	The year number
				*/
				virtual int16_t year() const;
								
				/**
					\brief	Sets the year that the target should represent
					\author	GW
					\date	11/2014
					
					\param	year_	Year, in range [1970,9999]
					
					\note	Values below or above the expected range will be forced to the minimum or maximum value,
							respectively.
				*/
				virtual void setYear(const int16_t year_);
				
				/**
					\brief		Gets the month of the year represented by the target
					\author		GW
					\date		11/2014
					
					\returns	The month of the year
				*/
				virtual int16_t month() const;
								
				/**
					\brief	Sets the month of the year represented by the target
					\author	GW
					\date	11/2014
					
					\param	month_	Month of the year, in range [1,12]
					
					\note	Values below or above the expected range will be forced to the minimum or maximum value,
							respectively.
				*/
				virtual void setMonth(const int16_t month_);
				
				/**
					\brief		Gets the day of the month represented by the target
					\author		GW
					\date		11/2014
					
					\returns	The day of the month
				*/
				virtual int16_t day() const;
								
				/**
					\brief	Sets the day of the month represented by the target
					\author	GW
					\date	11/2014
					
					\param	day_	Day of the month, in range [1,31]
					
					\note	Values below or above the expected range will be forced to the minimum or maximum value,
							respectively.
				*/
				virtual void setDay(const int16_t day_);
				
				/**
					\brief		Gets the day number of the week, from 0 to 6
					\author		GW
					\date		11/2014
					
					\returns	Numeric representation of the day of the week (0 is Sunday, 1 is Monday, etc.)
				*/
				virtual int16_t dayOfWeek() const;
				
				/**
					\brief		Gets the ISO-8601 week number of the year represented by the target, from 1 to 53
					\author		GW
					\date		11/2014
					
					\returns	Week number in the year of the date
				*/
				virtual int16_t weekNumber() const;

				/**
					\brief		Gets the last day of the month represented by the target, from 28 to 31
					\author		GW
					\date		11/2014
					
					\returns	The last day of the month
				*/
				virtual int16_t endOfMonth() const;
				
				/**
					\brief		Gets the day number of the year, from 1 to 366
					\author		GW
					\date		11/2014
					
					\returns	Day of the year (rather than day of the month)
				*/
				virtual int16_t dayOfYear() const;


			private:
				friend class DateTime;
			
				/**
					\brief	Stores the internal date object
				*/
				plat::Date* __impl;
				
				/**
					\brief	Internal use only
					\author	GW
					\date	11/2014
				*/
				Date(const plat::Date&);
		};
		
		typedef std::unique_ptr<Date> DateUP;
		typedef std::shared_ptr<Date> DateSP;
		typedef std::weak_ptr<Date> DateWP;
	}
}

/**
	\brief		Adds a duration to a date
	\author		GW
	\date		11/2014
	
	\param		lhs_	Duration to use to offset the date
	\param		rhs_	Date to be offset by a duration
	\returns	A new Date object for the result of the offset
*/
hdi::core::Date operator+(const hdi::core::Duration& lhs_, const hdi::core::Date& rhs_);

#endif
// __HDI_CORE_DATE__
