/**
	\file
	\brief		Header file for manipulating Illustrator document views
	\copyright	Hot Door, Inc. 2010-2025
*/

#ifndef __HDI_CORE_DOCUMENT_VIEW__
#define __HDI_CORE_DOCUMENT_VIEW__

#if defined(HDI_CORE_AIP_MODE)

#include "hdicoreArtboardPoint.h"
#include "hdicoreArtboardRect.h"
#include "hdicorePoint.h"
#include "hdicoreRect.h"

namespace hdi
{
	namespace aip
	{
		class DocumentView;
	}

	namespace core
	{
		/**
			\brief	Allow for interacting the document views (windows)
		*/
		class DocumentView
		{
			public:
				/**
					\brief	Describes the current layout of Illustrator's window(s)
				*/
				enum ScreenMode
				{
					UnknownScreenMode			= 0,
					NoDocumentsScreenMode		= 10,
					NormalScreenMode			= 20,
					FullScreenWithMenuMode		= 30,
					FullScreenWithoutMenuMode	= 40,
					PresentationMode			= 50
				};

				/**
					\brief	Constructs an empty DocumentView object
					\author	GW
					\date	08/2013
					
					\note	To test if a DocumentView object is empty, call isEmpty() on it
					\note	Empty DocumentView objects do not relate to any Illustrator view; they are designed to be
							"receivers" of some other DocumentView object via the overloaded assignment operator. Empty
							DocumentView objects are useless until such time (though it is safe to call any of their
							methods)
				*/
				DocumentView();

				/**
					\brief	DocumentView copy constructor
					\author	GW
					\date	08/2013

					\param	dv_		Existing DocumentView object to copy
				*/
				DocumentView(const DocumentView& dv_);
				
				/**
					\brief	Constructs a new DocumentView object from an AIDocumentViewHandle
					\author	GW
					\date	08/2015
					
					\param	aiDocView_	Raw Illustrator AIDocumentViewHandle
				*/
				DocumentView(const AIDocumentViewHandle aiDocView_);

				/**
					\brief	DocumentView destructor
					\author	GW
					\date	08/2013
				*/
				virtual ~DocumentView();
			
				/**
					\brief		Assigns one DocumentView object to another
					\author		GW
					\date		08/2013

					\param		rhs_	Existing DocumentView object to copy values from
					\returns	The target DocumentView object, but with its values updated to match that of the rhs_
								argument
				*/
				virtual DocumentView& operator=(const DocumentView& rhs_);

				/**
					\brief		Gets the document view handle around which the target object is wrapped
					\author		GW
					\date		11/2013
					
					\returns	AIDocumentViewHandle for this object

					\note		Generally, you shouldn't need to call this method. Only use it if you know what you're
								doing. If a specific piece of functionality provided by Illustrator is not handled by
								this class (or related classes), then it should probably be added to the hdi_core library.
				*/
				virtual AIDocumentViewHandle aiDocumentViewHandle() const;

				/**
					\brief		Gets whether the target DocumentView object is empty (constructed with the default ctor)
					\author		GW
					\date		08/2013
					
					\returns	true if the target DocumentView object is empty, false otherwise
				*/
				virtual bool isEmpty() const;

				/**
					\brief		Gets the bounds of the portion of the document that is currently visible in its window
					\author		GW
					\date		08/2013

					\returns	Rect representing the currently visible portion of the document
				*/
				virtual ArtboardRect bounds() const;

				/**
					\brief		Gets the center of the portion of the document that is currently visible in its window
					\author		GW
					\date		08/2013

					\returns	Point representing the center of the currently visible portion of the document
				*/
				virtual ArtboardPoint center() const;

				/**
					\brief	Moves the document view to be centered on the given point
					\author	GW
					\date	08/2013

					\param	center_		Point in the document on which to center in the document's containing window
				*/
				virtual void setCenter(const ArtboardPoint& center_);

				/**
					\brief		Gets the current zoom level of the document. This is the scale factor from artwork
								coordinates to window coordinates.
					\author		GW
					\date		08/2013

					\returns	The document's zoom level, where 1.0 is 100%, 0.5 is 50%, 2.0 is 200%, etc.
				*/
				virtual double zoom() const;

				/**
					\brief	Sets the current zoom level of the document. This is the scale factor from artwork
							coordinates to window coordinates.
					\author	GW
					\date	08/2013

					\param	zoom_	New document zoom level, where 1.0 is 100%, 0.5 is 50%, 2.0 is 200%, etc.
				*/
				virtual void setZoom(const double zoom_);

				/**
					\brief		Gets the current display zoom level of the document. This is the zoom value that the
								user sees on the document status bar.
					\author		GW
					\date		08/2023

					\returns	The document's display zoom level, where 1.0 is 100%, 0.5 is 50%, 2.0 is 200%, etc.
				*/
				virtual double displayZoom() const;

				/**
					\brief	Sets the current display zoom level of the document. This is the zoom value that the user
							sees on the document status bar.
					\author	GW
					\date	08/2023

					\param	zoom_	New document display zoom level, where 1.0 is 100%, 0.5 is 50%, 2.0 is 200%, etc.
				*/
				virtual void setDisplayZoom(const double zoom_);

				/**
					\brief		Gets the current screen mode of the document (full, windowed, etc.)
					\author		GW
					\date		08/2013

					\returns	Current document view screen mode
				*/
				virtual ScreenMode screenMode() const;

				/**
					\brief	Sets the current screen mode of the document (full, windowed, etc.)
					\author	GW
					\date	08/2013

					\param	mode_	New screen mode for the document view
				*/
				virtual void setScreenMode(const ScreenMode mode_);

				/**
					\brief		Gets the point and angle of rotation for the document view
					\author		GW
					\date		04/2021

					\param		pt__	Return-by-reference for the point in the document around which the view is rotated
					\param		angle__	Return-by-reference for the angle of rotation around the point
					\returns	true if the return-by-reference values were acquired successfully
				 
					\note		In Illustrator 25.2 and earlier rotation is unsupported, so the returned values will
								always be the document origin and zero degrees in those versions.
				*/
				virtual bool rotation(ArtboardPoint& pt__, Angle& angle__) const;
			
				/**
					\brief	Sets the point and angle of rotation for the document view
					\author	GW
					\date	04/2021

					\param	pt_		Point in the document around which to rotate the view
					\param	angle_	Angle of rotation around the point
				 
					\note	In Illustrator 25.2 and earlier rotation is unsupported, so calling this method will have
							no effect in those versions.
				*/
				virtual void setRotation(const ArtboardPoint& pt_, const Angle& angle_);
			
				/**
					\brief	Resets the angle of rotation for the document view back to zero degrees
					\author	GW
					\date	04/2021
				 
					\note	In Illustrator 25.2 and earlier rotation is unsupported, so calling this method will have
							no effect in those versions.
				*/
				virtual void resetRotation();

				/**
					\brief		Converts a point on the document to an absolute point on the screen
					\author		GW
					\date		08/2013

					\param		artPt_	Artboard point to convert to a UI point
					\returns	The provided artboard point converted to the same physical coordinates, but as a UI point
				 
					\note		In Illustrator 25.3 and later view rotation is supported, so calling this method will
								return the post-rotation point in those versions. If you want the pre-rotation point,
								call the artworkPointToViewPointUnrotated() method.
				*/
				virtual Point artworkPointToViewPoint(const ArtboardPoint& artPt_);

				/**
					\brief		Converts a point on the screen to a point on the document
					\author		GW
					\date		08/2013

					\param		viewPt_		UI point to convert to an artboard point
					\returns	The provided UI point converted to the same physical coordinates, but as an artboard point
				 
					\note		In Illustrator 25.3 and later view rotation is supported, so calling this method will
								return the post-rotation point in those versions. If you want the pre-rotation point,
								call the viewPointToArtworkPointUnrotated() method.
				*/
				virtual ArtboardPoint viewPointToArtworkPoint(const Point& viewPt_);

				/**
					\brief		Converts a rect on the document to an absolute rect on the screen
					\author		GW
					\date		08/2013

					\param		artRect_	Artboard rect to convert to a UI rect
					\returns	The provided artboard rect converted to the same physical coordinates, but as a UI rect
				 
					\note		In Illustrator 25.3 and later view rotation is supported, so calling this method will
								return the post-rotation rect in those versions. If you want the pre-rotation rect, call
								the artworkRectToViewRectUnrotated() method.
				*/
				virtual Rect artworkRectToViewRect(const ArtboardRect& artRect_);

				/**
					\brief		Converts a rect on the screen to a rect on the document
					\author		GW
					\date		08/2013

					\param		viewRect_	UI rect to convert to an artboard rect
					\returns	The provided UI rect converted to the same physical coordinates, but as an artboard rect
				 
					\note		In Illustrator 25.3 and later view rotation is supported, so calling this method will
								return the post-rotation rect in those versions. If you want the pre-rotation rect, call
								the viewRectToArtworkRectUnrotated() method.
				*/
				virtual ArtboardRect viewRectToArtworkRect(const Rect& viewRect_);
			
				/**
					\brief		Converts a point on the document to an absolute point on the screen without taking into
								account the view rotation
					\author		GW
					\date		04/2021

					\param		artPt_	Artboard point to convert to a UI point
					\returns	The provided artboard point converted to the same physical coordinates, but as a UI point
				 
					\note		In Illustrator 25.2 and earlier rotation is unsupported, so calling this method will
								return the same value as calling the artworkPointToViewPoint() method in those versions.
				*/
				virtual Point artworkPointToViewPointUnrotated(const ArtboardPoint& artPt_);

				/**
					\brief		Converts a point on the screen to a point on the document without taking into account
								the view rotation
					\author		GW
					\date		08/2011

					\param		viewPt_		UI point to convert to an artboard point
					\returns	The provided UI point converted to the same physical coordinates, but as an artboard point
				 
					\note		In Illustrator 25.2 and earlier rotation is unsupported, so calling this method will
								return the same value as calling the viewPointToArtworkPoint() method in those versions.
				*/
				virtual ArtboardPoint viewPointToArtworkPointUnrotated(const Point& viewPt_);

				/**
					\brief		Converts a rect on the document to an absolute rect on the screen without taking into
								account the view rotation
					\author		GW
					\date		08/2011

					\param		artRect_	Artboard rect to convert to a UI rect
					\returns	The provided artboard rect converted to the same physical coordinates, but as a UI rect
				 
					\note		In Illustrator 25.2 and earlier rotation is unsupported, so calling this method will
								return the same value as calling the artworkRectToViewRect() method in those versions.
				*/
				virtual Rect artworkRectToViewRectUnrotated(const ArtboardRect& artRect_);

				/**
					\brief		Converts a rect on the screen to a rect on the document without taking into account the
								view rotation
					\author		GW
					\date		08/2011

					\param		viewRect_	UI rect to convert to an artboard rect
					\returns	The provided UI rect converted to the same physical coordinates, but as an artboard rect
				 
					\note		In Illustrator 25.2 and earlier rotation is unsupported, so calling this method will
								return the same value as calling the viewRectToArtworkRect() method in those versions.
				*/
				virtual ArtboardRect viewRectToArtworkRectUnrotated(const Rect& viewRect_);
				/**
					\brief		Checks whether one DocumentView object represents the same Illustrator view as another
					\author		GW
					\date		08/2013

					\param		rhs_	Existing DocumentView object to compare with
					\returns	true if the represented views are the same, false otherwise
				*/
				bool operator==(const DocumentView& rhs_) const;
			
				/**
					\brief		Checks whether one DocumentView object does not represent the same Illustrator view as
								another
					\author		GW
					\date		08/2013

					\param		rhs_	Existing DocumentView object to compare with
					\returns	true if the represented views are not the same, false otherwise
				*/
				bool operator!=(const DocumentView& rhs_) const;


			private:
				friend aip::DocumentView* __accessImpl(const DocumentView&);
				friend DocumentView __accessCtor(aip::DocumentView*&);

				/**
					\brief	Private implementation data
				*/
				void* __data;
				
				/**
					\brief	Internal use only
					\author	GW
					\date	08/2013
				*/
				void* __impl() const;
				
				/**
					\brief	Internal use only
					\author	GW
					\date	08/2013
				*/
				DocumentView(aip::DocumentView*&);
		};
		
		typedef std::unique_ptr<DocumentView> DocumentViewUP;
		typedef std::shared_ptr<DocumentView> DocumentViewSP;
		typedef std::weak_ptr<DocumentView> DocumentViewWP;
		
		extern aip::DocumentView* __accessImpl(const DocumentView&);
		extern DocumentView __accessCtor(aip::DocumentView*&);
	}
}

#endif
// HDI_CORE_AIP_MODE

#endif
// __HDI_CORE_DOCUMENT_VIEW__
