/**
	\file
	\brief		Header file for an eye dropper tool that allows selection of a color from the current document
	\copyright	Hot Door, Inc. 2010-2025
*/

#ifndef __HDI_CORE_EYE_DROPPER_TOOL__
#define __HDI_CORE_EYE_DROPPER_TOOL__

#if defined(HDI_CORE_AIP_MODE)

#include "hdicoreMacros.h"

#include "hdicoreCursor.h"
#include "hdicorePathStyle.h"
#include "hdicoreTool.h"

namespace hdi
{
	namespace core
	{
		class PathStyle;
	
		/**
			\brief		Hidden eye dropper tool, to be used programmatically only, that allows for selection of a color
						from the current document (the Illustrator eye dropper always sets the toolbar fill/stroke color
						and does not allow for interception of the selected color, whereas this tool simply provides the
						color under the cursor to the caller)

			\warning	Usage of this class will cause your plugin to "become" a tool plugin, and therefore its file
						must be placed in the Illustrator Tools folder to function properly.
			\warning	This tool, like all Illustrator tools, cannot be used from within the context of a modal dialog.
						If you need this sort of behavior, consider switching to a floating dialog instead.

			\note		To use this tool, first create it during plugin startup and set your mouse event callbacks, then
						select it during e.g. some UI event. The user can then move the cursor around the document and
						click a desired color, and you can acquire the color in your track, drag, and/or mouse up
						callbacks. The tool will automatically deselect itself upon the mouse up event, restoring
						whichever tool was current before the eye dropper was selected. Remember to destroy the tool
						during plugin shutdown.
		*/
		class EyeDropperTool
		{
			public:
				/**
					\brief		Creates the singleton instance of the EyeDropperTool class
					\author		GW
					\date		09/2014
					
					\returns	The EyeDropperTool instance object
				*/
				static EyeDropperTool* const create();
			
				/**
					\brief		Acquires the singleton instance of the EyeDropperTool class
					\author		GW
					\date		09/2014
					
					\returns	The EyeDropperTool instance object
					
					\note		This method will always return NULL until create() has been called
				*/
				static EyeDropperTool* const instance();
				
				/**
					\brief	Destroys the singleton instance of the EyeDropperTool class
					\author	GW
					\date	09/2014
				*/
				static void destroy();

				/**
					\brief		Gets the tool handle for the eye dropper tool
					\author		GW
					\date		09/2014
					
					\returns	AIToolHandle for this object

					\note		Generally, you shouldn't need to call this method. Only use it if you know what you're
								doing. If a specific piece of functionality provided by Illustrator is not handled by
								this class (or related classes), then it should probably be added to the hdi_core library.
				*/
				virtual AIToolHandle aiToolHandle() const;
				
				/**
					\brief		Gets whether the eye dropper tool is the currently selected tool
					\author		GW
					\date		09/2014
					
					\returns	true if the tool is selected, false otherwise
				*/
				virtual bool selected() const;
				
				/**
					\brief	Sets the eye dropper tool as the current tool, storing the previously selected tool for
							later resuming
					\author	GW
					\date	09/2014
				*/
				virtual void select();
				
				/**
					\brief	Deselects the eye dropper tool, causing the previously selected tool to become current again
					\author	GW
					\date	09/2014
				*/
				virtual void deselect();

				/**
					\brief		Gets the eye dropper tool's title
					\author		GW
					\date		09/2014

					\returns	The title for the tool, as UTF-8
				*/
				virtual std::string title() const;

				/**
					\brief		Gets the eye dropper tool's tip string
					\author		GW
					\date		09/2014

					\returns	The rollover tip for the tool, as UTF-8
				*/
				virtual std::string tip() const;

				/**
					\brief		Gets the current location of the cursor
					\author		GW
					\date		09/2014

					\returns	The last known cursor location for the tool

					\note		The cursor location is automatically set when a mouse event occurs
				*/
				virtual ArtboardPoint cursorLoc() const;

				/**
					\brief		Gets the registered callback for the tool tracking event (mouse move)
					\author		GW
					\date		09/2014
					
					\returns	Callback for tracking, or NULL if none
				*/
				virtual Callback* const trackCallback() const;

				/**
					\brief	Sets the callback for the tool tracking event (mouse move)
					\author	GW
					\date	09/2014
					
					\param	cb_		The new tracking callback
				*/
				virtual void setTrackCallback(const Callback& cb_);

				/**
					\brief		Gets the registered callback for the drag event
					\author		GW
					\date		09/2014
					
					\returns	Callback for dragging, or NULL if none
				*/
				virtual Callback* const dragCallback() const;

				/**
					\brief	Sets the callback for the drag event
					\author	GW
					\date	09/2014
					
					\param	cb_		The new dragging callback
				*/
				virtual void setDragCallback(const Callback& cb_);

				/**
					\brief		Gets the registered callback for the mouse up event
					\author		GW
					\date		09/2014
					
					\returns	Callback for mouse up, or NULL if none
				*/
				virtual Callback* const mouseUpCallback() const;

				/**
					\brief	Sets the callback for the mouse up event
					\author	GW
					\date	09/2014
					
					\param	cb_		The new mouse up callback
				*/
				virtual void setMouseUpCallback(const Callback& cb_);

				/**
					\brief		Determines whether the shift key is down
					\author		GW
					\date		09/2014

					\returns	true if the shift key was down in the last tool event

					\note		The return value is only valid if the last message the plugin received was an eye dropper
								tool-related event.
				*/
				virtual bool shiftKeyDown() const;

				/**
					\brief		Determines whether the command key is down
					\author		GW
					\date		09/2014

					\returns	true if the command key was down in the last tool event

					\note		Windows machines do not have a command key. If this method is called on a Windows
								machine, it will return the same value as the controlKeyDown() method (because the
								control key is the closest command key analog on Windows).
					\note		The return value is only valid if the last message the plugin received was an eye dropper
								tool-related event.
				*/
				virtual bool commandKeyDown() const;

				/**
					\brief		Determines whether the option key is down
					\author		GW
					\date		09/2014

					\returns	true if the option key was down in the last tool event

					\note		The option key on Mac is akin to the alt key on Windows.
					\note		The return value is only valid if the last message the plugin received was an eye dropper
								tool-related event.
				*/
				virtual bool optionKeyDown() const;

				/**
					\brief		Determines whether the control key is down
					\author		GW
					\date		09/2014

					\returns	true if the control key was down in the last tool event

					\note		The return value is only valid if the last message the plugin received was an eye dropper
								tool-related event.
				*/
				virtual bool controlKeyDown() const;

				/**
					\brief		Gets the current path style for the eye dropper tool
					\author		GW
					\date		09/2014
					
					\returns	Either the path style for the art object under the cursor, or a "synthetic" path style
								for the pixel color under the cursor

					\note		The return value of this method is only valid when called from within an EyeDropperTool
								mouse event callback.
					\note		If artwork that can have a path style is under the cursor, then this is acquired and
								returned. If art that cannot have a path style is under the cursor, then the pixel color
								is acquired and placed in the fill color of the path style. If no art is under the
								cursor, then the pixel color is acquired and placed in the fill color of the path style.
				*/
				virtual PathStyle currentStyle() const;


			private:
				/**
					\brief	Flag for whether a callback is firing
				*/
				bool __firing;
				
				/**
					\brief	Stores the underlying tool itself
				*/
				Tool __tool;
			
				/**
					\brief	Crosshairs cursor
				*/
				Cursor __cursor;
				
				/**
					\brief	Previously selected tool before the eye dropper was selected
				*/
				std::string __prevTool;
				
				/**
					\brief	Stores whether tools snapped the cursor to artwork
				*/
				bool __oldArtSnap;
				
				/**
					\brief	Stores whether tools snapped the cursor to artwork bounds
				*/
				bool __oldArtBoundsSnap;
				
				/**
					\brief	Stores whether tools snapped the cursor to the artboard edges
				*/
				bool __oldArtboardSnap;
				
				/**
					\brief	Stores whether tools snapped the cursor to the Illustrator gridlines
				*/
				bool __oldGridSnap;
			
				/**
					\brief	Callback for the mouse track event
				*/
				Callback* __trackCB;
				
				/**
					\brief	Callback for the mouse drag event
				*/
				Callback* __dragCB;
				
				/**
					\brief	Callback for the mouse up event
				*/
				Callback* __mouseUpCB;

				/**
					\brief	Constructs an EyeDropperTool instance
					\author	GW
					\date	09/2014
				*/
				EyeDropperTool();
			
				/**
					\brief	Unused
					\author	GW
					\date	08/2014
				*/
				EyeDropperTool(const EyeDropperTool&);

				/**
					\brief	Destructs an EyeDropperTool object
					\author	GW
					\date	09/2014
				*/
				~EyeDropperTool();
				
				/**
					\brief	Unused
					\author	GW
					\date	08/2014
				*/
				EyeDropperTool& operator=(const EyeDropperTool&);
			
				/**
					\brief	Internal callback for the tool deselection callback
					\author	GW
					\date	09/2014
				*/
				void __deselected();
			
				/**
					\brief	Internal callback for the mouse track event
					\author	GW
					\date	09/2014
				*/
				void __track();
				
				/**
					\brief	Internal callback for the mouse drag event
					\author	GW
					\date	09/2014
				*/
				void __drag();
				
				/**
					\brief	Internal callback for the mouse up event
					\author	GW
					\date	09/2014
				*/
				void __mouseUp();
				
				/**
					\brief	Gets the color under the cursor
					\author	GW
					\date	09/2014
				*/
				ArtColor __cursorColor() const;
		};
	}
}

#endif
// HDI_CORE_AIP_MODE

#endif
// __HDI_CORE_EYE_DROPPER_TOOL__
