/**
	\file
	\brief		Header file for application font manipulation
	\copyright	Hot Door, Inc. 2010-2025
*/

#ifndef __HDI_CORE_FONT__
#define __HDI_CORE_FONT__

#include "hdicoreTypes.h"
	
namespace hdi
{
	#if defined(HDI_CORE_AIP_MODE)
		namespace aip
		{
			class Font;
		}
	#elif defined(HDI_CORE_PSP_MODE)
		namespace psp
		{
			class Font;
		}
	#endif

	namespace core
	{
		class Font;
	
		#if defined(HDI_CORE_AIP_MODE)
			/**
				\brief	Typeface class to allow for easy typeface manipulation
			*/
			class Typeface
			{
				public:
					/**
						\brief	Constructs an empty Typeface object
						\author	GW
						\date	08/2014
						
						\note	To test if a Typeface object is empty, call isEmpty() on it
						\note	Empty Typeface objects do not relate to any actual Illustrator typefaces; they are
								designed to be "receivers" of some other Typeface object via the overloaded assignment
								operator. Empty Typeface objects are useless until such time (though it is safe to call
								any of their methods)
					*/
					Typeface();
				
					/**
						\brief	Constructs a new Typeface object from an existing Typeface object (copy constructor)
						\author	GW
						\date	08/2014

						\param	tf_		Existing Typeface object
					*/
					Typeface(const Typeface& tf_);

					/**
						\brief	Constructs a new Typeface object from an AITypefaceKey
						\author	GW
						\date	08/2015
						
						\param	aiTF_		Raw Illustrator AITypefaceKey
					*/
					Typeface(const AITypefaceKey aiTF_);

					/**
						\brief	Typeface object destructor
						\author	GW
						\date	08/2014
					*/
					virtual ~Typeface();
					
					/**
						\brief		Assigns one Typeface object to another
						\author		GW
						\date		08/2014

						\param		rhs_	Existing Typeface object to copy values from
						\returns	The target Typeface object, but with its values updated to match that of the rhs_
									argument
					*/
					virtual Typeface& operator=(const Typeface& rhs_);

					/**
						\brief		Gets whether the target Typeface object is empty (constructed with the default ctor)
						\author		GW
						\date		08/2014
						
						\returns	true if the target Typeface object is empty, false otherwise
					*/
					virtual bool isEmpty() const;

					/**
						\brief		Gets the typeface key around which the target object is wrapped
						\author		GW
						\date		08/2014
						
						\returns	AITypefaceKey for this object

						\note		Generally, you shouldn't need to call this method. Only use it if you know what you're
									doing. If a specific piece of functionality provided by Illustrator is not handled by
									this class (or related classes), then it should probably be added to the hdi_core library.
					*/
					virtual AITypefaceKey aiTypefaceKey() const;
					
					/**
						\brief		Tests whether a given Typeface object is the same as another
						\author		GW
						\date		08/2014

						\param		rhs_	Typeface to compare against (righthand side of equality operator)
						\returns	true for the target and rhs_ being the same typefaces, false otherwise
					*/
					virtual bool operator==(const Typeface& rhs_) const;
					
					/**
						\brief		Tests whether a given Typeface object is not the same as another
						\author		GW
						\date		08/2014

						\param		rhs_	Typeface to compare against (righthand side of inequality operator)
						\returns	true for the target and rhs_ being different typefaces, false otherwise
					*/
					virtual bool operator!=(const Typeface& rhs_) const;
					
					/**
						\brief		Gets the total number of variants (such as bold and italic) of a font family
						\author		GW
						\date		08/2014
						
						\returns	The number of style variants, or -1 for error
					*/
					virtual int32_t styleCount() const;
					
					/**
						\brief		Gets a variant of a font family by index position
						\author		GW
						\date		08/2014
						
						\param		index_	Index of the style in question
						\returns	The font at index_, or NULL for none
					*/
					virtual std::unique_ptr<Font> styleAtIndex(const int32_t index_);

					/**
						\brief		Gets the name of the typeface
						\author		GW
						\date		08/2014
						
						\returns	The typeface name, as UTF-8
					*/
					virtual std::string name() const;


				private:
					friend void* __accessImpl(const Typeface&);

					/**
						\brief	Private implementation data
					*/
					void* __data;
					
					/**
						\brief	Internal use only
						\author	GW
						\date	08/2014
					*/
					void* __impl() const;
			};
			
			typedef std::unique_ptr<Typeface> TypefaceUP;
			typedef std::shared_ptr<Typeface> TypefaceSP;
			typedef std::weak_ptr<Typeface> TypefaceWP;
		#endif
		// HDI_CORE_AIP_MODE
	
		/**
			\brief	Font class to allow for easy font manipulation
		*/
		class Font
		{
			public:
				#if defined(HDI_CORE_AIP_MODE)
					/**
						\brief	Describes style information for an individual font
					*/
					class Info
					{
						public:
							/**
								\brief	Describes the font format
							*/
							enum Format
							{
								UnknownFormat	= 0,
								TypeOneFormat	= 10,
								TrueTypeFormat	= 20
							};
							
							/**
								\brief	Describes the font encoding
							*/
							enum Encoding
							{
								UnknownEncoding		= 0,
								DontCareEncoding	= 10,
								StandardEncoding	= 20,
								SpecialEncoding		= 30
							};
							
							/**
								\brief	Describes the font script
							*/
							enum Script
							{
								UnknownScript				= 0,
								NativeScript				= 10,
								RomanScript					= 20,
								JapaneseScript				= 30,
								ChineseTraditionalScript	= 40,
								ChineseSimplifiedScript		= 50,
								KoreanScript				= 60,
								ArabicScript				= 70,
								HebrewScript				= 80,
								GreekScript					= 90,
								CyrillicScript				= 100,
								EastEuropeanRomanScript		= 110,
								TurkishScript				= 120,
								BalticScript				= 130,
								SymbolScript				= 140,
								EmojiScript					= 150
							};

							/**
								\brief	Describes the font MojiZume type
							*/
							enum MojiZumeType
							{
								MojiZumeUnknown		= 0,
								MojiZumeNone		= 10,
								MojiZumeGXTrueType	= 20,
								MojiZumePageMakerJ	= 30,
								MojiZumeSfntATM		= 40
							};

							/**
								\brief	Font face formatting
							*/
							Format format;
							
							/**
								\brief	Font face encoding
							*/
							Encoding encoding;
							
							/**
								\brief	Font face script
							*/
							Script script;
							
							/**
								\brief	Is the font the original encoding?
							*/
							bool originalEncoding;
							
							/**
								\brief	Is the font family installed?
							*/
							bool familyInstalled;
							
							/**
								\brief	Is the font fauxed for another missing font?
							*/
							bool fauxedInstalled;
							
							/**
								\brief	Is the font character ID-keyed?
							*/
							bool isCID;
							
							/**
								\brief	MojiZume type of the font
							*/
							MojiZumeType mojiZumeType;
							
							/**
								\brief	Is the font OpenType?
							*/
							bool isOTF;
							
							/**
								\brief	Is the font bitmap?
							*/
							bool isBitmap;
							
							/**
								\brief	Is the font Original Composer Format?
							*/
							bool isOCF;
							
							/**
								\brief	Is the font Adobe Type Composer?
							*/
							bool isATC;
							
							/**
								\brief	Is the font Adobe Multiple Master?
							*/
							bool isMultipleMaster;
							
							/**
								\brief	Constructs an Info object with default values
								\author	GW
								\date	08/2013
							*/
							Info();
							
							/**
								\brief	Destructs an Info object
								\author	GW
								\date	08/2013
							*/
							virtual ~Info();
					};
			
					typedef std::unique_ptr<Info> InfoUP;
					typedef std::shared_ptr<Info> InfoSP;
					typedef std::weak_ptr<Info> InfoWP;
				#endif
				// HDI_CORE_AIP_MODE
			
				/**
					\brief		Creates a Font object with the default native script (e.g. roman, cyrillic, etc.)
					\author		GW
					\date		08/2013
					
					\returns	A unique Font object for the default script
				*/
				Font();
			
				/**
					\brief	Font copy constructor
					\author	GW
					\date	08/2013
					
					\param	f_	Font object to copy values from
				*/
				Font(const Font& f_);
				
				#if defined(HDI_CORE_AIP_MODE)
					/**
						\brief	Constructs a new Font object from an AIFontKey
						\author	GW
						\date	08/2015
						
						\param	aiFont_		Raw Illustrator AIFontKey
					*/
					Font(const AIFontKey aiFont_);
				#endif
				// HDI_CORE_AIP_MODE

				/**
					\brief	Destructs a Font object
					\author	GW
					\date	08/2013
				*/
				virtual ~Font();
			
				/**
					\brief		Overloaded assignment operator for Font objects
					\author		GW
					\date		08/2013

					\param		rhs_	Righthand side of the assignment operator; existing Font object to copy values from
					\returns	The lefthand side of the assignment operator, but with its values updated
				*/
				virtual Font& operator=(const Font& rhs_);
				
				/**
					\brief		Tests whether a given Font object is the same as another
					\author		GW
					\date		08/2013

					\param		rhs_	Font to compare against (righthand side of equality operator)
					\returns	true for the target and rhs_ being the same font, false otherwise
				*/
				virtual bool operator==(const Font& rhs_) const;
				
				/**
					\brief		Tests whether a given Font object is not the same as another
					\author		GW
					\date		08/2013

					\param		rhs_	Font to compare against (righthand side of inequality operator)
					\returns	true for the target and rhs_ being different fonts, false otherwise
				*/
				virtual bool operator!=(const Font& rhs_) const;

				#if defined(HDI_CORE_AIP_MODE)
					/**
						\brief		Gets the font key around which the target object is wrapped
						\author		GW
						\date		11/2013
						
						\returns	AIFontKey for this object

						\note		Generally, you shouldn't need to call this method. Only use it if you know what you're
									doing. If a specific piece of functionality provided by Illustrator is not handled by
									this class (or related classes), then it should probably be added to the hdi_core library.
					*/
					virtual AIFontKey aiFontKey() const;
					
					/**
						\brief		Gets the font's style info
						\author		GW
						\date		08/2013
						
						\returns	An Info object describing the font's style info
					*/
					virtual Info info() const;
				
					/**
						\brief		Gets the typeface (family) and style variant of the target font
						\author		GW
						\date		08/2014
						
						\param		tf__	Return-by-reference for the typeface
						\param		style__	Return-by-reference for the index position of the style variant
						\returns	true if the typeface and style index could be acquired, false otherwise
					*/
					virtual bool typefaceAndStyle(Typeface& tf__, int32_t& style__);
					
					/**
						\brief		Gets whether the target font is substituted or not
						\author		GW
						\date		08/2014
						
						\returns	true if substituted, false otherwise
					*/
					virtual bool substituted() const;
				#endif
				// HDI_CORE_AIP_MODE

				/**
					\brief		Gets the name of the font
					\author		GW
					\date		08/2013
					
					\returns	The name of the font, as UTF-8
				*/
				virtual std::string name() const;
				
				/**
					\brief		Gets the name of the family for the target font
					\author		GW
					\date		08/2014
					
					\returns	The family name of the font, as UTF-8
				*/
				virtual std::string familyName() const;
				
				/**
					\brief		Gets the name of the style for the target font
					\author		GW
					\date		08/2014
					
					\returns	The style name of the font, as UTF-8
				*/
				virtual std::string styleName() const;

				/**
					\brief		Gets the PostScript name of the font
					\author		GW
					\date		12/2017
						
					\returns	The name of the font
				*/
				virtual std::string postScriptName() const;
			
				/**
					\brief		Gets the platform font for the target Adobe app font, if available
					\author		GW
					\date		06/2018
				 
				 	\param		size_	Desired font size
					\returns	A pointer to the platform/OS font, or NULL if none/unavailable
				 
					\warning	On Windows, it is the caller's responsibility to cleanup the returned font pointer when
								no longer needed by passing it to the DeleteObject() function.
				*/
				virtual PlatformFontPtr platformFont(const double size_) const;


			private:
				#if defined(HDI_CORE_AIP_MODE)
					friend aip::Font* __accessImpl(const Font&);
					friend Font __accessCtor(const aip::Font&);
					
					/**
						\brief	Private implementation object
					*/
					aip::Font* __impl;

					/**
						\brief	Internal use only
						\author	GW
						\date	08/2013
					*/
					Font(const aip::Font&);
				#elif defined(HDI_CORE_PSP_MODE)
					friend psp::Font* __accessImpl(const Font&);
					friend Font __accessCtor(const psp::Font&);
					
					/**
						\brief	Private implementation object
					*/
					psp::Font* __impl;

					/**
						\brief	Internal use only
						\author	GW
						\date	08/2013
					*/
					Font(const psp::Font&);
				#endif
		};
		
		typedef std::unique_ptr<Font> FontUP;
		typedef std::shared_ptr<Font> FontSP;
		typedef std::weak_ptr<Font> FontWP;

		#if defined(HDI_CORE_AIP_MODE)
			extern void* __accessImpl(const Typeface&);
			extern aip::Font* __accessImpl(const Font&);
			extern Font __accessCtor(const aip::Font&);
		#elif defined(HDI_CORE_PSP_MODE)
			extern psp::Font* __accessImpl(const Font&);
			extern Font __accessCtor(const psp::Font&);
		#endif
	}
}

#endif
// __HDI_CORE_FONT__
