/**
	\file
	\brief		Header file for general Illustrator gradient manipulation
	\copyright	Hot Door, Inc. 2010-2025
*/

#ifndef __HDI_CORE_GRADIENT__
#define __HDI_CORE_GRADIENT__

#if defined(HDI_CORE_AIP_MODE)

#include "hdicoreArtColor.h"

namespace hdi
{
	namespace aip
	{
		class Gradient;
	}

	namespace core
	{
		/**
			\brief	Handles general gradient-related functionality
		*/
		class Gradient
		{
			public:
				/**
					\brief	Indicates the type of "gradient" an object is representing
				*/
				enum Type
				{
					UnknownGradientType	= 0,
					LinearGradientType	= 10,
					RadialGradientType	= 20
				};
				
				/**
					\brief	Describes the attributes of a gradient stop
				*/
				class Stop
				{
					public:
						/**
							\brief	The location between two ramp points where there is an equal mix of this color and
									the color of the next stop. This value is a percentage of the distance between the
									two ramp points, between 0.13 and 0.87. The midpoint for the previous color stop is
									not considered.
						*/
						double midPoint;
						
						/**
							\brief	The position on the blend ramp where this color begins, in the range [0,1]. The
									first point does not have to be at 0, but the first color begins at 0. Similarly,
									the last does not have to be at 1.
						*/
						double rampPoint;
						
						/**
							\brief	The color for the target gradient stop
						*/
						ArtColor color;
						
						/**
							\brief	The opacity value for the gradient stop, in the range [0,1]. 0 is completely
									transparent and 1 is completely opaque.
						*/
						double opacity;
						
						/**
							\brief	Constructs a new Stop object with default values
							\author	GW
							\date	12/2013
						*/
						Stop();
						
						/**
							\brief	Constructs a new Stop object with values from an existing Stop
							\author	GW
							\date	02/2015
							
							\param	s_	Existing Stop to copy values from
						*/
						Stop(const Stop& s_);
						
						/**
							\brief	Constructs a new Stop object with the given values
							\author	GW
							\date	12/2013
							
							\param	midPt_		(See midPoint member description)
							\param	rampPt_		(See rampPoint member description)
							\param	color_		(See color member description)
							\param	opacity_	(See opacity member description)
						*/
						Stop(
							const double midPt_,
							const double rampPt_,
							const ArtColor& color_,
							const double opacity_
						);

						/**
							\brief	Destructs a Stop object
							\author	GW
							\date	12/2013
						*/
						virtual ~Stop();
						
						/**
							\brief		Overloaded assignment operator to copy values from one Stop object to another
							\author		GW
							\date		02/2015
							
							\param		rhs_	Righthand side of the assignment
							\returns	The lefthand side of the assignment, but with its values updated
						*/
						Stop& operator=(const Stop& rhs_);
				};
		
				typedef std::unique_ptr<Stop> StopUP;
				typedef std::shared_ptr<Stop> StopSP;
				typedef std::weak_ptr<Stop> StopWP;

				/**
					\brief	Constructs an empty Gradient object
					\author	GW
					\date	12/2013
					
					\note	To test if a Gradient object is empty, call isEmpty() on it
					\note	Empty Gradient objects do not relate to any gradient in an Illustrator document; they are
							designed to be "receivers" of some other Gradient object via the overloaded assignment
							operator. Empty Gradient objects are useless until such time (though it is safe to call any
							of their methods).
				*/
				Gradient();
			
				/**
					\brief	Constructs a new Gradient object from an existing Gradient object (copy constructor)
					\author	GW
					\date	12/2013

					\param	grad_	Existing Gradient object
				*/
				Gradient(const Gradient& grad_);
				
				/**
					\brief		Creates a new Gradient object (and Illustrator gradient)
					\author		GW
					\date		12/2013
					
					\returns	A new Gradient object
				*/
				static Gradient create();

				/**
					\brief	Gradient object destructor
					\author	GW
					\date	12/2013
				*/
				virtual ~Gradient();
				
				/**
					\brief		Assigns one Gradient object to another
					\author		GW
					\date		12/2013

					\param		rhs_	Existing Gradient object to copy values from
					\returns	The target Gradient object, but with its values updated to match that of the rhs_
								argument
				*/
				virtual Gradient& operator=(const Gradient& rhs_);

				/**
					\brief		Removes the gradient from its document
					\author		GW
					\date		12/2013
					
					\returns	true for successful disposal, false otherwise

					\note		After calling dispose(), this object is basically useless! The actual gradient is gone,
								so obviously none of these methods will work after that.
				*/
				virtual bool dispose();

				/**
					\brief		Gets the gradient handle around which the target object is wrapped
					\author		GW
					\date		12/2013
					
					\returns	AIGradientHandle for this object

					\note		Generally, you shouldn't need to call this method. Only use it if you know what you're
								doing. If a specific piece of functionality provided by Illustrator is not handled by
								this class (or related classes), then it should probably be added to the hdi_core library.
				*/
				virtual AIGradientHandle aiGradientHandle() const;

				/**
					\brief		Gets whether the target Gradient object is empty (constructed with the default ctor)
					\author		GW
					\date		12/2013
					
					\returns	true if the target Gradient object is empty, false otherwise
				*/
				virtual bool isEmpty() const;
				
				/**
					\brief		Tests the validity of the target gradient by searching for it in the document
					\author		GW
					\date		12/2013
					
					\returns	true for a valid gradient, false otherwise
				*/
				virtual bool valid() const;
				
				/**
					\brief		Gets the type of gradient the target object is representing
					\author		GW
					\date		12/2013
					
					\returns	A Type enum value indicating the type of the underlying gradient
				*/
				virtual Type gradientType() const;
				
				/**
					\brief	Sets the type of gradient the target object is representing
					\author	GW
					\date	12/2013
					
					\param	type_	Enum value indicating the new type of the underlying gradient
				*/
				virtual void setGradientType(const Type type_);
				
				/**
					\brief		Tests whether a given Gradient object is the same as another
					\author		GW
					\date		12/2013

					\param		rhs_	Gradient to compare against (righthand side of equality operator)
					\returns	true for the target and rhs_ being the same gradient, false otherwise
				*/
				virtual bool operator==(const Gradient& rhs_) const;
				
				/**
					\brief		Tests whether a given Gradient object is not the same as another
					\author		GW
					\date		12/2013

					\param		rhs_	Gradient to compare against (righthand side of inequality operator)
					\returns	true for the target and rhs_ being different gradients, false otherwise
				*/
				virtual bool operator!=(const Gradient& rhs_) const;

				/**
					\brief		Gets the name of the gradient
					\author		GW
					\date		11/2013
					
					\returns	The gradient's name, as UTF-8
				*/
				virtual std::string name() const;
				
				/**
					\brief	Sets the name of the gradient
					\author	GW
					\date	11/2013
					
					\param	name_	New name for the gradient, as UTF-8
				*/
				virtual void setName(const std::string& name_);
				
				/**
					\brief		Gets the number of stops (color transition or ramp points) in the target gradient
					\author		GW
					\date		12/2013
					
					\returns	Number of stops; there are always at least two
				*/
				virtual int16_t stopCount() const;

				/**
					\brief		Gets a gradient stop at the given index
					\author		GW
					\date		12/2013
					
					\param		index_	Index of the desired stop, in the range [0,stopCount-1]
					\returns	The gradient stop at index_
				*/
				virtual std::unique_ptr<Stop> stopAtIndex(const int16_t index_);

				/**
					\brief		Sets a gradient stop at the given index
					\author		GW
					\date		12/2013
					
					\param		index_	Index of the stop to modify, in the range [0,stopCount-1]
					\param		stop_	New gradient stop settings
					\returns	true for success, false otherwise
				*/
				virtual bool setStopAtIndex(const int16_t index_, const Stop& stop_);

				/**
					\brief		Inserts a gradient stop at the given index
					\author		GW
					\date		12/2013

					\param		index_	The position at which to add this stop, in the range [0,stopCount]
					\param		stop__	Return-by-reference for the newly inserted gradient stop
					\returns	true for success, false otherwise
					
					\note		The new stop will have the given index, and the indices of all subsequent stops are
								incremented.
				*/
				virtual bool insertStopAtIndex(const int16_t index_, Stop& stop__);

				/**
					\brief		Removes a gradient stop at the given index
					\author		GW
					\date		12/2013

					\param		index_	The position of the stop to remove, in the range [0,stopCount-1]
					\param		stop__	Return-by-reference for the gradient stop that was removed
					\returns	true for success, false otherwise
				*/
				virtual bool removeStopAtIndex(const int16_t index_, Stop& stop__);

				/**
					\brief		Gets the color at a given "t-value" (position) on a gradient ramp
					\author		GW
					\date		12/2013
					
					\param		tVal_	The point on the ramp, a percentage value in the range [0,1]
					\param		color__	Return-by-reference for the color at the given position
					\returns	true for success, false otherwise
					
					\note		If the position is a stop, the color of that stop is returned. Otherwise, an
								interpolated color between stops is returned.
				*/
				virtual bool colorAtPosition(const double tVal_, ArtColor& color__);


			protected:
				/**
					\brief	Private implementation data
				*/
				void* _data;
				
				/**
					\brief	Internal use only
					\author	GW
					\date	11/2013
				*/
				void* _impl() const;


			private:
				friend aip::Gradient* __accessImpl(const Gradient&);
				friend Gradient __accessCtor(aip::Gradient*&);

				/**
					\brief	Internal use only
					\author	GW
					\date	11/2013
				*/
				Gradient(aip::Gradient*&);
		};
		
		typedef std::unique_ptr<Gradient> GradientUP;
		typedef std::shared_ptr<Gradient> GradientSP;
		typedef std::weak_ptr<Gradient> GradientWP;
		
		extern aip::Gradient* __accessImpl(const Gradient&);
		extern Gradient __accessCtor(aip::Gradient*&);
	}
}

#endif
// HDI_CORE_AIP_MODE

#endif
// __HDI_CORE_GRADIENT__
