/**
	\file
	\brief		Header file for plugin user interface list widgets
	\copyright	Hot Door, Inc. 2010-2025
*/

#ifndef __HDI_CORE_LIST_VIEW__
#define __HDI_CORE_LIST_VIEW__

#include <vector>

#include "hdicoreWidget.h"

namespace hdi
{
	namespace core
	{
		class Callback;
		class ListEntry;

		/**
			\brief	List view widget, much like a popup menu but allowing for multiple entries to be seen at a time
		*/
		class ListView : public Widget
		{
			public:
				typedef std::vector<ListEntry> EntryVector;
			
				/**
					\brief	Constructs an empty ListView object
					\author	GW
					\date	09/2013
					
					\note	To test if a ListView object is empty, call isEmpty() on it
					\note	Empty ListView objects do not relate to any actual UI widget; they are designed to be
							"receivers" of some other ListView object via the overloaded assignment operator. Empty
							ListView objects are useless until such time (though it is safe to call any of their methods).
				*/
				ListView();

				/**
					\brief	Constructs a new ListView object from an existing ListView object (copy constructor)
					\author	GW
					\date	09/2013

					\param	listView_	Existing ListView object
				*/
				ListView(const ListView& listView_);

				/**
					\brief	Constructs a list view with a frame only, no contents
					\author	GW
					\date	09/2013

					\param	frame_	Position and size of the list, in 1x resolution coordinates
				*/
				ListView(const Rect& frame_);

				/**
					\brief	Constructs a list view
					\author	GW
					\date	09/2013

					\param	frame_		Position and size of the list, in 1x resolution coordinates
					\param	entries_	Vector of entries
					\param	index_		Initial index to select (must exist in entries_)
				*/
				ListView(const Rect& frame_, const EntryVector& entries_, const int32_t index_);

				/**
					\brief	Constructor for a list view with an initial ID selected
					\author	GW
					\date	09/2013

					\param	frame_		Position and size of the list, in 1x resolution coordinates
					\param	entries_	Vector of entries
					\param	initialID_	Initial ID to select (must exist in entries_, and must not be a separator), as
										UTF-8
				*/
				ListView(const Rect& frame_, const EntryVector& entries_, const std::string& initialID_);

				/**
					\brief		Allows one ListView object to be assigned from another
					\author		GW
					\date		10/2013

					\param		rhs_	Righthand side of the = operator; the object to copy values from
					\returns	The target ListView object, but with its value updated to match that of rhs_
				*/
				virtual ListView& operator=(const ListView& rhs_);
				
				/**
					\brief	ListView destructor
					\author	GW
					\date	09/2013
				*/
				virtual ~ListView();
				
				/**
					\brief		Convenience method to clone a ListView object on the heap
					\author		GW
					\date		10/2013

					\returns	A pointer to the new ListView object
					
					\note		If you subclass ListView, you MUST overload this method yourself! If you don't and/or
								your clone() method does not return an instance of your ListView subclass, you will
								experience "object slicing" when adding the widget to a window.
								
					\warning	The caller must manage the memory for the returned ListView object.
				*/
				virtual ListView* clone() const;
				
				/**
					\brief		Convenience method to duplicate a ListView object, creating a new and identical UI
								element to the target (but not belonging to the same containing window)
					\author		GW
					\date		11/2013
					
					\returns	A pointer to the new ListView object (and new UI element)
					
					\note		The new list view will have copies of all entries present in the target, but be aware
								that copies of the user data in each entry cannot be made (because they are simply void*
								variables). As such, the user data in the copies will simply be set to NULL by this
								method.

					\warning	The caller must manage the memory for the returned ListView object.
				*/
				virtual ListView* duplicate() const;

				/**
					\brief		Gets the type of the widget
					\author		GW
					\date		07/2021
					
					\returns	The widget type
				*/
				virtual Type type() const;

				/**
					\brief	Deselects all currently selected entries in the list
					\author	GW
					\date	09/2013
				*/
				virtual void deselectAll();

				/**
					\brief		Gets the index of the currently selected entry
					\author		GW
					\date		09/2013

					\returns	The index number of the currently selected entry, or -1 if no entry is selected
				*/
				virtual int32_t currentIndex() const;
				
				/**
					\brief		Sets the currently selected entry by index
					\author		GW
					\date		09/2013

					\param		index_	New index to select
					\returns	true if the selection occurred successfully, or false if e.g. out of range
				*/
				virtual bool setCurrentIndex(const int32_t index_);
				
				/**
					\brief		Gets the ID of the currently selected entry
					\author		GW
					\date		09/2013

					\returns	UTF-8 ID string of the currently selected entry, or "" if no entry is currently selected
				*/
				virtual std::string currentID() const;
				
				/**
					\brief		Sets the currently selected entry by ID
					\author		GW
					\date		09/2013

					\param		id_		ID of new value to select in the list view, as UTF-8
					\returns	true if the entry with the given ID was selected, or false if invalid
				*/
				virtual bool setCurrentID(const std::string& id_);
				
				/**
					\brief		Gets the string value of the currently selected entry
					\author		GW
					\date		09/2013

					\returns	Value string of the list view, as UTF-8
				*/
				virtual std::string currentValue() const;
				
				/**
					\brief		Sets the currently selected entry by string value
					\author		GW
					\date		09/2013

					\param		value_	New value to set for the list view (if it exists as an entry), as UTF-8
					\returns	true if the value was set, false otherwise
				*/
				virtual bool setCurrentValue(const std::string& value_);

				/**
					\brief		Gets the current entry object
					\author		GW
					\date		09/2013

					\returns	The ListEntry object for the currently selected item
				*/
				virtual std::unique_ptr<ListEntry> currentEntry() const;
				
				/**
					\brief		Gets the user data for the current entry
					\author		GW
					\date		09/2013

					\param		data__	Return-by-reference for the current user data; ignore if false is returned
					\returns	true if the method succeeds in acquiring the contained entry data, false otherwise
				*/
				virtual bool currentUserData(void*& data__) const;

				/**
					\brief		Gets the entries vector
					\author		GW
					\date		09/2013

					\returns	The entries vector for the target widget, containing all currently known entries
					
					\note		Modifying any value in an entry will not cause the widget to update automatically; call
								update() to update the UI for changes made to entry data.
				*/
				virtual const std::vector< std::shared_ptr<ListEntry> >* const entries() const;
				
				/**
					\brief		Gets whether the entries vector already contains an entry, based on its ID
					\author		GW
					\date		09/2013

					\param		id_		ID of the entry to search for, as UTF-8
					\returns	true if the widget already contains an entry with the given ID, false otherwise
				*/
				virtual bool hasEntry(const std::string& id_) const;
				
				/**
					\brief		Gets the entry with the given ID, provided it exists within the widget
					\author		GW
					\date		12/2022

					\param		id_		ID of the entry to search for
					\returns	The entry with the given ID, or NULL if not found
				*/
				virtual std::shared_ptr<ListEntry> entryWithID(const std::string& id_) const;
				
				/**
					\brief	Allows the entries in a list view to be changed out entirely
					\author	GW
					\date	09/2013

					\param	entries_	New set of entries that the widget will contain
				*/
				virtual void setEntries(const EntryVector& entries_);

				/**
					\brief		Pushes an entry onto the list (i.e. at the end)
					\author		GW
					\date		09/2013

					\param		entry_	New entry object to insert into the list of entries
					\returns	true if the entry is pushed successfully, false otherwise

					\note		If the entry is already present in the list (based on its ID), it will not be inserted.
				*/
				virtual bool pushEntry(const ListEntry& entry_);

				/**
					\brief		Pops an entry off the list (i.e. from the end)
					\author		GW
					\date		09/2013

					\param		entry__		Return-by-reference for the popped entry, so the caller can inspect which
											one was removed
					\returns	true if an entry was popped and acquired, false otherwise
				*/
				virtual bool popEntry(ListEntry& entry__);

				/**
					\brief		Inserts an entry into the list of entries at a given position
					\author		GW
					\date		09/2013

					\param		id_		Existing entry immediately before the position at which the new entry will be
										inserted, as UTF-8
					\param		entry_	New entry object to insert into the list of entries
					\returns	true if the entry is inserted successfully, false otherwise

					\note		If the entry for id_ doesn't exist in the list, the new entry will not be inserted.
					\note		If the passed entry is already present in the list (based on its ID), it will not be
								inserted.
				*/
				virtual bool insertEntry(const std::string& id_, const ListEntry& entry_);

				/**
					\brief		Removes a specific entry from the list of entries
					\author		GW
					\date		09/2013

					\param		id_			ID of the entry to be removed, as UTF-8
					\param		entry__		Return-by-reference for the removed entry, so the caller can inspect which
											one was removed
					\returns	true if an entry was removed and acquired, false otherwise
				*/
				virtual bool removeEntry(const std::string& id_, ListEntry& entry__);

				/**
					\brief		Reorders a target entry (based on ID) to be newly located just after another existing
								entry in the list
					\author		GW
					\date		12/2022

					\param		targetID_	ID of an existing entry in the list to be relocated
					\param		positionID_	ID of an existing entry immediately before the position at which targetID_
											should be moved; if the passed value is an empty string, targetID_ will be
											repositioned at the beginning
					\returns	true if the entry is repositioned successfully, false otherwise

					\note		If the entry for targetID_ or for positionID_ doesn't exist in the list, this method
								will bail.
				*/
				virtual bool reorderEntry(const std::string& targetID_, const std::string& positionID_);

				/**
					\brief	Clears all entries contained by the widget
					\author	GW
					\date	09/2013
				*/
				virtual void clearEntries();

				/**
					\brief	Simulates the list view having its value changed
					\author	GW
					\date	09/2013
				*/
				virtual void valueChanged();
				
				/**
					\brief		Gets the value changed callback for the list view
					\author		GW
					\date		09/2013

					\returns	A pointer to the currently registered value changed callback

					\note		In this context, "value" refers to the selected index/ID changing.
				*/
				virtual Callback* const valueChangedCallback() const;
				
				/**
					\brief	Sets the value changed callback
					\author	GW
					\date	09/2013

					\param	callback_	New callback for when the list view's value has changed

					\note	In this context, "value" refers to the selected index/ID changing.
				*/
				virtual void setValueChangedCallback(const Callback& callback_);
				
				/**
					\brief		Gets the double-click callback for the list view
					\author		GW
					\date		11/2024

					\returns	A pointer to the currently registered double-click callback
				*/
				virtual Callback* const doubleClickCallback() const;
				
				/**
					\brief	Sets the double-click callback
					\author	GW
					\date	11/2024

					\param	callback_	New callback for when the list view's current entry is double clicked

					\note	This callback will not be fired when editable entries are double clicked.
				*/
				virtual void setDoubleClickCallback(const Callback& callback_);
				
				/**
					\brief	Force the list view to update its entries
					\author	GW
					\date	12/2013
				*/
				virtual void update();
				
				/**
					\brief		Gets the callback for when a cell is modified via double-click inline editing
					\author		GW
					\date		07/2022

					\returns	A pointer to the currently registered cell modification callback
				*/
				virtual Callback* const cellChangedCallback() const;
				
				/**
					\brief	Sets the callback for when a cell is modified via double-click inline editing
					\author	GW
					\date	07/2022

					\param	callback_	New callback for a user modifying a cell via inline editing

					\note	When your callback is executed, you can determine which entry was clicked by calling the
							changedEntry() method.
				*/
				virtual void setCellChangedCallback(const Callback& callback_);
				
				/**
					\brief		Gets the entry that was last modified via double-click inline editing
					\author		GW
					\date		07/2022
					
					\returns	The changed entry object
					
					\note		The return value of this method is only guaranteed to be valid during the execution of
								the cell changed callback.
				*/
				virtual std::unique_ptr<ListEntry> changedEntry() const;
				
				/**
					\brief		Gets whether reordering support for the list is enabled
					\author		GW
					\date		12/2022
				 
					\returns	true if drag-and-drop reordering support is enabled, false otherwise
				*/
				virtual bool reorderable() const;
			
				/**
					\brief	Sets whether reordering support for the list is enabled
					\author	GW
					\date	12/2022
				 
					\param	enable_		true to enable drag-and-drop reordering support, false otherwise
				*/
				virtual void setReorderable(const bool enable_);

				/**
					\brief		Gets the current potential-reorder callback for the list
					\author		GW
					\date		12/2022

					\returns	A pointer to the currently registered potential-reorder callback
				*/
				virtual Callback* const potentialReorderCallback() const;
			
				/**
					\brief	Sets the potential-reorder callback for the widget
					\author	GW
					\date	12/2022

					\param	callback_	New callback (target and action) for each time the potential reorder target changes
					
					\note	When your callback is executed, you can determine which entry is being proposed for reorder
							and its proposed new position by calling the potentialReorderSource() and
							potentialReorderPosition() methods, respectively.
				*/
				virtual void setPotentialReorderCallback(const Callback& callback_);
			
				/**
					\brief		Gets the source item involved in the potential reorder, i.e. the entry being dragged
					\author		GW
					\date		12/2022
				 
					\returns	The ID path of the entry that might be reordered
				 
					\note		The return value of this method is only guaranteed to be valid during the execution of
								the potential-reorder callback.
				*/
				virtual std::vector<std::string> potentialReorderSource() const;

				/**
					\brief		Gets the position for the reorder, i.e. the entry just before the proposed reposition
					\author		GW
					\date		12/2022
				 
					\returns	The ID path of the entry just before the potential drop point for a reposition; if the
								returned vector contains only an empty string, this refers to the proposed drop position
								being the first at the root level.
				 
					\note		The return value of this method is only guaranteed to be valid during the execution of
								the potential-reorder callback.
				*/
				virtual std::vector<std::string> potentialReorderPosition() const;

				/**
					\brief		Gets whether the potential reorder is acceptable; if not, the drag cursor will indicate
								the proposed drop position is not allowed
					\author		GW
					\date		12/2022
				
					\returns	true if the specific reorder drop position is allowed, false otherwise
					
					\note		The return value of this method is only guaranteed to be valid during the execution of
								the potential-reorder callback.
				*/
				virtual bool potentialReorderGood() const;

				/**
					\brief	Sets whether the potential reorder is acceptable; if not, the drag cursor will indicate the
							proposed drop position is not allowed
					\author	GW
					\date	12/2022
				
					\param	good_	Whether the potential reorder position is acceptable
					
					\note	This method will have no effect if it is called any other time than from within a potential-
							reorder callback.
					\note	A potential reorder is always assumed to be good, so the true/false nature of a potential
							reorder will always be true unless you have called this method within your callback and
							passed false.
					\note	Calling this method and passing false will not stop a potential reorder from occurring
							altogether; rather, it merely indicates whether a specific reorder proposed drop position is
							acceptable. It is still up to you whether the entries, and any underlying associated data,
							is actually reordered once your reorder callback is executed.
				*/
				virtual void setPotentialReorderGood(const bool good_);

				/**
					\brief		Gets the current reorder callback for the widget
					\author		GW
					\date		12/2022

					\returns	A pointer to the currently registered reorder callback
				*/
				virtual Callback* const reorderCallback() const;
			
				/**
					\brief	Sets the reorder callback for the widget
					\author	GW
					\date	12/2022

					\param	callback_	New callback (target and action) for when a reorder is finalized (via dropping)
					
					\note	When your callback is executed, you can determine which entry is being reordered and its new
							position by calling the reorderSource() and reorderPosition() methods, respectively.
				*/
				virtual void setReorderCallback(const Callback& callback_);
			
				/**
					\brief		Gets the source item involved in the reorder, i.e. the entry being dropped
					\author		GW
					\date		12/2022
				 
					\returns	The ID path of the entry that was reordered
				 
					\note		The return value of this method is only guaranteed to be valid during the execution of
								the reorder callback.
				*/
				virtual std::vector<std::string> reorderSource() const;

				/**
					\brief		Gets the position for the reorder, i.e. the entry just before the reposition
					\author		GW
					\date		12/2022
				 
					\returns	The ID path of the entry just before the drop point for a reposition; if the returned
								vector contains only an empty string, this refers to the drop position being the first
								at the root level.
				 
					\note		The return value of this method is only guaranteed to be valid during the execution of
								the reorder callback.
				*/
				virtual std::vector<std::string> reorderPosition() const;
		};
		
		typedef std::unique_ptr<ListView> ListViewUP;
		typedef std::shared_ptr<ListView> ListViewSP;
		typedef std::weak_ptr<ListView> ListViewWP;
	}
}

#endif
// __HDI_CORE_LIST_VIEW__
