/**
	\file
	\brief		Header file for Illustrator live effect manipulation
	\copyright	Hot Door, Inc. 2010-2025
*/

#ifndef __HDI_CORE_LIVE_EFFECT__
#define __HDI_CORE_LIVE_EFFECT__

#if defined(HDI_CORE_AIP_MODE)

#include "hdicorePathStyle.h"
#include "hdicoreTransformMatrix.h"
#include "hdicoreTypes.h"

namespace hdi
{
	namespace core
	{
		class LiveEffect
		{
			public:
				/**
					\brief	Describes the parameters of an Illustrator live effect
				*/
				class Parameters
				{
					public:
						/**
							\brief	Constructs an empty Parameters object
							\author	GW
							\date	08/2014
							
							\note	To test if a Parameters object is empty, call isEmpty() on it
							\note	Empty Parameters objects do not relate to any actual Illustrator live effect
									parameters; they are designed to be "receivers" of some other Parameters object via
									the overloaded assignment operator. Empty Parameters objects are useless until such
									time (though it is safe to call any of their methods)
						*/
						Parameters();
					
						/**
							\brief	Constructs a new Parameters object from an existing Parameters object (copy ctor)
							\author	GW
							\date	08/2014

							\param	p_	Existing Parameters object
						*/
						Parameters(const Parameters& p_);

						/**
							\brief	Parameters object destructor
							\author	GW
							\date	08/2014
						*/
						virtual ~Parameters();
						
						/**
							\brief		Assigns one Parameters object to another
							\author		GW
							\date		08/2014

							\param		rhs_	Existing Parameters object to copy values from
							\returns	The target Parameters object, but with its values updated to match that of the
										rhs_ argument
						*/
						virtual Parameters& operator=(const Parameters& rhs_);

						/**
							\brief		Gets whether the target Parameters object is empty (constructed with the default
										ctor)
							\author		GW
							\date		08/2014
							
							\returns	true if the target Parameters object is empty, false otherwise
						*/
						virtual bool isEmpty() const;

						/**
							\brief		Gets the live effect param ref around which the target object is wrapped
							\author		GW
							\date		08/2014
							
							\returns	AILiveEffectParameters for this object

							\note		Generally, you shouldn't need to call this method. Only use it if you know what
										you're doing. If a specific piece of functionality provided by Illustrator is
										not handled by this class (or related classes), then it should probably be added
										to the hdi_core library.
						*/
						virtual AILiveEffectParameters aiLiveEffectParams() const;
						
						/**
							\brief		Tests whether a given Parameters object is the same as another
							\author		GW
							\date		08/2014

							\param		rhs_	Parameters to compare against (righthand side of equality operator)
							\returns	true for the target and rhs_ being the same live effect parameters, false
										otherwise
						*/
						virtual bool operator==(const Parameters& rhs_) const;
						
						/**
							\brief		Tests whether a given Parameters object is not the same as another
							\author		GW
							\date		08/2014

							\param		rhs_	Parameters to compare against (righthand side of inequality operator)
							\returns	true for the target and rhs_ being different live effect parameters, false
										otherwise
						*/
						virtual bool operator!=(const Parameters& rhs_) const;
						
						/**
							\brief		Gets the string that is displayed by the panel in reference to the filter name,
										if it is set
							\author		GW
							\date		08/2014
							
							\returns	The display string, as UTF-8
						*/
						virtual std::string displayString() const;
						
						/**
							\brief		Gets the string that gives extra information for the panel to display for this
										filter, e.g. as a tool tip
							\author		GW
							\date		08/2014
							
							\returns	The extra string, as UTF-8
						*/
						virtual std::string extraString() const;
						
						/**
							\brief		Gets the fill style for the target param
							\author		GW
							\date		08/2014
							
							\param		fs__	Return-by-reference for the fill style
							\returns	true if the fill style could be acquired, false otherwise
						*/
						virtual bool fillStyle(FillStyle& fs__) const;
					
						/**
							\brief		Gets the fill transform for the target param
							\author		GW
							\date		08/2014
							
							\param		m__		Return-by-reference for the matrix
							\returns	true if the matrix could be acquired, false otherwise

							\note		If the object's fill style matrix is not identity, this will be set in the
										instance info. The matrix will also be factored into the transformed fill style,
										but it contains information that is not available in the gradient fields, such
										as whether there is a reflection.
						*/
						virtual bool fillStyleMatrix(TransformMatrix& m__) const;
					
						/**
							\brief		Gets the fill style for the target param, but with the object's style matrix
										multiplied in for gradient fills
							\author		GW
							\date		08/2014
							
							\param		fs__	Return-by-reference for the fill style
							\returns	true if the fill style could be acquired, false otherwise
						*/
						virtual bool xformedFillStyle(FillStyle& fs__) const;
						
						/**
							\brief		Gets the stroke style for the target param
							\author		GW
							\date		08/2014
							
							\param		ss__	Return-by-reference for the stroke style
							\returns	true if the stroke style could be acquired, false otherwise
						*/
						virtual bool strokeStyle(StrokeStyle& ss__) const;
					
						/**
							\brief		Gets the stroke transform for the target param
							\author		GW
							\date		08/2014
							
							\param		m__		Return-by-reference for the matrix
							\returns	true if the matrix could be acquired, false otherwise

							\note		If the object's stroke style matrix is not identity, this will be set in the
										instance info. The matrix will also be factored into the transformed stroke
										style, but it contains information that is not available in the gradient fields,
										such as whether there is a reflection.
						*/
						virtual bool strokeStyleMatrix(TransformMatrix& m__) const;
						
						/**
							\brief		Gets the stroke style for the target param, but with the object's style matrix
										multiplied in for gradient stroke
							\author		GW
							\date		08/2014
							
							\param		fs__	Return-by-reference for the stroke style
							\returns	true if the fill style could be acquired, false otherwise
						*/
						virtual bool xformedStrokeStyle(StrokeStyle& ss__) const;

						/**
							\brief		Gets the even-odd fill rule for the target param
							\author		GW
							\date		08/2014
							
							\returns	true if the even-odd rule is used to determine path insideness
						*/
						virtual bool evenOdd() const;
						
						/**
							\brief		Gets the scale factor for the target param
							\author		GW
							\date		08/2014
							
							\returns	The scale factor
						*/
						virtual double scaleFactor() const;


					private:
						friend void* __accessImpl(const Parameters&);
						friend Parameters __accessCtor(AILiveEffectParameters);

						/**
							\brief	Private implementation data
						*/
						void* __data;
						
						/**
							\brief	Internal use only
							\author	GW
							\date	08/2014
						*/
						void* __impl() const;

						/**
							\brief	Internal use only
							\author	GW
							\date	08/2014
						*/
						Parameters(AILiveEffectParameters);
				};
		
				typedef std::unique_ptr<Parameters> ParametersUP;
				typedef std::shared_ptr<Parameters> ParametersSP;
				typedef std::weak_ptr<Parameters> ParametersWP;
				
				enum PreferredInputArtType
				{
					InputArtTypeDynamic			= 0,
					InputArtTypeGroup			= 1 << 0,
					InputArtTypePath			= 1 << 1,
					InputArtTypeCompoundPath	= 1 << 2,
					InputArtTypePlaced			= 1 << 3,
					InputArtTypeRaster			= 1 << 4,
					InputArtTypeCustom			= 1 << 5,
					InputArtTypeMesh			= 1 << 6,
					InputArtTypeText			= 1 << 7,
					InputArtTypeSymbol			= 1 << 8,
					InputArtTypeForeign			= 1 << 9,
					InputArtTypeTextLegacy		= 1 << 10,
					InputArtTypeAll				= ~0L
				};
				
				/**
					\brief	Describes the style of a live effect
				*/
				enum StyleFlags
				{
					NoStyleFlags			= 0,
					
					// Applied by default before the object is painted with fill or stroke
					PreEffectFilter			= 1,
					
					// Applied by default after the object is painted with fill or stroke
					PostEffectFilter		= 2,
					
					// Replaces the default stroke behavior. Brushes are an example of an effect of this type.
					StrokeFilter			= 3,
					
					// Replaces the default fill behavior
					FillFilter				= 4,
					
					// A mask to OR with the filter-style value to retrieve specific bit flags. Don't use when
					// registering a live effect.
					FilterTypeMask			= 0xFFFF,
					
					// Parameters can be scaled
					HasScalableParams		= 1 << 17,
					
					// Supports automatic rasterization
					UsesAutoRasterize		= 1 << 18,
					
					// Supports the generation of an SVG filter
					CanGenerateSVGFilter	= 1 << 19,
					
					// Has parameters that can be modified by a LiveEffectAdjustColorsMessageType message
					HandlesAdjustColorMsg	= 1 << 20,
					
					// Handles LiveEffectIsCompatibleMessageType messages. If this flag is not set the message will not
					// be sent.
					HandlesIsCompatibleMsg	= 1 << 21
				};
				
				/**
					\brief	Describes the merge action of a live effect
				*/
				enum MergeAction
				{
					UnknownMergeAction			= 0,
					AppendMergeAction			= 10,
					ReplaceFillMergeAction		= 20,
					ReplaceStrokeMergeAction	= 30
				};

				/**
					\brief	Constructs an empty LiveEffect object
					\author	GW
					\date	08/2014
					
					\note	To test if a LiveEffect object is empty, call isEmpty() on it
					\note	Empty LiveEffect objects do not relate to any actual Illustrator live effects; they are
							designed to be "receivers" of some other LiveEffect object via the overloaded assignment
							operator. Empty LiveEffect objects are useless until such time (though it is safe to call
							any of their methods)
				*/
				LiveEffect();
			
				/**
					\brief	Constructs a new LiveEffect object from an existing LiveEffect object (copy constructor)
					\author	GW
					\date	08/2014

					\param	le_		Existing LiveEffect object
				*/
				LiveEffect(const LiveEffect& le_);

				/**
					\brief	Constructs a new LiveEffect object from an AILiveEffectHandle
					\author	GW
					\date	12/2014
					
					\param	aiLiveEff_	Raw Illustrator AILiveEffectHandle
				*/
				LiveEffect(const AILiveEffectHandle aiLiveEff_);

				/**
					\brief	LiveEffect object destructor
					\author	GW
					\date	08/2014
				*/
				virtual ~LiveEffect();
				
				/**
					\brief		Assigns one LiveEffect object to another
					\author		GW
					\date		08/2014

					\param		rhs_	Existing LiveEffect object to copy values from
					\returns	The target LiveEffect object, but with its values updated to match that of the rhs_
								argument
				*/
				virtual LiveEffect& operator=(const LiveEffect& rhs_);

				/**
					\brief		Gets whether the target LiveEffect object is empty (constructed with the default ctor)
					\author		GW
					\date		08/2014
					
					\returns	true if the target LiveEffect object is empty, false otherwise
				*/
				virtual bool isEmpty() const;

				/**
					\brief		Gets the live effect handle around which the target object is wrapped
					\author		GW
					\date		08/2014
					
					\returns	AILiveEffectHandle for this object

					\note		Generally, you shouldn't need to call this method. Only use it if you know what you're
								doing. If a specific piece of functionality provided by Illustrator is not handled by
								this class (or related classes), then it should probably be added to the hdi_core library.
				*/
				virtual AILiveEffectHandle aiLiveEffectHandle() const;
				
				/**
					\brief		Tests whether a given LiveEffect object is the same as another
					\author		GW
					\date		08/2014

					\param		rhs_	LiveEffect to compare against (righthand side of equality operator)
					\returns	true for the target and rhs_ being the same live effect, false otherwise
				*/
				virtual bool operator==(const LiveEffect& rhs_) const;
				
				/**
					\brief		Tests whether a given LiveEffect object is not the same as another
					\author		GW
					\date		08/2014

					\param		rhs_	LiveEffect to compare against (righthand side of inequality operator)
					\returns	true for the target and rhs_ being different live effects, false otherwise
				*/
				virtual bool operator!=(const LiveEffect& rhs_) const;
				
				/**
					\brief		Gets the unique name of the target effect
					\author		GW
					\date		08/2014
					
					\returns	Unique name given to the effect when it was created, as UTF-8
				*/
				virtual std::string name() const;
				
				/**
					\brief		Gets the UI-visible title of the effect
					\author		GW
					\date		08/2014
					
					\returns	The effect title, as UTF-8
				*/
				virtual std::string title() const;
				
				/**
					\brief		Gets the version info for the target live effect
					\author		GW
					\date		08/2014
					
					\param		major__		Return-by-reference for the major version
					\param		minor__		Return-by-reference for the minor version
					\returns	true if the version info could be acquired
				*/
				virtual bool version(int32_t& major__, int32_t& minor__) const;

				/**
					\brief		Gets the preferred input art types of the target effect
					\author		GW
					\date		08/2014
					
					\returns	The input art preferences, a bitwise OR of PreferredInputArtType values
				*/
				virtual PreferredInputArtType inputArtPreference() const;

				/**
					\brief		Gets the style flags of the target effect
					\author		GW
					\date		08/2014
					
					\returns	The style flags, a bitwise OR of StyleFlags values
				*/
				virtual StyleFlags styleFlags() const;

				/**
					\brief		Merges the target effect into the existing style of all selected objects
					\author		GW
					\date		08/2014
					
					\param		act_	The merge method (append, or replace fill and/or stroke)
					\returns	true if the merge could take place, false otherwise
				*/
				virtual bool mergeIntoSelection(const MergeAction act_);

				/**
					\brief		Gets the earliest Illustrator version that an effect supports
					\author		GW
					\date		08/2014
					
					\param		rgbVers__	Return-by-reference for the earliest version number that the effect supports
											for RGB documents
					\param		cmykVers__	Return-by-reference for the earliest version number that the effect supports
											for CMYK documents
					\returns	true if the version numbers could be acquired, false otherwise
					
					\note		The version number directly corresponds to the major version of Illustrator.
				*/
				virtual bool getEarliestAppVersion(int16_t& rgbVers__, int16_t& cmykVers__) const;

				/**
					\brief	Sets the earliest Illustrator version that an effect supports
					\author	GW
					\date	08/2014
					
					\param	rgbVers_	The earliest version number that the effect supports for RGB documents
					\param	cmykVers_	The earliest version number that the effect supports for CMYK documents
					
					\note	The version number directly corresponds to the major version of Illustrator, and must be at
							least 9.
				*/
				virtual void setEarliestAppVersion(const int16_t rgbVers_, const int16_t cmykVers_);


			private:
				friend void* __accessImpl(const LiveEffect&);

				/**
					\brief	Private implementation data
				*/
				void* __data;
				
				/**
					\brief	Internal use only
					\author	GW
					\date	08/2014
				*/
				void* __impl() const;
		};
		
		typedef std::unique_ptr<LiveEffect> LiveEffectUP;
		typedef std::shared_ptr<LiveEffect> LiveEffectSP;
		typedef std::weak_ptr<LiveEffect> LiveEffectWP;
		
		extern void* __accessImpl(const LiveEffect::Parameters&);
		extern LiveEffect::Parameters __accessCtor(AILiveEffectParameters);
		extern void* __accessImpl(const LiveEffect&);
	}
}

hdi::core::LiveEffect::PreferredInputArtType operator|(
	const hdi::core::LiveEffect::PreferredInputArtType lhs_,
	const hdi::core::LiveEffect::PreferredInputArtType rhs_
);

hdi::core::LiveEffect::PreferredInputArtType& operator|=(
	hdi::core::LiveEffect::PreferredInputArtType& lhs__,
	const hdi::core::LiveEffect::PreferredInputArtType rhs_
);

#endif
// HDI_CORE_AIP_MODE

#endif
// __HDI_CORE_LIVE_EFFECT__
