/**
	\file
	\brief		Header file for various plugin entry points
	\copyright	Hot Door, Inc. 2010-2025
*/

#ifndef __HDI_CORE_MAIN__
#define __HDI_CORE_MAIN__

#include "hdicoreTypes.h"

namespace hdi
{
	namespace core
	{
		class Plugin;
	
		/**
			\brief	Write this function yourself to receive a call immediately after your plugin dynamic library is
					loaded by the parent application (i.e. before setup/startup has begun)
			\author	GW
			\date	01/2018
			
			\note	It is imperative that you define this function yourself as a consumer of this library! First of
					all, your plugin won't link without a definition, and this will be the only chance you've got to
					perform pre-setup/pre-startup tasks (if needed). This is probably most applicable to Photoshop, where
					any non-action plugin does not receive a startup message until receiving its first message from the
					app (due to a user action).
		*/
		extern void pluginLoaded();
	
		/**
			\brief		Write this function yourself to receive a call immediately after the Plugin instance is
						instantiated and before Plugin::__startup() is called
			\author		GW
			\date		09/2013
			
			\param		plugin_		Recently-instantiated plugin object
			\returns	true if the setup process completes successfully, false otherwise
			
			\note		It is imperative that you define this function yourself as a consumer of this library! First of
						all, your plugin won't link without a definition, and this will be the only chance you've got to
						set some basic callbacks on the plugin_ argument. Without doing so, you'll never receive startup,
						shutdown, tool, etc. messages!
			\note		When this function returns false, an error dialog will be shown at startup displaying the error
						code from the setupFailedErrorCode constant. It is recommended that in your production build you
						handle error conditions properly in your pluginSetup() definition, such that an end-user will
						never receive said error code.
		*/
		extern bool pluginSetup(Plugin* const plugin_);
		
		/**
			\brief	Write this function yourself to receive a call immediately after the Plugin instance has been
					shutdown but before it is unloaded (after Plugin::__shutdown() is called)
			\author	GW
			\date	11/2013
			
			\param	plugin_		Current plugin object

			\note	It is imperative that you define this function yourself as a consumer of this library! First of all,
					your plugin won't link without a definition, and this will be the only chance you've got to destroy
					any remaining memory that your plugin allocated before it is unloaded.
		*/
		extern void pluginDestroy(Plugin* const plugin_);
	}
}



#if defined(MAC_ENV)
	#define __HDI_CORE_EXPORT	__attribute__((visibility ("default")))
#elif defined(WIN_ENV)
	#define __HDI_CORE_EXPORT	__declspec(dllexport)
#endif

#if defined(MAC_ENV)
	#pragma GCC visibility push(default)

	extern "C" __attribute__((constructor)) void hdicoreDylibLoaded();
	extern "C" __attribute__((destructor)) void hdicoreDylibUnloading();

	#pragma GCC visibility pop
#endif

#if defined(HDI_CORE_AIP_MODE)
	/**
		\brief	Main entry point for all plugins (required by Illustrator). This is called whenever a plugin should
				perform some action or respond to some message.
		\author	GW
		\date	09/2013
	*/
	extern "C" __HDI_CORE_EXPORT __MainError PluginMain(const char* caller_, const char* selector_, void* data_);
#endif
// HDI_CORE_AIP_MODE

#if defined(HDI_CORE_PSP_MODE)
	/**
		\brief	Main entry point for action plugins (required by Photoshop). This is called whenever a plugin should
				perform some action or respond to some message.
		\author	GW
		\date	12/2017
	*/
	extern "C" __HDI_CORE_EXPORT __MainError ActionsPluginMain(const char* caller_, const char* selector_, void* data_);

	/**
		\brief	Main entry point for acquire plugins (required by Photoshop). This is called whenever a plugin should
				perform some action or respond to some message.
		\author	GW
		\date	12/2017
	*/
	extern "C" __HDI_CORE_EXPORT void AcquirePluginMain(
		const int16_t selector_,
		void* /* AcquireRecordPtr */ pb_,
		intptr_t* data__,
		int16_t* result__
	);

	/**
		\brief	Main entry point for export plugins (required by Photoshop). This is called whenever a plugin should
				perform some action or respond to some message.
		\author	GW
		\date	12/2017
	*/
	extern "C" __HDI_CORE_EXPORT void ExportPluginMain(
		const int16_t selector_,
		void* /* ExportRecordPtr */ pb_,
		intptr_t* data__,
		int16_t* result__
	);

	/**
		\brief	Main entry point for extension plugins (required by Photoshop). This is called whenever a plugin should
				perform some action or respond to some message.
		\author	GW
		\date	12/2017
	*/
	extern "C" __HDI_CORE_EXPORT void ExtensionPluginMain(
		const int16_t selector_,
		void* /* unknown */ pb_,
		intptr_t* data__,
		int16_t* result__
	);

	/**
		\brief	Main entry point for filter plugins (required by Photoshop). This is called whenever a plugin should
				perform some action or respond to some message.
		\author	GW
		\date	12/2017
	*/
	extern "C" __HDI_CORE_EXPORT void FilterPluginMain(
		const int16_t selector_,
		void* /* FilterRecordPtr */ pb_,
		intptr_t* data__,
		int16_t* result__
	);

	/**
		\brief	Main entry point for format plugins (required by Photoshop). This is called whenever a plugin should
				perform some action or respond to some message.
		\author	GW
		\date	12/2017
	*/
	extern "C" __HDI_CORE_EXPORT void FormatPluginMain(
		const int16_t selector_,
		void* /* FormatRecordPtr */ pb_,
		intptr_t* data__,
		int16_t* result__
	);

	/**
		\brief	Main entry point for measurement plugins (required by Photoshop). This is called whenever a plugin should
				perform some action or respond to some message.
		\author	GW
		\date	12/2017
	*/
	extern "C" __HDI_CORE_EXPORT void MeasurementPluginMain(
		const int16_t selector_,
		void* /* MeasurementRegisterRecord, MeasurementPrepareRecord, MeasurementRecordRecord, or MeasurementExportRecord */ pb_,
		intptr_t* data__,
		int16_t* result__
	);

	/**
		\brief	Main entry point for picker plugins (required by Photoshop). This is called whenever a plugin should
				perform some action or respond to some message.
		\author	GW
		\date	12/2017
	*/
	extern "C" __HDI_CORE_EXPORT void PickerPluginMain(
		const int16_t selector_,
		void* /* PickerRecordPtr */ pb_,
		intptr_t* data__,
		int16_t* result__
	);

	/**
		\brief	Main entry point for selection plugins (required by Photoshop). This is called whenever a plugin should
				perform some action or respond to some message.
		\author	GW
		\date	12/2017
	*/
	extern "C" __HDI_CORE_EXPORT void SelectionPluginMain(
		const int16_t selector_,
		void* /* SelectionRecordPtr */ pb_,
		intptr_t* data__,
		int16_t* result__
	);
#endif
// HDI_CORE_PSP_MODE

#endif
// __HDI_CORE_MAIN__
