/**
	\file
	\brief		Header file for manipulating plugin menubar menu items
	\copyright	Hot Door, Inc. 2010-2025
*/

#ifndef __HDI_CORE_MENU_ITEM__
#define __HDI_CORE_MENU_ITEM__

#include "hdicoreTypes.h"

namespace hdi
{
	namespace core
	{
		class Callback;
		
		#if defined(HDI_CORE_AIP_MODE)
			class MenuGroup;
		#endif
	
		/**
			\brief	Wraps around an application menu item to allow for selection callbacks, en/disabling, setting text, etc.
		*/
		class MenuItem
		{
			public:
				/**
					\brief	Constructs an empty MenuItem object
					\author	GW
					\date	08/2013
					
					\note	To test if a MenuItem object is empty, call isEmpty() on it
					\note	Empty MenuItem objects do not relate to any actual application menu item; they are designed
							to be "receivers" of some other MenuItem object via the overloaded assignment operator.
							Empty MenuItem objects are useless until such time (though it is safe to call any of their
							methods).
				*/
				MenuItem();
			
				/**
					\brief	Constructs a new MenuItem object from an existing MenuItem object (copy constructor)
					\author	GW
					\date	08/2013

					\param	m_	Existing MenuItem object
				*/
				MenuItem(const MenuItem& m_);
					
				/**
					\brief	Constructs a new MenuItem object
					\author	GW
					\date	08/2013

					\param	group_		Built-in app menu group to contain the new MenuItem
					\param	text_		String that will show in the menu item, as UTF-8
					\param	callback_	Callback to perform when the menu item is selected
					
					\note	For Photoshop, the menu item will always be appended to the group.
				*/
				MenuItem(const BuiltinMenuGroup group_, const std::string& text_, const Callback& callback_);

				/**
					\brief		Constructs a new MenuItem object as a divider
					\author		GW
					\date		11/2013
					
					\param		group_	Built-in app menu group to contain the new MenuItem
					\returns	A new MenuItem object that's a divider
					
					\note	For Photoshop, the menu item will always be appended to the group.
				*/
				static MenuItem Divider(const BuiltinMenuGroup group_);

				#if defined(HDI_CORE_AIP_MODE)
					/**
						\brief	Constructs a new MenuItem object
						\author	GW
						\date	08/2013

						\param	group_		User-created menu group to contain the new MenuItem
						\param	text_		String that will show in the menu item, as UTF-8
						\param	callback_	Callback to perform when the menu item is selected
					*/
					MenuItem(const MenuGroup& group_, const std::string& text_, const Callback& callback_);

					/**
						\brief		Constructs a new MenuItem object as a divider
						\author		GW
						\date		11/2013
						
						\param		group_	User-created menu group to contain the new MenuItem
						\returns	A new MenuItem object that's a divider
					*/
					static MenuItem Divider(const MenuGroup& group_);
				#endif
				// HDI_CORE_AIP_MODE

				/**
					\brief	Destructs a MenuItem object
					\author	GW
					\date	08/2013
				*/
				virtual ~MenuItem();

				/**
					\brief	Destroys the UI menu item, converting the target object to an empty MenuItem object (see the
							default constructor for more info on empty MenuItem objects)
					\author	GW
					\date	10/2013

					\note	This method must be called for any existing menu items by the time the plugin is shutting
							down. If you no longer retain any of your own references to a menu item then the lib will
							call this method for you as it is shutting down.
				*/
				virtual void destroy();
			
				/**
					\brief		Assigns one MenuItem object to another
					\author		GW
					\date		08/2013

					\param		rhs_	Existing MenuItem object to copy values from
					\returns	The target MenuItem object, but with its values updated to match that of the rhs_
								argument
				*/
				virtual MenuItem& operator=(const MenuItem& rhs_);
				
				/**
					\brief		Tests whether a given MenuItem object is the same as another
					\author		GW
					\date		08/2013

					\param		rhs_	MenuItem to compare against (righthand side of equality operator)
					\returns	true for the target and rhs_ being the same menu item, false otherwise
				*/
				virtual bool operator==(const MenuItem& rhs_) const;
				
				/**
					\brief		Tests whether a given MenuItem object is not the same as another
					\author		GW
					\date		08/2013

					\param		rhs_	MenuItem to compare against (righthand side of inequality operator)
					\returns	true for the target and rhs_ being different menu items, false otherwise
				*/
				virtual bool operator!=(const MenuItem& rhs_) const;

				#if defined(HDI_CORE_AIP_MODE)
					/**
						\brief		Gets the menu item handle around which the target object is wrapped
						\author		GW
						\date		11/2013
						
						\returns	AIMenuItemHandle for this object

						\note		Generally, you shouldn't need to call this method. Only use it if you know what you're
									doing. If a specific piece of functionality provided by Illustrator is not handled by
									this class (or related classes), then it should probably be added to the hdi_core library.
					*/
					virtual AIMenuItemHandle aiMenuItemHandle() const;
				#elif defined(HDI_CORE_PSP_MODE)
					/**
						\brief		Gets the platform menu item around which the target object is wrapped
						\author		GW
						\date		12/2017
						
						\returns	Platform menu item for this object

						\note		Generally, you shouldn't need to call this method. Only use it if you know what you're
									doing. If a specific piece of functionality provided by Photoshop is not handled by
									this class (or related classes), then it should probably be added to the library.
					*/
					virtual PlatformMenuItemPtr platformMenuItem() const;
				#endif

				/**
					\brief		Gets whether the target MenuItem object is empty (constructed with the default ctor)
					\author		GW
					\date		08/2013
					
					\returns	true if the target MenuItem object is empty, false otherwise
				*/
				virtual bool isEmpty() const;
				
				/**
					\brief		Gets whether the target MenuItem is a divider
					\author		GW
					\date		09/2014
					
					\returns	true if the menu item is a divider/separate, false otherwise
				*/
				virtual bool isDivider() const;
				
				#if defined(HDI_CORE_AIP_MODE)
					/**
						\brief		Gets the parent of the target menu item
						\author		GW
						\date		09/2014
						
						\returns	The parent menu group, or NULL for none/error
					*/
					virtual std::unique_ptr<MenuGroup> parent() const;
				#endif
				// HDI_CORE_AIP_MODE

				/**
					\brief		Gets whether the menu item is enabled
					\author		GW
					\date		08/2013

					\returns	true if the item is enabled, false otherwise
				*/
				virtual bool enabled() const;

				/**
					\brief	Sets the menu item as enabled or disabled
					\author	GW
					\date	08/2013

					\param	enabled_	true to enable the item, false to disable
				*/
				virtual void setEnabled(const bool enabled_);

				/**
					\brief		Gets whether the menu item is checked
					\author		GW
					\date		08/2013

					\returns	true if the item is checked, false otherwise
				*/
				virtual bool checked() const;

				/**
					\brief	Sets the menu item as checked or unchecked
					\author	GW
					\date	08/2013

					\param	 checked_	true to check the menu item, false to uncheck
				*/
				virtual void setChecked(const bool checked_);

				/**
					\brief		Gets the text of the menu item itself
					\author		GW
					\date		08/2013

					\returns	Current text value of the menu item, as UTF-8
				*/
				virtual std::string text() const;

				/**
					\brief	Sets the text of the menu item itself
					\author	GW
					\date	08/2013

					\param	text_	New text value for the menu item, as UTF-8
				*/
				virtual void setText(const std::string& text_);

				/**
					\brief		Gets the registered callback for the menu item
					\author		GW
					\date		08/2013

					\returns	Currently registered menu item callback
				*/
				virtual Callback* const callback() const;
				
				/**
					\brief	Sets the callback for the menu item
					\author	GW
					\date	08/2013

					\param	cb_		New callback to be performed when the menu item is selected
				*/
				virtual void setCallback(const Callback& cb_);


			private:
				#if defined(HDI_CORE_AIP_MODE)
					friend MenuItem __accessCtor(AIMenuItemHandle);

					/**
						\brief	Internal use only
						\author	GW
						\date	08/2014
					*/
					MenuItem(AIMenuItemHandle);
				#endif
				// HDI_CORE_AIP_MODE

				/**
					\brief	Private implementation data
				*/
				void* __data;
		};
		
		typedef std::unique_ptr<MenuItem> MenuItemUP;
		typedef std::shared_ptr<MenuItem> MenuItemSP;
		typedef std::weak_ptr<MenuItem> MenuItemWP;
		
		#if defined(HDI_CORE_AIP_MODE)
			extern MenuItem __accessCtor(AIMenuItemHandle);
		#endif
	}
}

#endif
// __HDI_CORE_MENU_ITEM__
