/**
	\file
	\brief		Header file for plugin message data wrapper classes
	\copyright	Hot Door, Inc. 2010-2025
*/

#ifndef __HDI_CORE_MESSAGE__
#define __HDI_CORE_MESSAGE__

#include <vector>

#include "hdicoreAngle.h"
#include "hdicoreArtboardPoint.h"
#include "hdicoreArtColor.h"
#include "hdicoreFileFormat.h"
#include "hdicorePoint.h"
#include "hdicoreRect.h"
#include "hdicoreTypes.h"

namespace hdi
{
	namespace core
	{
		#if defined(HDI_CORE_AIP_MODE)
			class Annotator;
			class Art;
			class Artboard;
			class DocumentView;
			class FileFormat;
			class MenuItem;
			class Notification;
			class Notifier;
			class Timer;
			class Tool;
		#endif
		// HDI_CORE_AIP_MODE
		
		#if defined(HDI_CORE_PSP_MODE)
			class DocumentInfo;
			class PixelMap;
		#endif
		// HDI_CORE_PSP_MODE
	
		/**
			\brief	Base class for all *Message subclasses
		*/
		class Message
		{
			public:
				/**
					\brief	Destructs a Message object
					\author	GW
					\date	08/2013
				*/
				virtual ~Message();
			
				/**
					\brief		Gets the type of message data the target object represents
					\author		GW
					\date		08/2013

					\returns	Message type (useful for determining how to cast a Message object to its subclass)
				*/
				virtual MessageType type() const;
				
				/**
					\brief		Gets the raw Illustrator message data struct
					\author		GW
					\date		08/2022
					
					\returns	A pointer to the raw Illustrator message data that relates to the target object (and its
								subclass)
					
					\warning	Use this method at your own risk; generally, you shouldn't need to call it. Only use
								it if you know what you're doing. If a specific piece of functionality provided by
								the app is not handled by this class (or related classes), then it should probably
								be added to the hdi_core library.
					\warning	It is your responsibility to cast the void* to the proper data type based on the type of
								message to which the target relates. For example, the AnnotatorMessage class very likely
								wraps raw data of type AIAnnotatorMessage* (which you can determine by looking at which
								specific messages result in an AnnotatorMessage object and checking the Illustrator SDK
								documentation to see the related AIAnnotatorMessage struct).
				*/
				virtual void* rawData() const;
			
			
			protected:
				/**
					\brief	Stores private implementation data
				*/
				void* _data;

				/**
					\brief	Internal use only
					\author	GW
					\date	08/2013
				*/
				Message();
				
				/**
					\brief	Unused
					\author	GW
					\date	02/2015
				*/
				Message(const Message&);
				
				/**
					\brief	Unused
					\author	GW
					\date	02/2015
				*/
				Message& operator=(const Message&);
		};
		
		/**
			\brief	Received for ActionGoMessageType
		*/
		class ActionMessage : public Message
		{
			public:
				/**
					\brief	Destructs a ActionMessage object
					\author	GW
					\date	08/2013
				*/
				virtual ~ActionMessage();


			private:
				// Only Dispatcher can construct an ActionMessage object
				friend class Dispatcher;

				/**
					\brief	Internal use only
					\author	GW
					\date	08/2013
				*/
				ActionMessage(const MessageType, void*);
				
				/**
					\brief	Unused
					\author	GW
					\date	02/2015
				*/
				ActionMessage(const ActionMessage&);
				
				/**
					\brief	Unused
					\author	GW
					\date	02/2015
				*/
				ActionMessage& operator=(const ActionMessage&);
		};
		
		#if defined(HDI_CORE_PSP_MODE)
			/**
				\brief	Received for AcquireStartMessageType, AcquireContinueMessageType, AcquireFinishMessageType, 
						AcquirePrepareMessageType, and AcquireFinalizeMessageType
			*/
			class AcquireMessage : public Message
			{
				public:
					/**
						\brief	Destructs an AcquireMessage object
						\author	GW
						\date	12/2017
					*/
					virtual ~AcquireMessage();
					

				private:
					// Only Dispatcher can construct an AcquireMessage object
					friend class Dispatcher;

					/**
						\brief	Internal use only
						\author	GW
						\date	12/2017
					*/
					AcquireMessage(const MessageType, void*);
					
					/**
						\brief	Unused
						\author	GW
						\date	12/2017
					*/
					AcquireMessage(const AcquireMessage&);
					
					/**
						\brief	Unused
						\author	GW
						\date	12/2017
					*/
					AcquireMessage& operator=(const AcquireMessage&);
			};
		#endif
		// HDI_CORE_PSP_MODE
		
		#if defined(HDI_CORE_AIP_MODE)
			/**
				\brief	Received for AdaptersStartupMessageType, AdaptersShutdownMessageType, AdaptersDisposeInfoMessageType,
						AdaptersFlushMessageType, AdaptersRegisterMessageType, AdaptersLoadPluginMessageType,
						AdaptersReleasePluginMessageType, and AdaptersSendMessageMessageType
			*/
			class AdapterMessage : public Message
			{
				public:
					/**
						\brief	Destructs a AdapterMessage object
						\author	GW
						\date	08/2013
					*/
					virtual ~AdapterMessage();


				private:
					// Only Dispatcher can construct an AdapterMessage object
					friend class Dispatcher;

					/**
						\brief	Internal use only
						\author	GW
						\date	08/2013
					*/
					AdapterMessage(const MessageType, void*);
					
					/**
						\brief	Unused
						\author	GW
						\date	02/2015
					*/
					AdapterMessage(const AdapterMessage&);
					
					/**
						\brief	Unused
						\author	GW
						\date	02/2015
					*/
					AdapterMessage& operator=(const AdapterMessage&);
			};
		
			/**
				\brief	Received for AnnotationDrawMessageType and AnnotationInvalidateMessageType
			*/
			class AnnotatorMessage : public Message
			{
				public:
					typedef std::vector<Rect> RectVector;
				
					typedef AIAnnotatorDrawer* ContextPtr;
			
					/**
						\brief	Destructs a AnnotatorMessage object
						\author	GW
						\date	08/2013
					*/
					virtual ~AnnotatorMessage();
					
					/**
						\brief		Gets the Annotator object to which the message applies
						\author		GW
						\date		08/2013
						
						\returns	The annotator that should draw (or invalidate) as a result of this message
					*/
					Annotator* const annotator() const;
					
					/**
						\brief		Gets the annotation context for the target message
						\author		GW
						\date		03/2014
						
						\returns	A constant pointer to the annotation context
						
						\note		The caller must not dispose/release the annotation context itself.
					*/
					const ContextPtr context() const;
					
					/**
						\brief		Gets the DocumentView to which the message applies
						\author		GW
						\date		08/2013
						
						\returns	The document view, in which annotations should be drawn (or invalidated)
					*/
					DocumentView* const docView() const;
					
					/**
						\brief		Gets whether the message was sent due to the view scrolling during a mouse drag event
						\author		GW
						\date		08/2013
						
						\returns	true if auto-scrolling due to dragging occurred, false otherwise
					*/
					bool triggeredByDragScroll() const;
					
					/**
						\brief		Gets whether an outline should be drawn around the annotation area
						\author		GW
						\date		08/2013
						
						\returns	true when an outline should be drawn, false otherwise
					*/
					bool showEdges() const;

					/**
						\brief		Gets the regions of the view that have been invalidated and need to be redrawn
						\author		GW
						\date		08/2013
						
						\returns	All view regions that have been invalidated
						
						\note		These regions also double as clip rects when only portions of a drawn area have been
									invalidated.
					*/
					RectVector* const invalidationRects() const;
			

				private:
					// Only Dispatcher can construct an AnnotatorMessage object
					friend class Dispatcher;

					/**
						\brief	Internal use only
						\author	GW
						\date	08/2013
					*/
					AnnotatorMessage(const Annotator&, const MessageType, void*);
					
					/**
						\brief	Unused
						\author	GW
						\date	02/2015
					*/
					AnnotatorMessage(const AnnotatorMessage&);
					
					/**
						\brief	Unused
						\author	GW
						\date	02/2015
					*/
					AnnotatorMessage& operator=(const AnnotatorMessage&);
			};
			
			/**
				\brief	Received for ClipboardGoMessageType, ClipboardCanCopyMessageType, ClipboardCloneMessageType and
						ClipboardDisposeMessageType
			*/
			class ClipboardMessage : public Message
			{
				public:
					/**
						\brief	Destructs a ClipboardMessage object
						\author	GW
						\date	08/2013
					*/
					virtual ~ClipboardMessage();


				private:
					// Only Dispatcher can construct a ClipboardMessage object
					friend class Dispatcher;

					/**
						\brief	Internal use only
						\author	GW
						\date	08/2013
					*/
					ClipboardMessage(const MessageType, void*);
					
					/**
						\brief	Unused
						\author	GW
						\date	02/2015
					*/
					ClipboardMessage(const ClipboardMessage&);
					
					/**
						\brief	Unused
						\author	GW
						\date	02/2015
					*/
					ClipboardMessage& operator=(const ClipboardMessage&);
			};
		
			class CommandMessage : public Message
			{
				public:
					/**
						\brief	Destructs a CommandMessage object
						\author	GW
						\date	08/2013
					*/
					virtual ~CommandMessage();


				private:
					// Only Dispatcher can construct a CommandMessage object
					friend class Dispatcher;

					/**
						\brief	Internal use only
						\author	GW
						\date	08/2013
					*/
					CommandMessage(const MessageType, void*);
					
					/**
						\brief	Unused
						\author	GW
						\date	02/2015
					*/
					CommandMessage(const CommandMessage&);
					
					/**
						\brief	Unused
						\author	GW
						\date	02/2015
					*/
					CommandMessage& operator=(const CommandMessage&);
			};
		#endif
		// HDI_CORE_AIP_MODE
		
		/**
			\brief	Received for any custom messages (i.e. custom caller/selector pairs)
		*/
		class CustomMessage : public Message
		{
			public:
				/**
					\brief	Destructs a CustomMessage object
					\author	GW
					\date	08/2013
				*/
				virtual ~CustomMessage();
				
				/**
					\brief		Gets the caller string that was received with the message
					\author		GW
					\date		04/2014
					
					\returns	The caller string as received by the dispatcher
					
					\note		The return value might be "" for Photoshop, in which case the callerNumeric() method
								should be used.
				*/
				std::string caller() const;
				
				/**
					\brief		Gets the selector string that was received with the message
					\author		GW
					\date		04/2014
					
					\returns	The selector string as received by the dispatcher
					
					\note		The return value might be "" for Photoshop, in which case the selectorNumeric() method
								should be used.
				*/
				std::string selector() const;
				
				#if defined(HDI_CORE_PSP_MODE)
					/**
						\brief		Gets the caller type that was received with the message for non-action Photoshop plugins
						\author		GW
						\date		12/2017
						
						\returns	The caller "category" for the selector, as received by the dispatcher
						
						\note		The return value might be UnknownCallerType for Photoshop, in which case the caller()
									method should be used.
					*/
					CallerType callerNumeric() const;
					
					/**
						\brief		Gets the selector numeric value that was received with the message  for non-action
									Photoshop plugins
						\author		GW
						\date		12/2017
						
						\returns	The selector value as received by the dispatcher
						
						\note		The return value might be -1 for Photoshop, in which case the selector() method
									should be used.
					*/
					int16_t selectorNumeric() const;
				#endif


			private:
				// Only Dispatcher can construct a CustomMessage object
				friend class Dispatcher;

				/**
					\brief	Internal use only
					\author	GW
					\date	08/2013
				*/
				CustomMessage(const std::string&, const std::string&, const MessageType, void*);

				#if defined(HDI_CORE_PSP_MODE)
					/**
						\brief	Internal use only
						\author	GW
						\date	08/2013
					*/
					CustomMessage(const CallerType, const int16_t, const MessageType, void*);
				#endif
				
				/**
					\brief	Unused
					\author	GW
					\date	02/2015
				*/
				CustomMessage(const CustomMessage&);
				
				/**
					\brief	Unused
					\author	GW
					\date	02/2015
				*/
				CustomMessage& operator=(const CustomMessage&);
		};
		
		#if defined(HDI_CORE_PSP_MODE)
			/**
				\brief	Received for ExportStartMessageType, ExportContinueMessageType, ExportFinishMessageType,
						and ExportPrepareMessageType
			*/
			class ExportMessage : public Message
			{
				public:
					/**
						\brief	Destructs an ExportMessage object
						\author	GW
						\date	12/2017
					*/
					virtual ~ExportMessage();


				private:
					// Only Dispatcher can construct an ExportMessage object
					friend class Dispatcher;

					/**
						\brief	Internal use only
						\author	GW
						\date	12/2017
					*/
					ExportMessage(const MessageType, void*);
					
					/**
						\brief	Unused
						\author	GW
						\date	12/2017
					*/
					ExportMessage(const ExportMessage&);
					
					/**
						\brief	Unused
						\author	GW
						\date	12/2017
					*/
					ExportMessage& operator=(const ExportMessage&);
			};
		#endif
		// HDI_CORE_PSP_MODE
		
		#if defined(HDI_CORE_AIP_MODE)
			/**
				\brief	Received for FileFormatGetParamsMessageType, FileFormatSetParamsMessageType, FileFormatGoMessageType,
						FileFormatCheckMessageType, and FileFormatUpdateMessageType
			*/
			class FileFormatMessage : public Message
			{
				public:
					typedef std::vector< std::shared_ptr<Artboard> > ArtboardVector;
				
					/**
						\brief	Destructs a FileFormatMessage object
						\author	GW
						\date	08/2013
					*/
					virtual ~FileFormatMessage();
					
					/**
						\brief		Gets the FileFormat object to which the message applies
						\author		GW
						\date		12/2014
						
						\returns	The intended file format
					*/
					FileFormat* const fileFormat() const;
					
					/**
						\brief		Gets the requested operation
						\author		GW
						\date		12/2014
						
						\returns	A bitwise OR of the enum values for the file format operation
					*/
					FileFormat::Options operation() const;
					
					/**
						\brief		Gets the requested extended operation
						\author		GW
						\date		12/2014
						
						\returns	A bitwise OR of the enum values for the file format extended operation
					*/
					FileFormat::ExtendedOptions extendedOperation() const;
					
					/**
						\brief		Gets the path of the file to read, write, check, etc.
						\author		GW
						\date		12/2014
						
						\returns	File path for the format message, as UTF-8
					*/
					std::string filePath() const;
					
					/**
						\brief	Sets the path of the file to read, write, check, etc.
						\author	GW
						\date	12/2014
						
						\param	path_	New file path for the format message, as UTF-8
					*/
					void setFilePath(const std::string& path_);
					
					/**
						\brief		Gets the additional parameter data, if the ContainsPartialParametersOption is set in
									the return value of the operation() method
						\author		GW
						\date		12/2014
						
						\returns	Additional file format data
					*/
					void* actionParam() const;
					
					/**
						\brief		If the extended operation is SaveMultiArtboardsExtendedOption, this gets the artboards
									affected by the file format message
						\author		GW
						\date		12/2014
						
						\returns	A vector of artboards for the message
					*/
					ArtboardVector artboards() const;


				private:
					// Only Dispatcher can construct a FileFormatMessage object
					friend class Dispatcher;

					/**
						\brief	Internal use only
						\author	GW
						\date	08/2013
					*/
					FileFormatMessage(const MessageType, void*);
					
					/**
						\brief	Unused
						\author	GW
						\date	02/2015
					*/
					FileFormatMessage(const FileFormatMessage&);
					
					/**
						\brief	Unused
						\author	GW
						\date	02/2015
					*/
					FileFormatMessage& operator=(const FileFormatMessage&);
			};
		#endif
		// HDI_CORE_AIP_MODE
		
		#if defined(HDI_CORE_PSP_MODE)
			/**
				\brief	Received for FormatReadPrepareMessageType, FormatReadStartMessageType, FormatReadContinueMessageType,
					FormatReadFinishMessageType, FormatOptionsPrepareMessageType, FormatOptionsStartMessageType,
					FormatOptionsContinueMessageType, FormatOptionsFinishMessageType, FormatEstimatePrepareMessageType,
					FormatEstimateStartMessageType, FormatEstimateContinueMessageType, FormatEstimateFinishMessageType,
					FormatWritePrepareMessageType, FormatWriteStartMessageType, FormatWriteContinueMessageType,
					FormatWriteFinishMessageType, FormatFilterFileMessageType, FormatGetFilePropertyValueMessageType,
					FormatLosslessRotateMessageType, FormatBulkSettingsMessageType, FormatXMPReadMessageType,
					FormatXMPWriteMessageType, FormatXMPAppendMessageType, FormatBulkOpenMessageType, FormatLoadMessageType,
					FormatUnloadMessageType, FormatPreferencesMessageType, FormatSettingsCopyMessageType,
					FormatSettingsPasteMessageType, FormatSettingsClearMessageType, FormatSettingsDefaultMessageType,
					FormatSettingsPreviousMessageType, FormatSettingsPresetListMessageType, FormatSettingsChecksumMessageType,
					FormatReadLayerStartMessageType, FormatReadLayerContinueMessageType, FormatReadLayerFinishMessageType,
					FormatWriteLayerStartMessageType, FormatWriteLayerContinueMessageType, FormatWriteLayerFinishMessageType,
					and FormatLaunchExternalEditorMessageType
			*/
			class FileFormatMessage : public Message
			{
				public:
					/**
						\brief	Destructs a FileFormatMessage object
						\author	GW
						\date	12/2017
					*/
					virtual ~FileFormatMessage();
				

				private:
					// Only Dispatcher can construct a FileFormatMessage object
					friend class Dispatcher;

					/**
						\brief	Internal use only
						\author	GW
						\date	12/2017
					*/
					FileFormatMessage(const MessageType, void*);
					
					/**
						\brief	Unused
						\author	GW
						\date	12/2017
					*/
					FileFormatMessage(const FileFormatMessage&);
					
					/**
						\brief	Unused
						\author	GW
						\date	12/2017
					*/
					FileFormatMessage& operator=(const FileFormatMessage&);
			};
		#endif
		// HDI_CORE_AIP_MODE
		
		/**
			\brief	Received for FilterGetParamsMessageType and FilterGoMessageType for Illustrator, and 
					FilterParametersMessageType, FilterPrepareMessageType, FilterStartMessageType, 
					FilterContinueMessageType, and FilterFinishMessageType for Photoshop
		*/
		class FilterMessage : public Message
		{
			public:
				/**
					\brief	Destructs a FilterMessage object
					\author	GW
					\date	08/2013
				*/
				virtual ~FilterMessage();
				
				#if defined(HDI_CORE_PSP_MODE)
					/**
						\brief		Provides a mechanism to allow the user to abort the operation of the plug-in
						\author		GW
						\date		01/2018
						
						\returns	true if the user has requested to abort the filter operation, false otherwise
						
						\note		If needed, you should call this method several times a second during long operations.
									If the user has requested to abort the operation, the plug-in should return a positive
									error code. As a side effect, this changes the cursor to a watch and moves the watch
									hands periodically.
					*/
					bool shouldAbort() const;
					
					/**
						\brief	Provides a mechanism for updating progress indicators
						\author	GW
						\date	01/2018
						
						\note	You can call this method periodically to update a progress indicator. This method should
								only be called in the actual main operation of the plug-in, not while long operations are
								executing during the preliminary user interface. Photoshop automatically suppresses
								display of the progress bar during short operations.
					*/
					void setProgress(const int32_t completed_, const int32_t total_);
			
					/**
						\brief		Gets the number of planes in the image
						\author		GW
						\date		01/2018
						
						\returns	The total number of active planes in the image, including alpha channels
						
						\note		The image mode should be determined by calling the imageMode() method.
					*/
					int16_t planes() const;
			
					/**
						\brief		Gets the maximum number of bytes of information you can expect to be able to access
									at once (input area size + output area size + mask area size + bufferSpace)
						\author		GW
						\date		01/2018
						
						\returns	The max accessible bytes
					*/
					int32_t maxSpace() const;
			
					/**
						\brief		Gets how much buffer space you need (previously set via the setBufferSpace() method)
						\author		GW
						\date		01/2018
						
						\returns	The needed number of buffer space bytes
					*/
					int32_t bufferSpace() const;
			
					/**
						\brief	Sets how much buffer space you need (see notes)
						\author	GW
						\date	01/2018
						
						\param	bs_		The needed number of buffer space bytes
						
						\note	If you are planning on allocating any large internal buffers or tables, you should set
								this field during the FilterPrepareMessageType message to the number of bytes you plan
								to allocate. Photoshop then tries to free up the requested amount of space before sending
								the FilterStartMessageType message. Relocatable blocks should be used if possible.
					*/
					void setBufferSpace(const int32_t bs_);
			
					/**
						\brief		Gets the first input plane to process next (previously set via the setInLoPlane()
									method)
						\author		GW
						\date		01/2018
						
						\returns	Desired input plane
					*/
					int16_t inLoPlane() const;
			
					/**
						\brief	Sets the first input plane to process next (see notes)
						\author	GW
						\date	01/2018
						
						\param	ilp_	Desired input plane
						
						\note	This should be set in the FilterStartMessageType and FilterContinueMessageType message
								handlers.
					*/
					void setInLoPlane(const int16_t ilp_);
			
					/**
						\brief		Gets the last input plane to process next (previously set via the setInHiPlane()
									method)
						\author		GW
						\date		01/2018
						
						\returns	Desired input plane
					*/
					int16_t inHiPlane() const;
			
					/**
						\brief	Sets the last input plane to process next (see notes)
						\author	GW
						\date	01/2018
						
						\param	ihp_	Desired input plane
						
						\note	This should be set in the FilterStartMessageType and FilterContinueMessageType message
								handlers.
					*/
					void setInHiPlane(const int16_t ihp_);
			
					/**
						\brief		Gets the first output plane to process next (previously set via the setOutLoPlane()
									method)
						\author		GW
						\date		01/2018
						
						\returns	Desired output plane
					*/
					int16_t outLoPlane() const;
			
					/**
						\brief	Sets the first output plane to process next (see notes)
						\author	GW
						\date	01/2018
						
						\param	olp_	Desired output plane
						
						\note	This should be set in the FilterStartMessageType and FilterContinueMessageType message
								handlers.
					*/
					void setOutLoPlane(const int16_t olp_);
			
					/**
						\brief		Gets the last output plane to process next (previously set via the setOutHiPlane()
									method)
						\author		GW
						\date		01/2018
						
						\returns	Desired output plane
					*/
					int16_t outHiPlane() const;
			
					/**
						\brief	Sets the last output plane to process next (see notes)
						\author	GW
						\date	01/2018
						
						\param	ohp_	Desired output plane
						
						\note	This should be set in the FilterStartMessageType and FilterContinueMessageType message
								handlers.
					*/
					void setOutHiPlane(const int16_t ohp_);
			
					/**
						\brief		Gets pointer to the requested input image data. If more than one plane has been
									requested (via inLoPlane() and inHiPlane() methods), the data is interleaved.
						\author		GW
						\date		01/2018
						
						\returns	Pointer to the start of input image data
					*/
					void* const inData() const;
			
					/**
						\brief		The offset between rows of the input image data. The end of each row may or may not
									include pad bytes.
						\author		GW
						\date		01/2018
						
						\returns	The row offset, in bytes
					*/
					int32_t inRowBytes() const;
			
					/**
						\brief		Gets a pointer to the requested output image data. If more than one plane has been
									requested (via outLoPlane() and outHiPlane() methods), the data is interleaved.
						\author		GW
						\date		01/2018
						
						\returns	Pointer to the start of output image data
					*/
					void* const outData() const;
			
					/**
						\brief		The offset between rows of the output image data. The end of each row may or may not
									include pad bytes.
						\author		GW
						\date		01/2018
						
						\returns	The row offset, in bytes
					*/
					int32_t outRowBytes() const;
			
					/**
						\brief		Allows you to ask Photoshop to update (i.e. "advance the state of") the various data
									structures in the message (see notes)
						\author		GW
						\date		01/2018
						
						\returns	true for success, false otherwise
						
						\note		Use this method where you expect your plug-in to be called repeatedly; for example,
									a scanner module that scans and delivers images in chunks. When working with large
									images (larger than available RAM), you should process the image in pieces. Without 
									this method, your plug-in is called from (and returns to) Photoshop for each chunk
									of data; each repeated call must go through your plug-in's entry point. With this
									method, your plug-in can complete its general operation within a single call from
									the plug-in host.
						\note		Proxies really put this method to work, because if your plug-in allows the user to
									drag around an image, you'll be constantly updating and asking for more image data.
									To keep the lag time down, and the update/wait cursor from appearing often, keep
									these items in mind: 1) this method buffers as much of the image as it can, so make
									the first call for the setInRect() method as large as possible. In subsequent calls,
									as long as the request falls within the set rect, the image data comes right out of
									the buffer, and 2) as soon as you set the rectangle to {0,0,0,0}, or you call for one
									pixel outside of the buffer area, this method flushes the buffer and loads new image
									data from the VM system.
					*/
					bool advanceState();
			
					/**
						\brief		Draws the provided pixel data, typically provided by Photoshop itself originally (but
									might be further modified by your plugin), into the provided drawing context
						\author		GW
						\date		02/2018
						
						\param		dc_			Platform drawing context in which the drawing will occur
						\param		srcPM_		Describes a block of pixels from Photoshop to display
						\param		srcArea_	Indicates the rectangle of the source pixel map to display
						\param		dstTL_		Provides the top-left destination pixel in the context
						\param		dstArea_	Provides the destination rectangle for drawing the pixels (often matches
												srcArea_ - only Mac necessitates this argument; Windows can acquire this
												from the given drawing context)
						\returns	true if Photoshop reported successful drawing, false otherwise
						
						\note		If you have a CustomWidget object you would like to draw into, utilize that class'
									drawPixelData() method instead.
					*/
					bool drawPixelData(
						const PlatformDrawingContextPtr dc_,
						const PixelMap& srcPM_,
						const Rect& srcArea_,
						const Point& dstTL_,
						const Rect& dstArea_
					);
			
					/**
						\brief		Gets if the selection is floating
						\author		GW
						\date		01/2018
						
						\returns	true if and only if the selection is floating
					*/
					bool isFloating() const;
			
					/**
						\brief		Gets if the selection has a mask
						\author		GW
						\date		01/2018
						
						\returns	true if and only if a non-rectangular area has been selected
					*/
					bool hasMask() const;
			
					/**
						\brief		Gets whether auto-masking is on or off
						\author		GW
						\date		01/2018
						
						\returns	The auto-mask state
					*/
					bool autoMask() const;
			
					/**
						\brief	Sets whether auto-masking is on or off
						\author	GW
						\date	01/2018
						
						\param	masked_		
						
						\note	By default, Photoshop automatically masks any changes to the area actually selected. If
								the isFloating() method returns false, and hasMask() returns true, then you can turn off
								this feature by passing false. You can then perform your own masking. If you have set the
								bit writesOutsideSelection in your PiPL, this will always be false and you must supply
								your own mask, if needed.
					*/
					void setAutoMask(const bool masked_);
			
					/**
						\brief		Gets a pointer to the requested mask data
						\author		GW
						\date		01/2018
						
						\returns	Pointer to the start of mask data
						
						\note		The data is in the form of an array of bytes, one byte per pixel of the selected
									area. The bytes are in range [0,255], where 0=no mask (selected) and 255=masked
									(not selected). Use the maskRowBytes() method to iterate over the scan lines of the
									mask.
					*/
					void* const maskData() const;
			
					/**
						\brief		Gets the offset between rows of the mask data
						\author		GW
						\date		01/2018
						
						\returns	Mask data row offset bytes
					*/
					int32_t maskRowBytes() const;
			
					/**
						\brief		Gets the current background color, in the color space native to the image
						\author		GW
						\date		01/2018

						\returns	The background color
					*/
					ArtColor backColor() const;
			
					/**
						\brief		Gets the current foreground color, in the color space native to the image
						\author		GW
						\date		01/2018

						\returns	The foreground color
					*/
					ArtColor foreColor() const;
			
					/**
						\brief		Gets the mode of the image being filtered
						\author		GW
						\date		01/2018

						\returns	The image mode (i.e. color space)
					*/
					ColorSpace imageMode() const;
			
					/**
						\brief		Gets the horizontal resolution of the image in terms of pixels per inch
						\author		GW
						\date		01/2018

						\returns	The horizontal resolution
					*/
					double hRes() const;
			
					/**
						\brief		Gets the vertical resolution of the image in terms of pixels per inch
						\author		GW
						\date		01/2018

						\returns	The vertical resolution
					*/
					double vRes() const;
			
					/**
						\brief		Gets the monitor setup information for Photoshop
						\author		GW
						\date		01/2018

						\returns	A pointer to the monitor specs object
					*/
					MonitorSpecs* const monitor() const;
			
					/**
						\brief		Gets whether Photoshop supports the plug-in requesting nonexistent planes
						\author		GW
						\date		01/2018

						\returns	Whether dummy channels are allowable
					*/
					bool supportsDummyChannels() const;
			
					/**
						\brief		Indicates whether the host support data layouts other than rows of columns of planes
						\author		GW
						\date		01/2018

						\returns	Whether alternate data layout are supported
						
						\note		If this returns true then Photoshop will respect the setWantLayout() method.
					*/
					bool supportsAlternateLayouts() const;
			
					/**
						\brief		Gets the desired layout for the data
						\author		GW
						\date		01/2018

						\returns	The desired data layout, as previously set by setWantLayout()
					*/
					DataLayout wantLayout() const;
			
					/**
						\brief	Sets the desired layout for the data
						\author	GW
						\date	01/2018
						
						\param	wl_		The desired data layout
						
						\note	This results of this method will be ignored by Photoshop if supportsAlternateLayouts()
								returns false.
					*/
					void setWantLayout(const DataLayout wl_);
			
					/**
						\brief		Gets the type of data being filtered
						\author		GW
						\date		01/2018

						\returns	The data filter case enum value
						
						\note		If this method returns UnknownFilterCase, then it should be ignored and you should
									instead call the hasMask() and isFloating() methods.
					*/
					FilterCase filterCase() const;
			
					/**
						\brief		Gets the value to store into any dummy planes, as previously set by setDummyPlaneValue()
						\author		GW
						\date		01/2018

						\returns	The desired dummy plane value
						
						\note		A value of -1 means that the value stored in dummy planes will be undefined.
					*/
					int16_t dummyPlaneValue() const;
			
					/**
						\brief	Sets the value to store into any dummy planes
						\author	GW
						\date	01/2018

						\param	dpv_	The desired dummy plane value
						
						\note	Setting a value of -1 means that the value stored in dummy planes will be undefined,
								whereas a value in range [0,255] indicates the specific desired value.
					*/
					void setDummyPlaneValue(const int16_t dpv_);
			
					/**
						\brief		Gets whether Photoshop supports absolute channel indexing. Absolute channel indexing
									ignores visibility concerns and numbers the channels from zero starting with the
									first composite channel. If existing, transparency follows, followed by any layer
									masks, then alpha channels.
						\author		GW
						\date		01/2018

						\returns	Whether absolute indexing is supported
					*/
					bool supportsAbsolute() const;
			
					/**
						\brief		Gets whether absolute channel indexing is enabled for the input (this is only useful
									if supportsAbsolute() returns true)
						\author		GW
						\date		01/2018

						\returns	Whether absolute indexing is enabled
					*/
					bool wantsAbsolute() const;
			
					/**
						\brief		Gets whether undo abilities are disabled for the user for the filter (as previously
									set by the setCannotUndo() method)
						\author		GW
						\date		01/2018

						\returns	true if Photoshop is prevented from offering undo abilities
					*/
					bool cannotUndo() const;
			
					/**
						\brief	Sets whether undo abilities are disabled for the user for the filter
						\author	GW
						\date	01/2018
						
						\param	undo_
						
						\note	This is rarely needed and usually frustrates users, so strongly consider whether this is
								truly necessary to be set to true.
					*/
					void setCannotUndo(const bool undo_);
			
					/**
						\brief		Gets whether Photoshop supports requests outside the image area
						\author		GW
						\date		01/2018

						\returns	true if out-of-range image area requests are supported
					*/
					bool supportsPadding() const;
			
					/**
						\brief		Gets the value previously set by the setInputPadding() method
						\author		GW
						\date		01/2018

						\returns	How the input should be padded
					*/
					int16_t inputPadding() const;
			
					/**
						\brief	Sets the value describing how input should be padded
						\author	GW
						\date	01/2018

						\param	pad_	A value in range [0,255] to specify the exact padding value, -1 to indicate edge
										replication, -2 to indicate no specific padding (i.e. the data will be random),
										or -3 to indicate that an error should be signaled for out-of-bounds requests
										(the default)
					*/
					void setInputPadding(const int16_t pad_);
			
					/**
						\brief		Gets the value previously set by the setOutputPadding() method
						\author		GW
						\date		01/2018

						\returns	How the output should be padded
					*/
					int16_t outputPadding() const;
			
					/**
						\brief	Sets the value describing how output should be padded
						\author	GW
						\date	01/2018

						\param	pad_	A value in range [0,255] to specify the exact padding value, -1 to indicate edge
										replication, -2 to indicate no specific padding (i.e. the data will be random),
										or -3 to indicate that an error should be signaled for out-of-bounds requests
										(the default)
					*/
					void setOutputPadding(const int16_t pad_);
			
					/**
						\brief		Gets the value previously set by the setMaskPadding() method
						\author		GW
						\date		01/2018

						\returns	How the mask should be padded
					*/
					int16_t maskPadding() const;
			
					/**
						\brief	Sets the value describing how the mask should be padded
						\author	GW
						\date	01/2018

						\param	pad_	A value in range [0,255] to specify the exact padding value, -1 to indicate edge
										replication, -2 to indicate no specific padding (i.e. the data will be random),
										or -3 to indicate that an error should be signaled for out-of-bounds requests
										(the default)
					*/
					void setMaskPadding(const int16_t pad_);
			
					/**
						\brief		Gets whether Photoshop supports non-1:1 sampling of the input and mask
						\author		GW
						\date		01/2018

						\returns	true if supported, false otherwise
					*/
					bool samplingSupport() const;
			
					/**
						\brief		Gets the sampling rate for the input
						\author		GW
						\date		01/2018

						\returns	The sampling rate (see notes)
						
						\note		The effective input rectangle in normal sampling coordinates is inRect() * inputRate().
									See example below.
						\note		The returned value is rounded to the nearest integer. Since the scaled rectangle
									may exceed the real source data, it is a good idea to set some sort of padding for
									the input as well.
						
						\code
							hdi::core::Rect(
								inRect().origin.x * inputRate(),
								inRect().origin.y * inputRate(),
								inRect().size.width * inputRate(),
								inRect().size.height * inputRate()
							)
						\endcode
					*/
					double inputRate() const;
			
					/**
						\brief	Sets the sampling rate for the input
						\author	GW
						\date	01/2018

						\param	rate_	The new sampling rate (see notes for inputRate() method)
					*/
					void setInputRate(const double rate_);
			
					/**
						\brief		Gets the sampling rate for the mask
						\author		GW
						\date		01/2018

						\returns	The sampling rate (see notes)
						
						\note		The effective input rectangle in normal sampling coordinates is maskRect() * maskRate().
									See example below.
						\note		The returned value is rounded to the nearest integer. Since the scaled rectangle
									may exceed the real source data, it is a good idea to set some sort of padding for
									the input as well.
						
						\code
							hdi::core::Rect(
								maskRect().origin.x * maskRate(),
								maskRect().origin.y * maskRate(),
								maskRect().size.width * maskRate(),
								maskRect().size.height * maskRate()
							)
						\endcode
					*/
					double maskRate() const;
			
					/**
						\brief	Sets the sampling rate for the mask
						\author	GW
						\date	01/2018

						\param	rate_	The new sampling rate (see notes for maskRate() method)
					*/
					void setMaskRate(const double rate_);
			
					/**
						\brief		Gets the number of target layer planes for the input data
						\author		GW
						\date		01/2018

						\returns	The target input layer plane count
						
						\note		If all the input plane values are zero, then you should assume that Photoshop has
									not set them.
						\note		When processing layer data, Photoshop structures its input and output data as follows:
									1) target layer channels, 2) transparency mask for target layer, 3) layer mask
									channels for target layer, 4) inverted layer mask channels for target layer, 5) non-
									layer channels.
						\note		The output planes are a prefix of the input planes. For example, in the protected
									transparency case, the input can contain a transparency mask and a layer mask while
									the output can contain just the layerPlanes.
						\note		When processing non-layer data (including running a filter on the layer mask alone),
									Photoshop structures the data as consisting only of non-layer channels. It indicates
									this structure through a series of short counts. The transparency count must be
									either 0 or 1.
					*/
					int16_t inLayerPlanes() const;
			
					/**
						\brief		Gets the number of transparency masks for the input target layer data
						\author		GW
						\date		01/2018

						\returns	The target input transparency mask count
						
						\note		See documentation for inLayerPlanes() for more info.
					*/
					int16_t inTransparencyMask() const;
			
					/**
						\brief		Gets the number of layer mask channels for the input target layer
						\author		GW
						\date		01/2018

						\returns	The target input layer mask channel count
						
						\note		See documentation for inLayerPlanes() for more info.
					*/
					int16_t inLayerMasks() const;
			
					/**
						\brief		Gets the number of inverted layer mask channels for the input target layer
						\author		GW
						\date		01/2018

						\returns	The target input inverted layer mask channel count
						
						\note		See documentation for inLayerPlanes() for more info.
					*/
					int16_t inInvertedLayerMasks() const;
			
					/**
						\brief		Gets the number of non-layer channels for the input data
						\author		GW
						\date		01/2018

						\returns	The input non-layer channel count
						
						\note		See documentation for inLayerPlanes() for more info.
					*/
					int16_t inNonLayerPlanes() const;
			
					/**
						\brief		Gets the number of target layer planes for the output data
						\author		GW
						\date		01/2018

						\returns	The target output layer plane count
						
						\note		See documentation for inLayerPlanes() for more info.
					*/
					int16_t outLayerPlanes() const;
			
					/**
						\brief		Gets the number of transparency masks for the output target layer data
						\author		GW
						\date		01/2018

						\returns	The target output transparency mask count
						
						\note		See documentation for inLayerPlanes() for more info.
					*/
					int16_t outTransparencyMask() const;
			
					/**
						\brief		Gets the number of layer mask channels for the output target layer
						\author		GW
						\date		01/2018

						\returns	The target output layer mask channel count
						
						\note		See documentation for inLayerPlanes() for more info.
					*/
					int16_t outLayerMasks() const;
			
					/**
						\brief		Gets the number of inverted layer mask channels for the output target layer
						\author		GW
						\date		01/2018

						\returns	The target output inverted layer mask channel count
						
						\note		See documentation for inLayerPlanes() for more info.
					*/
					int16_t outInvertedLayerMasks() const;
			
					/**
						\brief		Gets the number of non-layer channels for the output data
						\author		GW
						\date		01/2018

						\returns	The output non-layer channel count
						
						\note		See documentation for inLayerPlanes() for more info.
					*/
					int16_t outNonLayerPlanes() const;
			
					/**
						\brief		Gets the number of target layer planes for the input data, used for the structure of
									the input data when wantsAbsolute() returns true
						\author		GW
						\date		01/2018

						\returns	The target input layer planes count
						
						\note		See documentation for inLayerPlanes() for more info.
					*/
					int16_t absLayerPlanes() const;
			
					/**
						\brief		Gets the number of transparency masks for the input data, used for the structure of
									the input data when wantsAbsolute() returns true
						\author		GW
						\date		01/2018

						\returns	The input transparency mask count
						
						\note		See documentation for inLayerPlanes() for more info.
					*/
					int16_t absTransparencyMask() const;
			
					/**
						\brief		Gets the number of layer mask channels for the input data, used for the structure of
									the input data when wantsAbsolute() returns true
						\author		GW
						\date		01/2018

						\returns	The input layer mask channel count
						
						\note		See documentation for inLayerPlanes() for more info.
					*/
					int16_t absLayerMasks() const;
			
					/**
						\brief		Gets the number of inverted layer mask channels for the input data, used for the
									structure of the input data when wantsAbsolute() returns true
						\author		GW
						\date		01/2018

						\returns	The input inverted layer mask channel count
						
						\note		See documentation for inLayerPlanes() for more info.
					*/
					int16_t absInvertedLayerMasks() const;
			
					/**
						\brief		Gets the number of target layer planes for the input data, used for the structure of
									the input data when wantsAbsolute() returns true
						\author		GW
						\date		01/2018

						\returns	The target input layer planes count
						
						\note		See documentation for inLayerPlanes() for more info.
					*/
					int16_t absNonLayerPlanes() const;
			
					/**
						\brief		Gets the number of extra planes before the input data. Only available if
									supportsDummyChannels() returns true.
						\author		GW
						\date		01/2018

						\returns	Count of extra prefix dummy planes
					*/
					int16_t inPreDummyPlanes() const;
			
					/**
						\brief		Gets the number of extra planes after the input data. Only available if
									supportsDummyChannels() returns true.
						\author		GW
						\date		01/2018

						\returns	Count of extra postfix dummy planes
					*/
					int16_t inPostDummyPlanes() const;
			
					/**
						\brief		Gets the number of extra planes before the output data. Only available if
									supportsDummyChannels() returns true.
						\author		GW
						\date		01/2018

						\returns	Count of extra prefix dummy planes
					*/
					int16_t outPreDummyPlanes() const;
			
					/**
						\brief		Gets the number of extra planes after the output data. Only available if
									supportsDummyChannels() returns true.
						\author		GW
						\date		01/2018

						\returns	Count of extra postfix dummy planes
					*/
					int16_t outPostDummyPlanes() const;
			
					/**
						\brief		Gets the step from column to column in the input
						\author		GW
						\date		01/2018

						\returns	The column step, in bytes
						
						\note		If using the layout options, this value may change from being equal to the number of
									planes. If zero, assume Photoshop has not set it.
					*/
					int32_t inColumnBytes() const;
			
					/**
						\brief		Gets the step from plane to plane in the input
						\author		GW
						\date		01/2018

						\returns	The plane step, in bytes
						
						\note		If using the layout options, this value may change from being equal to the number of
									planes. If zero, assume Photoshop has not set it.
					*/
					int32_t inPlaneBytes() const;
			
					/**
						\brief		Gets the step from column to column in the output
						\author		GW
						\date		01/2018

						\returns	The column step, in bytes
						
						\note		If using the layout options, this value may change from being equal to the number of
									planes. If zero, assume Photoshop has not set it.
					*/
					int32_t outColumnBytes() const;
			
					/**
						\brief		Gets the step from plane to plane in the output
						\author		GW
						\date		01/2018

						\returns	The plane step, in bytes
						
						\note		If using the layout options, this value may change from being equal to the number of
									planes. If zero, assume Photoshop has not set it.
					*/
					int32_t outPlaneBytes() const;
			
					/**
						\brief		Gets the tiling height for the input
						\author		GW
						\date		01/2018

						\returns	The tiling height, or 0 if not set by Photoshop
						
						\note		Your filter plugin should work at this size, if possible.
					*/
					int16_t inTileHeight() const;
			
					/**
						\brief		Gets the tiling width for the input
						\author		GW
						\date		01/2018

						\returns	The tiling width, or 0 if not set by Photoshop
						
						\note		Your filter plugin should work at this size, if possible.
					*/
					int16_t inTileWidth() const;
			
					/**
						\brief		Gets the tiling origin for the input
						\author		GW
						\date		01/2018

						\returns	The tiling origin, or (0,0) if not set by Photoshop
					*/
					Point inTileOrigin() const;
			
					/**
						\brief		Gets the tiling height for the absolute data
						\author		GW
						\date		01/2018

						\returns	The tiling height, or 0 if not set by Photoshop
						
						\note		Your filter plugin should work at this size, if possible.
					*/
					int16_t absTileHeight() const;
			
					/**
						\brief		Gets the tiling width for the absolute data
						\author		GW
						\date		01/2018

						\returns	The tiling width, or 0 if not set by Photoshop
						
						\note		Your filter plugin should work at this size, if possible.
					*/
					int16_t absTileWidth() const;
			
					/**
						\brief		Gets the tiling origin for the absolute data
						\author		GW
						\date		01/2018

						\returns	The tiling origin, or (0,0) if not set by Photoshop
					*/
					Point absTileOrigin() const;
			
					/**
						\brief		Gets the tiling height for the output
						\author		GW
						\date		01/2018

						\returns	The tiling height, or 0 if not set by Photoshop
						
						\note		Your filter plugin should work at this size, if possible.
					*/
					int16_t outTileHeight() const;
			
					/**
						\brief		Gets the tiling width for the output
						\author		GW
						\date		01/2018

						\returns	The tiling width, or 0 if not set by Photoshop
						
						\note		Your filter plugin should work at this size, if possible.
					*/
					int16_t outTileWidth() const;
			
					/**
						\brief		Gets the tiling origin for the output
						\author		GW
						\date		01/2018

						\returns	The tiling origin, or (0,0) if not set by Photoshop
					*/
					Point outTileOrigin() const;
			
					/**
						\brief		Gets the tiling height for the mask
						\author		GW
						\date		01/2018

						\returns	The tiling height, or 0 if not set by Photoshop
						
						\note		Your filter plugin should work at this size, if possible.
					*/
					int16_t maskTileHeight() const;
			
					/**
						\brief		Gets the tiling width for the mask
						\author		GW
						\date		01/2018

						\returns	The tiling width, or 0 if not set by Photoshop
						
						\note		Your filter plugin should work at this size, if possible.
					*/
					int16_t maskTileWidth() const;
			
					/**
						\brief		Gets the tiling origin for the mask
						\author		GW
						\date		01/2018

						\returns	The tiling origin, or (0,0) if not set by Photoshop
					*/
					Point maskTileOrigin() const;
			
					/**
						\brief		Gets the error to be reported by Photoshop, previously set via setErrorString()
						\author		GW
						\date		01/2018

						\returns	The error reporting string, if any, in UTF-8
					*/
					std::string errorString() const;
			
					/**
						\brief		Sets the error to be reported by Photoshop
						\author		GW
						\date		01/2018
						
						\param		error_	Desired error string, in UTF-8
						
						\warning	Photoshop enforces a limit on this string of 255 bytes.
						
						\note		If this is set to a non-empty value and your plugin returns errReportString (from the
									Photoshop SDK) to Photoshop, then a dialog will be shown with the text: "Cannot complete
									operation because " + error_
					*/
					void setErrorString(const std::string& error_);
			
					/**
						\brief		Gets the Channel Ports document information for the document being filtered
						\author		GW
						\date		01/2018

						\returns	Document info struct pointer
					*/
					DocumentInfo* const documentInfo() const;
			
					/**
						\brief		Gets the bit depth per channel
						\author		GW
						\date		01/2018

						\returns	The bit depth; will be one of 1, 8, 16, or 32
					*/
					int32_t depth() const;
			
					/**
						\brief		Gets if Photoshop has image scrap
						\author		GW
						\date		01/2018

						\returns	A non-zero value if Photoshop has image scrap
						
						\note		You can ask for the exporting of image scrap by setting the PiPL resource property
									PIWantsScrapProperty. The document info for the image scrap is chained right behind
									the targeted document pointed by the documentInfo() method. Photoshop sets the
									results of this method to indicate that an image scrap is available. You can use it
									to tell whether Photoshop failed to export the scrap because some unknown reasons or
									there is no scrap at all.
					*/
					int32_t hasImageScrap() const;
			
					/**
						\brief		Gets the size of the image
						\author		GW
						\date		01/2018

						\returns	Image dimensions
					*/
					Size imageSize() const;
			
					/**
						\brief		Gets the rectangle to filter
						\author		GW
						\date		01/2018

						\returns	Filter rect
						
						\note		This is the bounding box of the selection, or if there is no selection, the bounding
									box of the image. If the selection is not a perfect rectangle, Photoshop automatically
									masks the changes to the area actually selected (unless you turns off this feature
									using the setAutoMask() method). This allows most filters to ignore the selection
									mask, and still operate correctly.
					*/
					Rect filterRect() const;
			
					/**
						\brief		Gets the input rectangle, as previously set via setInRect()
						\author		GW
						\date		01/2018

						\returns	Input rect
					*/
					Rect inRect() const;
			
					/**
						\brief	Sets the input rectangle
						\author	GW
						\date	01/2018

						\param	ir_		Input rect
						
						\note	You should set this in the FilterStartMessageType and FilterContinueMessageType
								handlers to request access to an area of the input image. The area requested must be
								a subset of the image's bounding rectangle. After the entire filter rect has been
								filtered, this should be set to an empty rectangle.
					*/
					void setInRect(const Rect& ir_);
			
					/**
						\brief		Gets the output rectangle, as previously set via setOutRect()
						\author		GW
						\date		01/2018

						\returns	Output rect
					*/
					Rect outRect() const;
			
					/**
						\brief	Sets the output rectangle
						\author	GW
						\date	01/2018

						\param	or_		Output rect
						
						\note	You should set this in the FilterStartMessageType and FilterContinueMessageType
								handlers to request access to an area of the input image. The area requested must be
								a subset of the image's bounding rectangle. After the entire filter rect has been
								filtered, this should be set to an empty rectangle.
					*/
					void setOutRect(const Rect& or_);
			
					/**
						\brief		Gets the mask rectangle, as previously set via setMaskRect()
						\author		GW
						\date		01/2018

						\returns	Mask rect
					*/
					Rect maskRect() const;
			
					/**
						\brief	Sets the mask rectangle
						\author	GW
						\date	01/2018

						\param	mr_		Mask rect
						
						\note	If hasMask() returns true, and you need access to the selection mask, you should set
								this field in your FilterStartMessageType and FilterContinueMessageType handlers to
								request access to an area of the selection mask. The requested area must be a subset of
								filterRect(). This method is ignored if there is no selection mask.
					*/
					void setMaskRect(const Rect& mr_);
			
					/**
						\brief		Gets the coordinate of the selection
						\author		GW
						\date		01/2018

						\returns	Selection origin
					*/
					Point floatCoordinate() const;
			
					/**
						\brief		Gets the size of the image the selection is over
						\author		GW
						\date		01/2018

						\returns	Whole image size
					*/
					Size wholeSize() const;
				#endif
				// HDI_CORE_PSP_MODE


			private:
				// Only Dispatcher can construct a FilterMessage object
				friend class Dispatcher;

				/**
					\brief	Internal use only
					\author	GW
					\date	08/2013
				*/
				FilterMessage(const MessageType, void*);
				
				/**
					\brief	Unused
					\author	GW
					\date	02/2015
				*/
				FilterMessage(const FilterMessage&);
				
				/**
					\brief	Unused
					\author	GW
					\date	02/2015
				*/
				FilterMessage& operator=(const FilterMessage&);
		};
		
		#if defined(HDI_CORE_AIP_MODE)
			/**
				\brief	Received for LayerListPushMessageType, LayerListPopMessageType, and LayerListDeleteMessageType
			*/
			class LayerListMessage : public Message
			{
				public:
					/**
						\brief	Destructs a LayerListMessage object
						\author	GW
						\date	08/2013
					*/
					virtual ~LayerListMessage();


				private:
					// Only Dispatcher can construct a LayerListMessage object
					friend class Dispatcher;

					/**
						\brief	Internal use only
						\author	GW
						\date	08/2013
					*/
					LayerListMessage(const MessageType, void*);
					
					/**
						\brief	Unused
						\author	GW
						\date	02/2015
					*/
					LayerListMessage(const LayerListMessage&);
					
					/**
						\brief	Unused
						\author	GW
						\date	02/2015
					*/
					LayerListMessage& operator=(const LayerListMessage&);
			};
			
			/**
				\brief	Base class for all LiveEffect*Message subclasses
			*/
			class LiveEffectMessage : public Message
			{
				public:
					/**
						\brief	Destructs a LiveEffectMessage object
						\author	GW
						\date	08/2013
					*/
					virtual ~LiveEffectMessage();


				protected:
					/**
						\brief	Internal use only
						\author	GW
						\date	08/2013
					*/
					LiveEffectMessage(const MessageType, void*);
					
					/**
						\brief	Unused
						\author	GW
						\date	02/2015
					*/
					LiveEffectMessage(const LiveEffectMessage&);
					
					/**
						\brief	Unused
						\author	GW
						\date	02/2015
					*/
					LiveEffectMessage& operator=(const LiveEffectMessage&);
			};
			
			/**
				\brief	Received for LiveEffectGoMessageType
			*/
			class LiveEffectGoMessage : public LiveEffectMessage
			{
				public:
					/**
						\brief	Destructs a LiveEffectGoMessage object
						\author	GW
						\date	08/2013
					*/
					virtual ~LiveEffectGoMessage();


				private:
					// Only Dispatcher can construct a LiveEffectGoMessage object
					friend class Dispatcher;

					/**
						\brief	Internal use only
						\author	GW
						\date	08/2013
					*/
					LiveEffectGoMessage(const MessageType, void*);
					
					/**
						\brief	Unused
						\author	GW
						\date	02/2015
					*/
					LiveEffectGoMessage(const LiveEffectGoMessage&);
					
					/**
						\brief	Unused
						\author	GW
						\date	02/2015
					*/
					LiveEffectGoMessage& operator=(const LiveEffectGoMessage&);
			};
			
			/**
				\brief	Received for LiveEffectEditParamsMessageType
			*/
			class LiveEffectEditParamsMessage : public LiveEffectMessage
			{
				public:
					/**
						\brief	Destructs a LiveEffectEditParamsMessage object
						\author	GW
						\date	08/2013
					*/
					virtual ~LiveEffectEditParamsMessage();


				private:
					// Only Dispatcher can construct a LiveEffectEditParamsMessage object
					friend class Dispatcher;

					/**
						\brief	Internal use only
						\author	GW
						\date	08/2013
					*/
					LiveEffectEditParamsMessage(const MessageType, void*);
					
					/**
						\brief	Unused
						\author	GW
						\date	02/2015
					*/
					LiveEffectEditParamsMessage(const LiveEffectEditParamsMessage&);
					
					/**
						\brief	Unused
						\author	GW
						\date	02/2015
					*/
					LiveEffectEditParamsMessage& operator=(const LiveEffectEditParamsMessage&);
			};
			
			/**
				\brief	Received for LiveEffectInterpolateMessageType
			*/
			class LiveEffectInterpolateMessage : public LiveEffectMessage
			{
				public:
					/**
						\brief	Destructs a LiveEffectInterpolateMessage object
						\author	GW
						\date	08/2013
					*/
					virtual ~LiveEffectInterpolateMessage();


				private:
					// Only Dispatcher can construct a LiveEffectInterpolateMessage object
					friend class Dispatcher;

					/**
						\brief	Internal use only
						\author	GW
						\date	08/2013
					*/
					LiveEffectInterpolateMessage(const MessageType, void*);
					
					/**
						\brief	Unused
						\author	GW
						\date	02/2015
					*/
					LiveEffectInterpolateMessage(const LiveEffectInterpolateMessage&);
					
					/**
						\brief	Unused
						\author	GW
						\date	02/2015
					*/
					LiveEffectInterpolateMessage& operator=(const LiveEffectInterpolateMessage&);
			};
			
			/**
				\brief	Received for LiveEffectInputMessageType
			*/
			class LiveEffectInputMessage : public LiveEffectMessage
			{
				public:
					/**
						\brief	Destructs a LiveEffectInputMessage object
						\author	GW
						\date	08/2013
					*/
					virtual ~LiveEffectInputMessage();


				private:
					// Only Dispatcher can construct a LiveEffectInputMessage object
					friend class Dispatcher;

					/**
						\brief	Internal use only
						\author	GW
						\date	08/2013
					*/
					LiveEffectInputMessage(const MessageType, void*);
					
					/**
						\brief	Unused
						\author	GW
						\date	02/2015
					*/
					LiveEffectInputMessage(const LiveEffectInputMessage&);
					
					/**
						\brief	Unused
						\author	GW
						\date	02/2015
					*/
					LiveEffectInputMessage& operator=(const LiveEffectInputMessage&);
			};
			
			/**
				\brief	Received for LiveEffectConvertColorSpaceMessageType
			*/
			class LiveEffectConvertColorSpaceMessage : public LiveEffectMessage
			{
				public:
					/**
						\brief	Destructs a LiveEffectConvertColorSpaceMessage object
						\author	GW
						\date	08/2013
					*/
					virtual ~LiveEffectConvertColorSpaceMessage();


				private:
					// Only Dispatcher can construct a LiveEffectConvertColorSpaceMessage object
					friend class Dispatcher;

					/**
						\brief	Internal use only
						\author	GW
						\date	08/2013
					*/
					LiveEffectConvertColorSpaceMessage(const MessageType, void*);
					
					/**
						\brief	Unused
						\author	GW
						\date	02/2015
					*/
					LiveEffectConvertColorSpaceMessage(const LiveEffectConvertColorSpaceMessage&);
					
					/**
						\brief	Unused
						\author	GW
						\date	02/2015
					*/
					LiveEffectConvertColorSpaceMessage& operator=(const LiveEffectConvertColorSpaceMessage&);
			};
			
			/**
				\brief	Received for LiveEffectAdjustColorsMessageType
			*/
			class LiveEffectAdjustColorsMessage : public LiveEffectMessage
			{
				public:
					/**
						\brief	Destructs a LiveEffectAdjustColorsMessage object
						\author	GW
						\date	08/2013
					*/
					virtual ~LiveEffectAdjustColorsMessage();


				private:
					// Only Dispatcher can construct a LiveEffectAdjustColorsMessage object
					friend class Dispatcher;

					/**
						\brief	Internal use only
						\author	GW
						\date	08/2013
					*/
					LiveEffectAdjustColorsMessage(const MessageType, void*);
					
					/**
						\brief	Unused
						\author	GW
						\date	02/2015
					*/
					LiveEffectAdjustColorsMessage(const LiveEffectAdjustColorsMessage&);
					
					/**
						\brief	Unused
						\author	GW
						\date	02/2015
					*/
					LiveEffectAdjustColorsMessage& operator=(const LiveEffectAdjustColorsMessage&);
			};
			
			/**
				\brief	Received for LiveEffectScaleParamsMessageType
			*/
			class LiveEffectScaleParamsMessage : public LiveEffectMessage
			{
				public:
					/**
						\brief	Destructs a LiveEffectScaleParamsMessage object
						\author	GW
						\date	08/2013
					*/
					virtual ~LiveEffectScaleParamsMessage();


				private:
					// Only Dispatcher can construct a LiveEffectScaleParamsMessage object
					friend class Dispatcher;

					/**
						\brief	Internal use only
						\author	GW
						\date	08/2013
					*/
					LiveEffectScaleParamsMessage(const MessageType, void*);
					
					/**
						\brief	Unused
						\author	GW
						\date	02/2015
					*/
					LiveEffectScaleParamsMessage(const LiveEffectScaleParamsMessage&);
					
					/**
						\brief	Unused
						\author	GW
						\date	02/2015
					*/
					LiveEffectScaleParamsMessage& operator=(const LiveEffectScaleParamsMessage&);
			};
			
			/**
				\brief	Received for LiveEffectMergeMessageType
			*/
			class LiveEffectMergeMessage : public LiveEffectMessage
			{
				public:
					/**
						\brief	Destructs a LiveEffectMergeMessage object
						\author	GW
						\date	08/2013
					*/
					virtual ~LiveEffectMergeMessage();


				private:
					// Only Dispatcher can construct a LiveEffectMergeMessage object
					friend class Dispatcher;

					/**
						\brief	Internal use only
						\author	GW
						\date	08/2013
					*/
					LiveEffectMergeMessage(const MessageType, void*);
					
					/**
						\brief	Unused
						\author	GW
						\date	02/2015
					*/
					LiveEffectMergeMessage(const LiveEffectMergeMessage&);
					
					/**
						\brief	Unused
						\author	GW
						\date	02/2015
					*/
					LiveEffectMergeMessage& operator=(const LiveEffectMergeMessage&);
			};
			
			/**
				\brief	Received for LiveEffectGetSVGFilterMessageType
			*/
			class LiveEffectGetSVGFilterMessage : public LiveEffectMessage
			{
				public:
					/**
						\brief	Destructs a LiveEffectGetSVGFilterMessage object
						\author	GW
						\date	08/2013
					*/
					virtual ~LiveEffectGetSVGFilterMessage();


				private:
					// Only Dispatcher can construct a LiveEffectGetSVGFilterMessage object
					friend class Dispatcher;

					/**
						\brief	Internal use only
						\author	GW
						\date	08/2013
					*/
					LiveEffectGetSVGFilterMessage(const MessageType, void*);
					
					/**
						\brief	Unused
						\author	GW
						\date	02/2015
					*/
					LiveEffectGetSVGFilterMessage(const LiveEffectGetSVGFilterMessage&);
					
					/**
						\brief	Unused
						\author	GW
						\date	02/2015
					*/
					LiveEffectGetSVGFilterMessage& operator=(const LiveEffectGetSVGFilterMessage&);
			};
			
			/**
				\brief	Received for LiveEffectIsCompatibleMessageType
			*/
			class LiveEffectIsCompatibleMessage : public LiveEffectMessage
			{
				public:
					/**
						\brief	Destructs a LiveEffectIsCompatibleMessage object
						\author	GW
						\date	08/2013
					*/
					virtual ~LiveEffectIsCompatibleMessage();


				private:
					// Only Dispatcher can construct a LiveEffectIsCompatibleMessage object
					friend class Dispatcher;

					/**
						\brief	Internal use only
						\author	GW
						\date	08/2013
					*/
					LiveEffectIsCompatibleMessage(const MessageType, void*);
					
					/**
						\brief	Unused
						\author	GW
						\date	02/2015
					*/
					LiveEffectIsCompatibleMessage(const LiveEffectIsCompatibleMessage&);
					
					/**
						\brief	Unused
						\author	GW
						\date	02/2015
					*/
					LiveEffectIsCompatibleMessage& operator=(const LiveEffectIsCompatibleMessage&);
			};
		#endif
		// HDI_CORE_AIP_MODE
		
		#if defined(HDI_CORE_PSP_MODE)
			/**
				\brief	Received for MeasurementRegisterDataPointDataTypesMessageType, 
						MeasurementRegisterDataPointsMessageType, MeasurementPrepareMeasurementsMessageType,
						MeasurementRecordMeasurementsMessageType, and MeasurementExportMeasurementMessageType
			*/
			class MeasurementMessage : public Message
			{
				public:
					/**
						\brief	Destructs a MeasurementMessage object
						\author	GW
						\date	12/2017
					*/
					virtual ~MeasurementMessage();


				private:
					// Only Dispatcher can construct a MeasurementMessage object
					friend class Dispatcher;

					/**
						\brief	Internal use only
						\author	GW
						\date	12/2017
					*/
					MeasurementMessage(const MessageType, void*);
					
					/**
						\brief	Unused
						\author	GW
						\date	12/2017
					*/
					MeasurementMessage(const MeasurementMessage&);
					
					/**
						\brief	Unused
						\author	GW
						\date	12/2017
					*/
					MeasurementMessage& operator=(const MeasurementMessage&);
			};
		#endif
		// HDI_CORE_PSP_MODE
		
		#if defined(HDI_CORE_AIP_MODE)
			/**
				\brief	Received for MenuGoMessageType and MenuUpdateMessageType
			*/
			class MenuMessage : public Message
			{
				public:
					/**
						\brief	Destructs a MenuMessage object
						\author	GW
						\date	08/2013
					*/
					virtual ~MenuMessage();
					
					/**
						\brief		Gets the MenuItem object to which the message applies
						\author		GW
						\date		08/2013
						
						\returns	The menu item that should be updated
					*/
					MenuItem* const menuItem() const;


				private:
					// Only Dispatcher can construct a MenuMessage object
					friend class Dispatcher;

					/**
						\brief	Internal use only
						\author	GW
						\date	08/2013
					*/
					MenuMessage(const MenuItem&, const MessageType, void*);
					
					/**
						\brief	Unused
						\author	GW
						\date	02/2015
					*/
					MenuMessage(const MenuMessage&);
					
					/**
						\brief	Unused
						\author	GW
						\date	02/2015
					*/
					MenuMessage& operator=(const MenuMessage&);
			};
			
			/**
				\brief	Received for NotifyMessageType
			*/
			class NotifyMessage : public Message
			{
				public:
					/**
						\brief	Destructs a NotifyMessage object
						\author	GW
						\date	08/2013
					*/
					virtual ~NotifyMessage();
					
					/**
						\brief		Gets the Notifier object to which the message applies
						\author		GW
						\date		08/2013
						
						\returns	The notifier that should fire
					*/
					Notifier* const notifier() const;
				
					/**
						\brief		Gets the Notification data object for the received notifier message, if any is relevant
						\author		GW
						\date		07/2017
						
						\param		Relevant Notification data object, or NULL for none
					*/
					Notification* const notification() const;


				private:
					// Only Dispatcher can construct a NotifyMessage object
					friend class Dispatcher;

					/**
						\brief	Internal use only
						\author	GW
						\date	08/2013
					*/
					NotifyMessage(const Notifier&, const MessageType, void*);
					
					/**
						\brief	Unused
						\author	GW
						\date	02/2015
					*/
					NotifyMessage(const NotifyMessage&);
					
					/**
						\brief	Unused
						\author	GW
						\date	02/2015
					*/
					NotifyMessage& operator=(const NotifyMessage&);
			};
			
			/**
				\brief	Received for ObjectSetDisposeContentsMessageType, ObjectSetMarkUsageMessageType,
						ObjectSetUpdateInternalMessageType, ObjectSetUpdateExternalMessageType, ObjectSetPurgeMessageType,
						ObjectSetChangeMessageType, ObjectSetChangeIndirectMessageType, ObjectSetCopyMessageType, and
						ObjectSetInitDoneMessageType
			*/
			class ObjectSetMessage : public Message
			{
				public:
					/**
						\brief	Destructs a ObjectSetMessage object
						\author	GW
						\date	08/2013
					*/
					virtual ~ObjectSetMessage();


				private:
					// Only Dispatcher can construct a ObjectSetMessage object
					friend class Dispatcher;

					/**
						\brief	Internal use only
						\author	GW
						\date	08/2013
					*/
					ObjectSetMessage(const MessageType, void*);
					
					/**
						\brief	Unused
						\author	GW
						\date	02/2015
					*/
					ObjectSetMessage(const ObjectSetMessage&);
					
					/**
						\brief	Unused
						\author	GW
						\date	02/2015
					*/
					ObjectSetMessage& operator=(const ObjectSetMessage&);
			};
		#endif
		// HDI_CORE_AIP_MODE
		
		/**
			\brief	Received for PluginReloadMessageType, PluginPurgeCachesMessageType, and PluginUnloadMessageType
		*/
		class PluginMessage : public Message
		{
			public:
				/**
					\brief	Destructs a PluginMessage object
					\author	GW
					\date	08/2013
				*/
				virtual ~PluginMessage();


			private:
				// Only Dispatcher can construct a PluginMessage object
				friend class Dispatcher;

				/**
					\brief	Internal use only
					\author	GW
					\date	08/2013
				*/
				PluginMessage(const MessageType, void*);
				
				/**
					\brief	Unused
					\author	GW
					\date	02/2015
				*/
				PluginMessage(const PluginMessage&);
				
				/**
					\brief	Unused
					\author	GW
					\date	02/2015
				*/
				PluginMessage& operator=(const PluginMessage&);
		};
		
		#if defined(HDI_CORE_AIP_MODE)
			/**
				\brief	Received for CustomArtNotifyEditsMessageType, CustomArtUpdateViewableArtMessageType,
						CustomArtCanShowContentsMessageType, CustomArtInterpolateDataMessageType,
						CustomArtReplaceSymbolMessageType, CustomArtDisallowsArtTypeQueryMessageType,
						CustomArtPerformCustomHitTestMessageType, CustomArtCollectPaintStylesMessageType,
						CustomArtApplyPaintStylesMessageType, CustomArtAdjustColorsMessageType, and
						CustomArtShowContentsMessageType
			*/
			class CustomArtMessage : public Message
			{
				public:
					/**
						\brief	Destructs a CustomArtMessage object
						\author	GW
						\date	08/2013
					*/
					virtual ~CustomArtMessage();
				
					/**
						\brief		Custom art to which the received message applies
						\author		GW
						\date		07/2017
						
						\returns	A pointer to the art received with the message, NULL for none/error
						
						\note		This applies to all custom art message types aside from CustomArtDisallowsArtTypeQueryMessageType.
					*/
					std::unique_ptr<Art> customArt() const;


				private:
					// Only Dispatcher can construct a CustomArtMessage object
					friend class Dispatcher;

					/**
						\brief	Internal use only
						\author	GW
						\date	08/2013
					*/
					CustomArtMessage(const MessageType, void*);
					
					/**
						\brief	Unused
						\author	GW
						\date	02/2015
					*/
					CustomArtMessage(const CustomArtMessage&);
					
					/**
						\brief	Unused
						\author	GW
						\date	02/2015
					*/
					CustomArtMessage& operator=(const CustomArtMessage&);
			};
		#endif
		// HDI_CORE_AIP_MODE
		
		#if defined(HDI_CORE_PSP_MODE)
			/**
				\brief	Received for PickerPickMessageType
			*/
			class PickerMessage : public Message
			{
				public:
					/**
						\brief	Destructs a PickerMessage object
						\author	GW
						\date	12/2017
					*/
					virtual ~PickerMessage();
				

				private:
					// Only Dispatcher can construct a PickerMessage object
					friend class Dispatcher;

					/**
						\brief	Internal use only
						\author	GW
						\date	12/2017
					*/
					PickerMessage(const MessageType, void*);
					
					/**
						\brief	Unused
						\author	GW
						\date	12/2017
					*/
					PickerMessage(const PickerMessage&);
					
					/**
						\brief	Unused
						\author	GW
						\date	12/2017
					*/
					PickerMessage& operator=(const PickerMessage&);
			};
		#endif
		// HDI_CORE_PSP_MODE
		
		#if defined(HDI_CORE_AIP_MODE)
			/**
				\brief	Received for PreferencesInitMessageType, PreferencesOKMessageType, PreferencesCancelMessageType, and
						PreferencesUpdateMessageType
			*/
			class PreferencesMessage : public Message
			{
				public:
					/**
						\brief	Destructs a PreferencesMessage object
						\author	GW
						\date	08/2013
					*/
					virtual ~PreferencesMessage();


				private:
					// Only Dispatcher can construct a PreferencesMessage object
					friend class Dispatcher;

					/**
						\brief	Internal use only
						\author	GW
						\date	08/2013
					*/
					PreferencesMessage(const MessageType, void*);
					
					/**
						\brief	Unused
						\author	GW
						\date	02/2015
					*/
					PreferencesMessage(const PreferencesMessage&);
					
					/**
						\brief	Unused
						\author	GW
						\date	02/2015
					*/
					PreferencesMessage& operator=(const PreferencesMessage&);
			};
			
			/**
				\brief	Received for PropertiesAcquireMessageType and PropertiesReleaseMessageType
			*/
			class PropertiesMessage : public Message
			{
				public:
					/**
						\brief	Destructs a PropertiesMessage object
						\author	GW
						\date	08/2013
					*/
					virtual ~PropertiesMessage();


				private:
					// Only Dispatcher can construct a PropertiesMessage object
					friend class Dispatcher;

					/**
						\brief	Internal use only
						\author	GW
						\date	08/2013
					*/
					PropertiesMessage(const MessageType, void*);
					
					/**
						\brief	Unused
						\author	GW
						\date	02/2015
					*/
					PropertiesMessage(const PropertiesMessage&);
					
					/**
						\brief	Unused
						\author	GW
						\date	02/2015
					*/
					PropertiesMessage& operator=(const PropertiesMessage&);
			};
		#endif
		// HDI_CORE_AIP_MODE
		
		#if defined(HDI_CORE_PSP_MODE)
			/**
				\brief	Received for SelectionExecuteMessageType
			*/
			class SelectionMessage : public Message
			{
				public:
					/**
						\brief	Destructs a SelectionMessage object
						\author	GW
						\date	12/2017
					*/
					virtual ~SelectionMessage();
				

				private:
					// Only Dispatcher can construct a SelectionMessage object
					friend class Dispatcher;

					/**
						\brief	Internal use only
						\author	GW
						\date	12/2017
					*/
					SelectionMessage(const MessageType, void*);
					
					/**
						\brief	Unused
						\author	GW
						\date	12/2017
					*/
					SelectionMessage(const SelectionMessage&);
					
					/**
						\brief	Unused
						\author	GW
						\date	12/2017
					*/
					SelectionMessage& operator=(const SelectionMessage&);
			};
		#endif
		// HDI_CORE_PSP_MODE
		
		#if defined(HDI_CORE_AIP_MODE)
			/**
				\brief	Received for TimerGoMessageType
			*/
			class TimerMessage : public Message
			{
				public:
					/**
						\brief	Destructs a TimerMessage object
						\author	GW
						\date	08/2013
					*/
					virtual ~TimerMessage();
					
					/**
						\brief		Gets the Timer object to which the message applies
						\author		GW
						\date		08/2013
						
						\returns	The timer that should fire
					*/
					Timer* const timer() const;


				private:
					// Only Dispatcher can construct a TimerMessage object
					friend class Dispatcher;

					/**
						\brief	Internal use only
						\author	GW
						\date	08/2013
					*/
					TimerMessage(const Timer&, const MessageType, void*);
					
					/**
						\brief	Unused
						\author	GW
						\date	02/2015
					*/
					TimerMessage(const TimerMessage&);
					
					/**
						\brief	Unused
						\author	GW
						\date	02/2015
					*/
					TimerMessage& operator=(const TimerMessage&);
			};
			
			/**
				\brief	Received for ToolEditMessageType, ToolTrackMessageType, ToolMouseDownMessageType,
						ToolMouseDragMessageType, ToolMouseUpMessageType, ToolSelectMessageType, ToolReselectMessageType,
						ToolDeselectMessageType, ToolDecreaseDiameterMessageType, ToolIncreaseDiameterMessageType,
						ToolAlternateSelectionQueryMessageType, ToolAlternateSelectionActivateMessageType, and 
						ToolAlternateSelectionDeactivateMessageType
			*/
			class ToolMessage : public Message
			{
				public:
					/**
						\brief	Normal/default tool pressure value (tool pressure values are in the range [0.0, 1.0])
					*/
					static const double normalPressure;
					
					/**
						\brief		Gets a normal/default pen tilt angle
						\author		GW
						\date		10/2013
						
						\returns	An angle for the default pen tilt
					*/
					static Angle normalTilt();
					
					/**
						\brief		Gets a normal/default pen bearing angle
						\author		GW
						\date		10/2013
						
						\returns	An angle for the default pen bearing
					*/
					static Angle normalBearing();
					
					/**
						\brief		Gets a normal/default pen rotation angle
						\author		GW
						\date		10/2013
						
						\returns	An angle for the default pen rotation
					*/
					static Angle normalRotation();
				
					/**
						\brief	Destructs a ToolMessage object
						\author	GW
						\date	08/2013
					*/
					virtual ~ToolMessage();
					
					/**
						\brief		Gets the Tool object to which the message applies
						\author		GW
						\date		08/2013
						
						\returns	The tool for which a message was received
					*/
					Tool* const tool() const;

					/**
						\brief		Gets the current position of the cursor on the artboard
						\author		GW
						\date		08/2013
						
						\returns	The position of the cursor at the time the message was received
					*/
					ArtboardPoint cursor() const;
					
					/**
						\brief		Gets the pressure of the input device
						\author		GW
						\date		08/2013
						
						\returns	The pressure at the time the message was received, or normalPressure if the device is
									not pressure-sensitive (the value is in range [0.0,1.0])
					*/
					double pressure() const;

					/**
						\brief		Gets the tangential pressure on the finger wheel of the airbrush tool
						\author		GW
						\date		09/2013
						
						\returns	The finger wheel pressure, or normalPressure if the device is not pressure-sensitive
									(the value is in range [0.0,1.0])
					*/
					double stylusWheel() const;

					/**
						\brief		Gets how the tool is angled; also called altitude or elevation
						\author		GW
						\date		09/2013
						
						\returns	The tool tilt
					*/
					Angle tilt() const;

					/**
						\brief		Gets the direction of tilt, measured clockwise around the z-axis; also called azimuth
						\author		GW
						\date		09/2013
						
						\returns	The tool bearing
					*/
					Angle bearing() const;

					/**
						\brief		Gets the rotation of the tool, measure clockwise around the tool's barrel
						\author		GW
						\date		09/2013
						
						\returns	The tool rotation
					*/
					Angle rotation() const;
					
					/**
						\brief		Gets the modifier keys that were present when the message was received
						\author		GW
						\date		08/2013
						
						\returns	The modifier keys that were pressed
					*/
					ModifierKey modifiers() const;


				private:
					// Only Dispatcher can construct a ToolMessage object
					friend class Dispatcher;

					/**
						\brief	Internal use only
						\author	GW
						\date	08/2013
					*/
					ToolMessage(const Tool&, const MessageType, void*);
					
					/**
						\brief	Unused
						\author	GW
						\date	02/2015
					*/
					ToolMessage(const ToolMessage&);
					
					/**
						\brief	Unused
						\author	GW
						\date	02/2015
					*/
					ToolMessage& operator=(const ToolMessage&);
			};
			
			/**
				\brief	Received for ToolboxStartMessageType, ToolboxEndMessageType, ToolboxStartGroupMessageType,
						ToolboxEndGroupMessageType, ToolboxStartSetMessageType, ToolboxEndSetMessageType,
						ToolboxAddToolMessageType, ToolboxAddToolRolloverMessageType, ToolboxToolSelectedMessageType,
						ToolboxToolAlternateActionSelectedMessageType, ToolboxCycleToolMessageType, and
						ToolboxSoftCycleToolMessageType
			*/
			class ToolboxMessage : public Message
			{
				public:
					/**
						\brief	Destructs a ToolboxMessage object
						\author	GW
						\date	08/2013
					*/
					virtual ~ToolboxMessage();


				private:
					// Only Dispatcher can construct a ToolboxMessage object
					friend class Dispatcher;

					/**
						\brief	Internal use only
						\author	GW
						\date	08/2013
					*/
					ToolboxMessage(const MessageType, void*);
					
					/**
						\brief	Unused
						\author	GW
						\date	02/2015
					*/
					ToolboxMessage(const ToolboxMessage&);
					
					/**
						\brief	Unused
						\author	GW
						\date	02/2015
					*/
					ToolboxMessage& operator=(const ToolboxMessage&);
			};
			
			/**
				\brief	Received for TransformAgainMessageType
			*/
			class TransformAgainMessage : public Message
			{
				public:
					/**
						\brief	Destructs a TransformAgainMessage object
						\author	GW
						\date	08/2013
					*/
					virtual ~TransformAgainMessage();


				private:
					// Only Dispatcher can construct a TransformAgainMessage object
					friend class Dispatcher;

					/**
						\brief	Internal use only
						\author	GW
						\date	08/2013
					*/
					TransformAgainMessage(const MessageType, void*);
					
					/**
						\brief	Unused
						\author	GW
						\date	02/2015
					*/
					TransformAgainMessage(const TransformAgainMessage&);
					
					/**
						\brief	Unused
						\author	GW
						\date	02/2015
					*/
					TransformAgainMessage& operator=(const TransformAgainMessage&);
			};
			
			/**
				\brief	Received for WorkspaceWriteMessageType, WorkspaceRestoreMessageType, and WorkspaceDefaultMessageType
			*/
			class WorkspaceMessage : public Message
			{
				public:
					/**
						\brief	Destructs a WorkspaceMessage object
						\author	GW
						\date	08/2013
					*/
					virtual ~WorkspaceMessage();


				private:
					// Only Dispatcher can construct a WorkspaceMessage object
					friend class Dispatcher;

					/**
						\brief	Internal use only
						\author	GW
						\date	08/2013
					*/
					WorkspaceMessage(const MessageType, void*);
					
					/**
						\brief	Unused
						\author	GW
						\date	02/2015
					*/
					WorkspaceMessage(const WorkspaceMessage&);
					
					/**
						\brief	Unused
						\author	GW
						\date	02/2015
					*/
					WorkspaceMessage& operator=(const WorkspaceMessage&);
			};
		#endif
		// HDI_CORE_AIP_MODE
	}
}

#endif
// __HDI_CORE_MESSAGE__
