/**
	\file
	\brief		Header file for number preference data storage
	\copyright	Hot Door, Inc. 2010-2025
*/

#ifndef __HDI_CORE_NUMBER_PREF__
#define __HDI_CORE_NUMBER_PREF__

#include "hdicorePrefData.h"

namespace hdi
{
	namespace core
	{
		/**
			\brief	Allows for the storage of numbers
		*/
		class NumberPref : public PrefData
		{
			public:
				/**
					\brief	Constructs an empty NumberPref object
					\author	GW
					\date	09/2013
					
					\note	To test if a NumberPref object is empty, call isEmpty() on it
					\note	Empty NumberPref objects do not relate to any actual preference data; they are designed to be
							"receivers" of some other NumberPref object via the overloaded assignment operator. Empty
							NumberPref objects are useless until such time (though it is safe to call any of their methods)
				*/
				NumberPref();
			
				/**
					\brief	Constructs a new NumberPref object from an existing NumberPref object (copy constructor)
					\author	GW
					\date	09/2013

					\param	n_	Existing NumberPref object
				*/
				NumberPref(const NumberPref& n_);

				/**
					\brief	Constructs a NumberPref object, with new data, to be added to a container at a later time
					\author	GW
					\date	09/2013

					\param	value_	Initial integer value
					\param	name_	Name for the data once it has been added to its container, as UTF-8

					\note	The name_ argument can only be "" (omitted) if the object will be added directly to an
							ArrayPref object.
				*/
				explicit NumberPref(const int64_t value_, const std::string& name_ = "");

				/**
					\brief	Constructs a NumberPref object, with new data, to be added to a container at a later time
					\author	GW
					\date	09/2013

					\param	value_	Initial double value
					\param	name_	Name for the data once it has been added to its container, as UTF-8

					\note	The name_ argument can only be "" (omitted) if the object will be added directly to an
							ArrayPref object.
				*/
				explicit NumberPref(const double value_, const std::string& name_ = "");

				/**
					\brief	Destructs a NumberPref object
					\author	GW
					\date	09/2013
				*/
				virtual ~NumberPref();

				/**
					\brief		Allows one NumberPref object to be assigned from another
					\author		GW
					\date		09/2013

					\param		rhs_	Righthand side of the = operator; the object to copy values from
					\returns	The target NumberPref object, but with its value updated to match that of rhs_
				*/
				virtual NumberPref& operator=(const NumberPref& rhs_);
				
				/**
					\brief		Convenience method to clone an NumberPref object on the heap
					\author		GW
					\date		10/2013

					\returns	A pointer to the new NumberPref object
					
					\note		If you subclass NumberPref, you MUST overload this method yourself! If you don't and/or 
								your clone() method does not return an instance of your NumberPref subclass, you will
								experience "object slicing" when adding the pref to a container.

					\warning	The caller must manage the memory for the returned NumberPref object.
				*/
				virtual NumberPref* clone() const;

				/**
					\brief		Gets the integer value for this object
					\author		GW
					\date		09/2013

					\returns	The integer value being described by this wrapper
				*/
				virtual int64_t intValue() const;

				/**
					\brief		Sets the integer value for this object
					\author		GW
					\date		09/2013

					\param		value_	New integer value
					\returns	true if the number is successfully updated, or false otherwise
				*/
				virtual bool setIntValue(const int64_t value_);

				/**
					\brief		Gets the floating point value for this object
					\author		GW
					\date		09/2013

					\returns	The double value being described by this wrapper
				*/
				virtual double floatValue() const;

				/**
					\brief		Sets the floating point value for this object
					\author		GW
					\date		09/2013

					\param		value_	New double value
					\returns	true if the number is successfully updated, or false otherwise
				*/
				virtual bool setFloatValue(double value_);
		};
		
		typedef std::unique_ptr<NumberPref> NumberPrefUP;
		typedef std::shared_ptr<NumberPref> NumberPrefSP;
		typedef std::weak_ptr<NumberPref> NumberPrefWP;
	}
}

#endif
// __HDI_CORE_NUMBER_PREF__
