/**
	\file
	\brief		Header file for path style map preference data storage
	\copyright	Hot Door, Inc. 2010-2025
*/

#ifndef __HDI_CORE_PATH_STYLE_MAP_PREF__
#define __HDI_CORE_PATH_STYLE_MAP_PREF__

#if defined(HDI_CORE_AIP_MODE)

#include "hdicorePathStyle.h"
#include "hdicorePrefData.h"

namespace hdi
{
	namespace core
	{
		/**
			\brief	Allows for storage of path style maps
		*/
		class PathStyleMapPref : public PrefData
		{
			public:
				/**
					\brief	Constructs an empty PathStyleMapPref object
					\author	GW
					\date	11/2015
					
					\note	To test if a PathStyleMapPref object is empty, call isEmpty() on it
					\note	Empty PathStyleMapPref objects do not relate to any actual preference data; they are designed to
							be "receivers" of some other PathStyleMapPref object via the overloaded assignment operator.
							Empty PathStyleMapPref objects are useless until such time (though it is safe to call any of
							their methods)
				*/
				PathStyleMapPref();
			
				/**
					\brief	Constructs a new PathStyleMapPref object from an existing PathStyleMapPref object (copy constructor)
					\author	GW
					\date	11/2015

					\param	ps_		Existing PathStyleMapPref object
				*/
				PathStyleMapPref(const PathStyleMapPref& ps_);

				/**
					\brief	Constructs a PathStyleMapPref object, with new data, to be added to a container at a later time
					\author	GW
					\date	11/2015

					\param	value_	Initial path style map value
					\param	name_	Name for the data once it has been added to its container, as UTF-8

					\note	The name_ argument can only be "" (omitted) if the object will be added directly to an
							ArrayPref object.
				*/
				PathStyleMapPref(const PathStyle::Map& value_, const std::string& name_ = "");

				/**
					\brief	Destructs a PathStyleMapPref object
					\author	GW
					\date	11/2015
				*/
				virtual ~PathStyleMapPref();

				/**
					\brief		Allows one PathStyleMapPref object to be assigned from another
					\author		GW
					\date		11/2015

					\param		rhs_	Righthand side of the = operator; the object to copy values from
					\returns	The target PathStyleMapPref object, but with its value updated to match that of rhs_
				*/
				virtual PathStyleMapPref& operator=(const PathStyleMapPref& rhs_);
				
				/**
					\brief		Convenience method to clone an PathStyleMapPref object on the heap
					\author		GW
					\date		11/2015

					\returns	A pointer to the new PathStyleMapPref object
					
					\note		If you subclass PathStyleMapPref, you MUST overload this method yourself! If you don't and/or
								your clone() method does not return an instance of your PathStyleMapPref subclass, you will
								experience "object slicing" when adding the pref to a container.

					\warning	The caller must manage the memory for the returned PathStyleMapPref object.
				*/
				virtual PathStyleMapPref* clone() const;

				/**
					\brief		Gets the type of persistent data
					\author		GW
					\date		11/2015

					\returns	PathStyleMapType

					\note		This is useful for knowing how to cast a PrefData object to its proper subclass
				*/
				virtual Type dataType() const;

				/**
					\brief		Gets the path style map value for this object
					\author		GW
					\date		11/2015

					\returns	The path style map value being described by this wrapper
				*/
				virtual PathStyle::Map pathStyleMap() const;

				/**
					\brief		Sets the path style map value for this object
					\author		GW
					\date		11/2015

					\param		value_	New path style map value
					\returns	true if the path style map is successfully updated, or false otherwise
				*/
				virtual bool setPathStyleMap(const PathStyle::Map& value_);
		};
		
		typedef std::unique_ptr<PathStyleMapPref> PathStyleMapPrefUP;
		typedef std::shared_ptr<PathStyleMapPref> PathStyleMapPrefSP;
		typedef std::weak_ptr<PathStyleMapPref> PathStyleMapPrefWP;
	}
}

#endif
// HDI_CORE_AIP_MODE

#endif
// __HDI_CORE_PATH_STYLE_MAP_PREF__
