/**
	\file
	\brief		Header file for general Illustrator placed art manipulation
	\copyright	Hot Door, Inc. 2010-2025
*/

#ifndef __HDI_CORE_PLACED_ART__
#define __HDI_CORE_PLACED_ART__

#if defined(HDI_CORE_AIP_MODE)

#include "hdicoreArtboardRect.h"
#include "hdicoreTransformMatrix.h"

namespace hdi
{
	namespace core
	{
		class Art;
	
		/**
			\brief	Allows for placed art manipulation, metadata access, etc.
		*/
		class PlacedArt
		{
			public:
				/**
					\brief	Describes what type of data is placed
				*/
				enum PlacedType
				{
					PlacedTypeUnknown	= 0,
					PlacedTypeEPS		= 10,
					PlacedTypeOther		= 20
				};
				
				/**
					\brief	Describes how the data has been placed
				*/
				enum PlaceMethod
				{
					PlaceMethodUnknown		= 0,
					
					// Preserve original dimensions regardless of bounding box size and transformations
					PlaceMethodAsIs			= 10,
					
					// Fill bounding box while preserving proportions. Can overlap edges in one dimension.
					PlaceMethodFill			= 20,
					
					// Fit fully inside bounding box while preserving proportions
					PlaceMethodFit			= 30,
					
					// Fit to bounding box; replaced file preserves bounds but not proportions
					PlaceMethodConform		= 40,
					
					// Fit to bounding box; replaced file preserves transformations and tries to preserve size
					PlaceMethodReconform	= 50
				};
			
				enum PlaceMode
				{
					// If the file can be read, it's simply placed
					PlaceModeVanilla				= 0,
					
					// Consult selection for whether to place or replace. If exactly one placed object is selected, show
					// "File.Place" dialog to query user on whether to simply place the new art, or use it to replace
					// the selection.
					PlaceModeQueryReplace			= 10,
					
					// Replaces the existing art
					PlaceModeForceReplace			= 20,
					
					// Replace placed art with its parsed contents
					PlaceModeUnlinkPlaced			= 30,
					
					// Unsupported by this library
//					PlaceModeCreateNewArt			= 40,
					
					// Like PlaceModeForceReplace, but validates the given file first
					PlaceModeForceReplaceValidate	= 50,
					
					// Replace placed art with its parsed contents, querying user for parameters
					PlaceModeQueryUnlinkPlaced		= 60
				};
			
				/**
					\brief	Destructs a PlacedArt object
					\author	GW
					\date	08/2014
				*/
				~PlacedArt();
			
				/**
					\brief		Creates a new piece of art of type ArtTypePlaced
					\author		GW
					\date		01/2016
					
					\param		mode_		Type of (re)placement operation to perform
					\param		path_		File path of the file to place; if empty, shows a dialog to query the user instead
					\param		link_		true to link the file, false to place/import the file
					\param		oldArt_		Art to be replaced, if mode_ is a replacement operation; pass NULL otherwise
					\param		newArt__	Return-by-reference for the newly created art
					\returns	true if the art was created, false otherwise
				*/
				static bool Create(
					const PlaceMode mode_,
					const std::string& path_,
					const bool link_,
					core::Art* const oldArt_,
					core::Art& newArt__
				);
				
				/**
					\brief		Gets the type of placed art
					\author		GW
					\date		08/2014
					
					\returns	The place art type
				*/
				PlacedType placedType() const;
				
				/**
					\brief		Gets the file path for the placed art
					\author		GW
					\date		08/2014
					
					\returns	The file path as UTF-8, or "" for error
				*/
				std::string filePath() const;
				
				/**
					\brief	Sets the file path for the placed art
					\author	GW
					\date	08/2014
					
					\param	path_	New file path, as UTF-8
				*/
				void setFilePath(const std::string& path_);
				
				/**
					\brief		Retrieves the transformation matrix that positions the content of a placed art object's
								linked file in the document
					\author		GW
					\date		08/2014
					
					\returns	The transformation matrix for the placed art
				*/
				core::TransformMatrix matrix() const;
				
				/**
					\brief	Sets the transformation matrix of the target placed art object
					\author	GW
					\date	08/2014
					
					\param	matrix_	New transformation matrix
				*/
				void setMatrix(const core::TransformMatrix& matrix_);
				
				/**
					\brief		Retrieves the boundary dimensions of placed art, regardless of the transformations
								applied to it
					\author		GW
					\date		08/2014
					
					\returns	The bounding rect
				*/
				core::ArtboardRect bounds() const;
				
				/**
					\brief	Sets the preliminary bounding box information for a placed object
					\author	GW
					\date	08/2014
					
					\param	b_	The new bounding box
				*/
				void setBounds(const core::ArtboardRect& b_);
				
				/**
					\brief		Retrieves the placement options for a placed object
					\author		GW
					\date		08/2014
					
					\param		method__	Return-by-reference for the method used to position and size the linked object
					\param		align__		Return-by-reference for the alignment of the linked object
					\param		clip__		Return-by-reference for true if the linked object is clipped
					\returns	true if the options could be acquired, false otherwise
				*/
				bool options(PlaceMethod& method__, BoundsPinPointType& align__, bool& clip__);

				/**
					\brief	Sets the placement options for a placed object
					\author	GW
					\date	08/2014
					
					\param	method_	New method used to position and size the linked object
					\param	align_	New alignment of the linked object
					\param	clip_	true if the linked object should be clipped
				*/
				void setOptions(const PlaceMethod method_, const BoundsPinPointType align_, const bool clip_);


			private:
				// Only Art can construct a new PlacedArt object
				friend class Art;

				/**
					\brief	Art to which this mesh belongs
				*/
				void* __data;
				
				/**
					\brief	Empty PlacedArt objects cannot be constructed
					\author	GW
					\date	08/2014
				*/
				PlacedArt();

				/**
					\brief	Constructs a new PlacedArt object from a given Art object
					\author	GW
					\date	08/2014

					\param	art_	Pointer for the existing art we're wrapping around
				*/
				PlacedArt(void* const art_);
			
				/**
					\brief	PlacedArt objects cannot be copied
					\author	GW
					\date	08/2014
				*/
				PlacedArt(const PlacedArt&);
			
				/**
					\brief	PlacedArt objects cannot be copied
					\author	GW
					\date	08/2013
				*/
				PlacedArt& operator=(const PlacedArt&);
		};
	}
}

#endif
// HDI_CORE_AIP_MODE

#endif
// __HDI_CORE_PLACED_ART__
