/**
	\file
	\brief		Header file for plugin user interface slider widgets
	\copyright	Hot Door, Inc. 2010-2025
*/

#ifndef __HDI_CORE_SLIDER__
#define __HDI_CORE_SLIDER__

#include "hdicoreWidget.h"

namespace hdi
{
	namespace core
	{
		class Callback;

		/**
			\brief	Slider widget, which has a sliding interface to allow for arbitrary value selections from 0.0 to 1.0
		*/
		class Slider : public Widget
		{
			public:
				/**
					\brief	Constructs an empty Slider object
					\author	GW
					\date	09/2013
					
					\note	To test if a Slider object is empty, call isEmpty() on it
					\note	Empty Slider objects do not relate to any actual UI widget; they are designed to be
							"receivers" of some other Slider object via the overloaded assignment operator. Empty
							Slider objects are useless until such time (though it is safe to call any of their methods).
				*/
				Slider();

				/**
					\brief	Constructs a new Slider object from an existing Slider object (copy constructor)
					\author	GW
					\date	09/2013

					\param	slider_		Existing Slider object
				*/
				Slider(const Slider& slider_);

				/**
					\brief	Constructs a slider
					\author	GW
					\date	09/2013

					\param	loc_	Position of the slider, in 1x resolution coordinates
					\param	value_	Initial value of the Slider in the range [0.0, 1.0]
					\param	width_	Width of the slider, at 1x resolution 
				*/
				Slider(const Point& loc_, const double value_, const double width_);
				
				/**
					\brief	Slider destructor
					\author	GW
					\date	09/2013
				*/
				virtual ~Slider();

				/**
					\brief		Allows one Slider object to be assigned from another
					\author		GW
					\date		10/2013

					\param		rhs_	Righthand side of the = operator; the object to copy values from
					\returns	The target Slider object, but with its value updated to match that of rhs_
				*/
				virtual Slider& operator=(const Slider& rhs_);
				
				/**
					\brief		Convenience method to clone a Slider object on the heap
					\author		GW
					\date		10/2013

					\returns	A pointer to the new Slider object
					
					\note		If you subclass Slider, you MUST overload this method yourself! If you don't and/or your
								clone() method does not return an instance of your Slider subclass, you will experience
								"object slicing" when adding the widget to a window.
								
					\warning	The caller must manage the memory for the returned Slider object.
				*/
				virtual Slider* clone() const;
				
				/**
					\brief		Convenience method to duplicate a Slider object, creating a new and identical UI element
								to the target (but not belonging to the same containing window)
					\author		GW
					\date		11/2013
					
					\returns	A pointer to the new Slider object (and new UI element)
					
					\warning	The caller must manage the memory for the returned Slider object.
				*/
				virtual Slider* duplicate() const;

				/**
					\brief		Gets the current value of the slider
					\author		GW
					\date		09/2013

					\returns	The current value of the target Slider object, in range [0.0, 1.0]
				*/
				virtual double value() const;

				/**
					\brief	Sets the current value of the slider
					\author	GW
					\date	09/2013

					\param	value_	New slider value in the range [0.0, 1.0]

					\note	The provided value will be forced within the range [0.0, 1.0]
				*/
				virtual void setValue(const double value_);

				/**
					\brief	Simulates the value of the slider changing
					\author	GW
					\date	09/2013
				*/
				virtual void valueChanging();
				
				/**
					\brief		Gets the value changing callback for the slider
					\author		GW
					\date		09/2013

					\returns	A pointer to the currently registered value changing callback
				*/
				virtual Callback* const valueChangingCallback() const;
				
				/**
					\brief	Sets the value changing callback
					\author	GW
					\date	09/2013

					\param	callback_	New callback for when the slider's value is changing
				*/
				virtual void setValueChangingCallback(const Callback& callback_);
		};
		
		typedef std::unique_ptr<Slider> SliderUP;
		typedef std::shared_ptr<Slider> SliderSP;
		typedef std::weak_ptr<Slider> SliderWP;
	}
}

#endif
// __HDI_CORE_SLIDER__
