/**
	\file
	\brief		Header file for encapsulating suite information in a convenient struct
	\copyright	Hot Door, Inc. 2010-2025
*/

#ifndef __HDI_CORE_SUITEINFO__
#define __HDI_CORE_SUITEINFO__

#include "hdicoreTypes.h"

namespace hdi
{
	namespace core
	{
		/**
			\brief	Stores basic information about an application suite
		*/
		struct SuiteInfo
		{
			/**
				\brief	Stores the name of the suite
			*/
			const std::string name;
			
			/**
				\brief	Stores the internal version number of the suite
			*/
			const int32_t internalVersion;
			
			/**
				\brief	Stores the external (public) version number of the suite (what would be used to acquire
						the suite)
			*/
			const int32_t externalVersion;
			
			/**
				\brief	Constructs a new SuiteInfo object with default values
				\author	GW
				\date	04/2014
			*/
			SuiteInfo();
			
			/**
				\brief	Constructs a new SuiteInfo object from an existing object (copy ctor)
				\author	GW
				\date	04/2014
				
				\param	si_		Existing SuiteInfo object to copy values from
			*/
			SuiteInfo(const SuiteInfo& si_);
								
			/**
				\brief	Constructs a new SuiteInfo object with the given values
				\author	GW
				\date	04/2014
				
				\param	name_		Name of the suite
				\param	intVers_	Internal version number of the suite
				\param	extVers_	External version number of the suite
			*/
			SuiteInfo(const std::string& name_, const int32_t intVers_, const int32_t extVers_);
			
			/**
				\brief	Destructs a SuiteInfo object
				\author	GW
				\date	04/2014
			*/
			virtual ~SuiteInfo();
			
			/**
				\brief		Compares one SuiteInfo object to another to check for equality
				\author		GW
				\date		04/2014
				
				\param		rhs_	Righthand side of the equality operator
				\returns	true if the objects have equal values, false otherwise
			*/
			bool operator==(const SuiteInfo& rhs_) const;
								
			/**
				\brief		Compares one SuiteInfo object to another to check for inequality
				\author		GW
				\date		04/2014
				
				\param		rhs_	Righthand side of the inequality operator
				\returns	true if the objects have unequal values, false otherwise
			*/
			bool operator!=(const SuiteInfo& rhs_) const;
		};

		typedef std::unique_ptr<SuiteInfo> SuiteInfoUP;
		typedef std::shared_ptr<SuiteInfo> SuiteInfoSP;
		typedef std::weak_ptr<SuiteInfo> SuiteInfoWP;
	}
}

#endif
// __HDI_CORE_SUITEINFO__
