/**
	\file
	\brief		Header file for general Illustrator symbol manipulation
	\copyright	Hot Door, Inc. 2010-2025
*/

#ifndef __HDI_CORE_SYMBOL__
#define __HDI_CORE_SYMBOL__

#if defined(HDI_CORE_AIP_MODE)

#include "hdicorePattern.h"

namespace hdi
{
	namespace aip
	{
		class Symbol;
	}

	namespace core
	{
		/**
			\brief	Handles general symbol-related functionality
		*/
		class Symbol : public Pattern
		{
			public:
				/**
					\brief	Constructs an empty Symbol object
					\author	GW
					\date	11/2013
					
					\note	To test if a Symbol object is empty, call isEmpty() on it
					\note	Empty Symbol objects do not relate to any symbol in a Illustrator document; they are
							designed to be "receivers" of some other Symbol object via the overloaded assignment
							operator. Empty Symbol objects are useless until such time (though it is safe to call any
							of their methods).
				*/
				Symbol();
			
				/**
					\brief	Constructs a new Symbol object from an existing Symbol object (copy constructor)
					\author	GW
					\date	11/2013

					\param	sym_	Existing Symbol object
				*/
				Symbol(const Symbol& sym_);
				
				/**
					\brief		Creates a new Symbol object (and Illustrator symbol)
					\author		GW
					\date		11/2013
					
					\param		src_	Source art for the symbol definition
					\returns	A new Symbol object
				*/
				static Symbol create(const Art& src_);

				/**
					\brief	Symbol object destructor
					\author	GW
					\date	11/2013
				*/
				virtual ~Symbol();
				
				/**
					\brief		Assigns one Symbol object to another
					\author		GW
					\date		11/2013

					\param		rhs_	Existing Symbol object to copy values from
					\returns	The target Symbol object, but with its values updated to match that of the rhs_ argument
				*/
				virtual Symbol& operator=(const Symbol& rhs_);

				/**
					\brief		Removes the symbol from its document
					\author		GW
					\date		11/2013
					
					\returns	true for successful disposal, false otherwise

					\note		After calling dispose(), this object is basically useless! The actual symbol is gone,
								so obviously none of these methods will work after that.
				*/
				virtual bool dispose();
				
				/**
					\brief		Tests the validity of the target symbol by searching for it in the document
					\author		GW
					\date		11/2013
					
					\returns	true for a valid symbol, false otherwise
				*/
				virtual bool valid() const;
				
				/**
					\brief		Tests whether a given Symbol object is the same as another
					\author		GW
					\date		11/2013

					\param		rhs_	Symbol to compare against (righthand side of equality operator)
					\returns	true for the target and rhs_ being the same symbol, false otherwise
				*/
				virtual bool operator==(const Symbol& rhs_) const;
				
				/**
					\brief		Tests whether a given Symbol object is not the same as another
					\author		GW
					\date		11/2013

					\param		rhs_	Symbol to compare against (righthand side of inequality operator)
					\returns	true for the target and rhs_ being different symbols, false otherwise
				*/
				virtual bool operator!=(const Symbol& rhs_) const;
				
				/**
					\brief		Gets the Art object that defines the source art for the symbol
					\author		GW
					\date		11/2013
					
					\returns	An Art object for the source definition of the symbol
				*/
				virtual std::unique_ptr<Art> sourceArt() const;
				
				/**
					\brief	Sets the Art object that defines the source art for the symbol
					\author	GW
					\date	11/2013
					
					\param	art_	New source art for the target Symbol object
				*/
				virtual void setSourceArt(const Art& art_);

				/**
					\brief		Gets whether the target symbol is currently being edited in symbol editing mode
					\author		GW
					\date		11/2015
					
					\returns	true if currently being edited, false otherwise
				*/
				virtual bool isBeingEdited() const;

				/**
					\brief		Puts the document into symbol editing mode, with the target symbol being the one to edit
					\author		GW
					\date		11/2015
					
					\param		inst_	An optional instance of the pattern or symbol
					\returns	true if symbol editing mode was entered successfully, false otherwise
					
					\note		Illustrator will automatically create a prototype (i.e. temporary copy) of the symbol
								source art for in-place editing.
					\note		If inst_ is NULL, then editing will take place in the center of the artboard. Otherwise,
								editing will take place wherever the instance is located.
				*/
				virtual bool setBeingEdited(const Art* const inst_ = NULL);
			
				/**
					\brief		Creates a new instance of the symbol in the current document
					\author		GW
					\date		11/2013
					
					\param		center_		Center point at which to place the instance
					\param		order_		Order in which to create the art
					\param		prep_		Prepositional piece of art to which order_ will apply
					\returns	The newly created symbol art instance
					
					\warning	This method does not assume responsibility of the memory for the prep_ argument.
				*/
				virtual std::unique_ptr<Art> newInstance(
					const ArtboardPoint& center_,
					const PaintOrder order_,
					const Art* const prep_ = NULL
				) const;


			private:
				friend aip::Symbol* __accessImpl(const Symbol&);
				friend Symbol __accessCtor(aip::Symbol*&);

				/**
					\brief	Internal use only
					\author	GW
					\date	11/2013
				*/
				Symbol(aip::Symbol*&);
		};
		
		typedef std::unique_ptr<Symbol> SymbolUP;
		typedef std::shared_ptr<Symbol> SymbolSP;
		typedef std::weak_ptr<Symbol> SymbolWP;
		
		extern aip::Symbol* __accessImpl(const Symbol&);
		extern Symbol __accessCtor(aip::Symbol*&);
	}
}

#endif
// HDI_CORE_AIP_MODE

#endif
// __HDI_CORE_SYMBOL__
