/**
	\file
	\brief		Header file for a class to aid in widget tab-key ordering
	\copyright	Hot Door, Inc. 2010-2025
*/

#ifndef __HDI_CORE_TAB_ORDER__
#define __HDI_CORE_TAB_ORDER__

#include <list>

#include "hdicoreTypes.h"

namespace hdi
{
	namespace core
	{
		class Widget;

		/**
			\brief	Allows for finer control of widget focus/tab key ordering
		*/
		class TabOrder
		{
			public:
				typedef std::list< std::shared_ptr<Widget> > WidgetList;
			
				/**
					\brief		Tests whether a given TabOrder object is the same as another
					\author		GW
					\date		09/2013

					\param		rhs_	TabOrder to compare against (righthand side of equality operator)
					\returns	true for the target and rhs_ being the same UI widget, false otherwise
				*/
				virtual bool operator==(const TabOrder& rhs_) const;

				/**
					\brief		Tests whether a given TabOrder object is not the same as another
					\author		GW
					\date		09/2013

					\param		rhs_	TabOrder to compare against (righthand side of inequality operator)
					\returns	true for the target and rhs_ being different UI widgets, false otherwise
				*/
				virtual bool operator!=(const TabOrder& rhs_) const;
				
				/**
					\brief		Checks whether the target TabOrder object already contains a given widget
					\author		GW
					\date		09/2013

					\param		widget_		The widget to check for existence in the TabOrder object
					\returns	true if the target contains the given widget, false otherwise
				*/
				virtual bool hasWidget(const Widget& widget_) const;

				/**
					\brief		Gets the list of widgets affected by the target TabOrder object
					\author		GW
					\date		09/2013

					\returns	A list containing all widgets affected by the tab order
				*/
				virtual const WidgetList widgets() const;

				/**
					\brief		Gets the Widget object that is at the front of the target TabOrder object
					\author		GW
					\date		09/2013

					\returns	The frontmost Widget object, or NULL if none
				*/
				virtual std::shared_ptr<Widget> front() const;

				/**
					\brief		Pushes a new Widget object on the front of the list for the target TabOrder
					\author		GW
					\date		09/2013

					\param		widget_		The widget to add to the front of the tab order
				*/
				virtual void pushFront(const Widget& widget_);

				/**
					\brief		Gets the Widget at the front of the target TabOrder, and pops it
					\author		GW
					\date		09/2013

					\param		widget__	Return-by-reference for the frontmost Widget object
					\returns	true if the frontmost widget was popped and acquired, false otherwise
				*/
				virtual bool popFront(std::unique_ptr<Widget>& widget__);

				/**
					\brief		Gets the Widget object that is at the back of the target TabOrder object
					\author		GW
					\date		09/2013

					\returns	The backmost Widget object, or NULL if none
				*/
				virtual std::shared_ptr<Widget> back() const;

				/**
					\brief		Pushes a new Widget object on the back of the list for the target TabOrder
					\author		GW
					\date		09/2013

					\param		widget_		The widget to add to the back of the tab order
				*/
				virtual void pushBack(const Widget& widget_);

				/**
					\brief		Gets the Widget at the back of the target TabOrder, and pops it
					\author		GW
					\date		09/2013

					\param		widget__	Return-by-reference for the Widget object at the back
					\returns	true if the widget at the back was popped and acquired, false otherwise
				*/
				virtual bool popBack(std::unique_ptr<Widget>& widget__);

				/**
					\brief		Focuses the next (enabled) widget after the one provided
					\author		GW
					\date		09/2013

					\param		position_	Widget immediately before the one that should be focused (typically, the
											currently focused widget is passed)
				*/
				virtual void focusNext(const Widget& position_);

				/**
					\brief		Focuses the previous (enabled) widget before the one provided
					\author		GW
					\date		09/2013

					\param		position_	Widget immediately after the one that should be focused (typically, the
											currently focused widget is passed)
				*/
				virtual void focusPrevious(const Widget& position_);

				/**
					\brief		Gets whether the target TabOrder object should fire a value-changed callback on a widget
								if a) the user presses tab, b) the currently focused widget is the only one than can
								be focused in the group, and c) the currently focused widget support value-changed events
					\author		GW
					\date		04/2023

					\returns	true if the target should fire value-changed callbacks in the above condition

					\note		The default for all new TabOrder objects is true
				*/
				bool fireValueChangedIfOnlyOneAvailable() const;

				/**
					\brief	Sets whether the target TabOrder object should fire a value-changed callback on a widget
							if a) the user presses tab, b) the currently focused widget is the only one than can
							be focused in the group, and c) the currently focused widget support value-changed events
					\author	GW
					\date	04/2023

					\param	fire_	true if the target should fire value-changed callbacks in the above condition, false
									otherwise

					\note	The default for all new TabOrder objects is true
				*/
				void setFireValueChangedIfOnlyOneAvailable(const bool fire_);


			private:
				friend class Panel;
				friend class Window;
			
				/**
					\brief	Private implementation data
				*/
				void* __data;

				/**
					\brief	Internal use only
					\author	GW
					\date	09/2013
				*/
				TabOrder();
				
				/**
					\brief	Internal use only
					\author	GW
					\date	02/2015
				*/
				TabOrder(const TabOrder&);

				/**
					\brief	Internal use only
					\author	GW
					\date	09/2013
				*/
				virtual ~TabOrder();

				/**
					\brief		Internal use only
					\author		GW
					\date		09/2013
				*/
				virtual TabOrder& operator=(const TabOrder&);
		};
	}
}

#endif
// __HDI_CORE_TAB_ORDER__
