/**
	\file
	\brief		Header file for general Illustrator text glyph run manipulation
	\copyright	Hot Door, Inc. 2010-2025
*/

#ifndef __HDI_CORE_TEXT_GLYPH_RUN__
#define __HDI_CORE_TEXT_GLYPH_RUN__

#if defined(HDI_CORE_AIP_MODE)

#include "hdicoreTypes.h"

namespace hdi
{
	namespace aip
	{
		class TextGlyphRun;
	}
	
	namespace core
	{
		class ArtboardPoint;
		class Font;
		class TextCharFeatures;
		class TextLine;
	
		class TextGlyphRun
		{
			public:
				/**
					\brief	Creates a new TextGlyphRun object
					\author	GW
					\date	08/2014
					
					\note	New TextGlyphRun objects do not relate to any text glyph run on the Illustrator artboard;
							they are designed to be "receivers" of some other TextGlyphRun object via the overloaded
							assignment operator or assigned to some other existent text art.
				*/
				TextGlyphRun();
			
				/**
					\brief	TextGlyphRun copy constructor
					\author	GW
					\date	08/2014
					
					\param	gr_	TextGlyphRun object to copy values from
				*/
				TextGlyphRun(const TextGlyphRun& gr_);

				/**
					\brief	Destructs a TextGlyphRun object
					\author	GW
					\date	08/2014
				*/
				virtual ~TextGlyphRun();
			
				/**
					\brief	Overloaded assignment operator for TextGlyphRun objects
					\author	GW
					\date	08/2014
				*/
				virtual TextGlyphRun& operator=(const TextGlyphRun& rhs_);
				
				/**
					\brief		Tests whether a given TextGlyphRun object is the same as another
					\author		GW
					\date		08/2014

					\param		rhs_	TextGlyphRun to compare against (righthand side of equality operator)
					\returns	true for the target and rhs_ being the same text glyph run, false otherwise
				*/
				virtual bool operator==(const TextGlyphRun& rhs_) const;
				
				/**
					\brief		Tests whether a given TextGlyphRun object is not the same as another
					\author		GW
					\date		08/2014

					\param		rhs_	TextGlyphRun to compare against (righthand side of inequality operator)
					\returns	true for the target and rhs_ being different text glyph run, false otherwise
				*/
				virtual bool operator!=(const TextGlyphRun& rhs_) const;

				/**
					\brief		Gets the ATE glyph run ref around which the target object is wrapped
					\author		GW
					\date		08/2014
					
					\returns	ATE::IGlyphRunRef for the target object

					\note		Generally, you shouldn't need to call this method. Only use it if you know what you're
								doing. If a specific piece of functionality provided by Illustrator is not handled by
								this class (or related classes), then it should probably be added to the hdi_core library.
				*/
				virtual std::unique_ptr<ATE::IGlyphRun> ateGlyphRun() const;

				/**
					\brief		Gets whether the target TextGlyphRun object is empty
					\author		GW
					\date		08/2014
					
					\returns	true if the target TextGlyphRun object is empty, false otherwise
				*/
				virtual bool isEmpty() const;
				
				/**
					\brief		Retrieves the text-line parent of this glyph run
					\author		GW
					\date		08/2014

					\returns	The text-line object
				*/
				virtual std::unique_ptr<TextLine> textLine() const;

				/**
					\brief		Retrieves the number of glyphs in this run
					\author		GW
					\date		08/2014

					\returns	The number of glyphs, or -1 for error
					
					\note		The number of the glyphs can be different from number of characters returned by
								charCount(), because of factors such as ligature and hyphenation.
				*/
				virtual int32_t glyphCount() const;

				/**
					\brief		Retrieves the number of characters in this run
					\author		GW
					\date		08/2014

					\returns	The number of characters, or -1 for error
				*/
				virtual int32_t charCount() const;
				
				/**
					\brief		Retrieves the count of origin points in this run
					\author		GW
					\date		08/2014

					\returns	The count of origin points, or -1 for error
				*/
				virtual int32_t originsCount() const;
				
				/**
					\brief		Retrieves an origin point for a glyph in this run
					\author		GW
					\date		08/2014

					\param		index_	Index of the glyph in question
					\param		pt__	Return-by-reference for the glyph origin point
					\returns	true if the origin point could be acquired, false otherwise
				*/
				virtual bool origin(const int32_t index_, ArtboardPoint& pt__) const;

				/**
					\brief		Retrieves the count of glyph identifiers in this run
					\author		GW
					\date		08/2014

					\returns	The count of glyph IDs, or -1 for error
				*/
				virtual int32_t glyphIDCount() const;
				
				/**
					\brief		Retrieves a glyph identifier for a glyph in this run
					\author		GW
					\date		08/2014

					\param		index_	Index of the glyph in question
					\param		id__	Return-by-reference for the glyph ID
					\returns	true if the ID could be acquired, false otherwise
				*/
				virtual bool glyphID(const int32_t index_, int32_t& id__) const;

				/**
					\brief		Retrieves the character features of this run
					\author		GW
					\date		08/2014

					\returns	The character features object
					
					\note		Only the following features will be defined: Font, FontSize, HorizontalScale,
								VerticalScale, Tracking, BaselineShift, FillColor, StrokeColor, fBlend, Fill, Stroke,
								FillFirst, FillOverPrint, StrokeOverPrint, FillBackgroundColor, FillBackground, LineCap,
								LineJoin, LineWidth, MiterLimit, LineDashOffset, and LineDashArray.
				*/
				virtual std::unique_ptr<TextCharFeatures> charFeatures() const;

				/**
					\brief		Retrieves the glyph orientation of this run
					\author		GW
					\date		08/2014

					\returns	The orientation enum value
				*/
				virtual GlyphOrientation glyphOrientation() const;

				/**
					\brief		Retrieves the ascent of this run
					\author		GW
					\date		08/2014

					\returns	The ascent value, in document points
				*/
				virtual double ascent() const;
				
				/**
					\brief		Retrieves the descent of this run
					\author		GW
					\date		08/2014

					\returns	The descent value, in document points
				*/
				virtual double descent() const;

				/**
					\brief		Retrieves the tracking (space between each character) of this run
					\author		GW
					\date		08/2014

					\returns	The tracking value, in document points
				*/
				virtual double tracking() const;
				
				/**
					\brief		Retrieves the width of the space glyph in the font for this run
					\author		GW
					\date		08/2014

					\returns	The width value, in document points
				*/
				virtual double spaceGlyphWidth() const;

				/**
					\brief		Retrieves the distance to the baseline in the font for this run
					\author		GW
					\date		08/2014

					\returns	The distance value, in document points
				*/
				virtual double distToBaseline() const;

				/**
					\brief		Retrieves the underline position in the font for this run
					\author		GW
					\date		08/2014

					\returns	The position value, in document points
				*/
				virtual double underlinePosition() const;

				/**
					\brief		Retrieves the underline thickness in the font for this run
					\author		GW
					\date		08/2014

					\returns	The thickness value, in document points
				*/
				virtual double underlineThickness() const;

				/**
					\brief		Retrieves the minimum height for capital letters in the font for this run
					\author		GW
					\date		08/2014

					\returns	The height value, in document points
				*/
				virtual double minCapHeight() const;
				
				/**
					\brief		Retrieves the maximum height for capital letters in the font for this run
					\author		GW
					\date		08/2014

					\returns	The height value, in document points
				*/
				virtual double maxCapHeight() const;

				/**
					\brief		Retrieves the component font for this run if the character-feature font is a composite
								font, otherwise retrieves the character-feature font
					\author		GW
					\date		08/2014

					\returns	The font object
				*/
				virtual std::unique_ptr<Font> flattenedFont() const;

				/**
					\brief		Gets the value (contents) of the glyph run
					\author		GW
					\date		08/2014
					
					\param		le_		Indicates the desired line ending for the output string
					\returns	A string for the value of the text art, as UTF-8
					
					\note		Illustrator forces text art to utilize the "\r" character for line endings; if this
								behavior is unexpected or undesired for your application (as most systems use either
								"\n" or "\r\n" these days), you can specify what you'd prefer with the optional le_
								argument.
				*/
				virtual std::string contents(const std::string& le_ = "\n") const;


			private:
				friend aip::TextGlyphRun* __accessImpl(const TextGlyphRun&);
				friend TextGlyphRun __accessCtor(const aip::TextGlyphRun&);

				/**
					\brief	Private implementation object
				*/
				aip::TextGlyphRun* __impl;

				/**
					\brief	Internal use only
					\author	GW
					\date	08/2014
				*/
				TextGlyphRun(const aip::TextGlyphRun&);
		};
		
		typedef std::unique_ptr<TextGlyphRun> TextGlyphRunUP;
		typedef std::shared_ptr<TextGlyphRun> TextGlyphRunSP;
		typedef std::weak_ptr<TextGlyphRun> TextGlyphRunWP;
		
		extern aip::TextGlyphRun* __accessImpl(const TextGlyphRun&);
		extern TextGlyphRun __accessCtor(const aip::TextGlyphRun&);
	}
}

#endif
// HDI_CORE_AIP_MODE

#endif
// __HDI_CORE_TEXT_CHAR_STYLE__
