/**
	\file
	\brief		Header file for general Illustrator story set manipulation
	\copyright	Hot Door, Inc. 2010-2025
*/

#ifndef __HDI_CORE_TEXT_STORY_SET__
#define __HDI_CORE_TEXT_STORY_SET__

#if defined(HDI_CORE_AIP_MODE)

#include <vector>

#include "hdicoreTypes.h"

namespace hdi
{
	namespace aip
	{
		class TextStorySet;
	}
	
	namespace core
	{
		class TextPara;
		class TextRange;
		class TextRangeSet;
		class TextStory;
		class TextWord;
	
		class TextStorySet
		{
			public:
				typedef std::vector< std::shared_ptr<TextPara> > TextParaVector;
				typedef std::vector< std::shared_ptr<TextRange> > TextRangeVector;
				typedef std::vector< std::shared_ptr<TextWord> > TextWordVector;
				
				/**
					\brief	Creates a new TextStorySet object
					\author	GW
					\date	08/2014
					
					\note	New TextStorySet objects do not relate to any text story set on the Illustrator artboard;
							they are designed to be "receivers" of some other TextStorySet object via the overloaded
							assignment operator or assigned to some other existent text art.
				*/
				TextStorySet();
			
				/**
					\brief	TextStorySet copy constructor
					\author	GW
					\date	08/2014
					
					\param	ss_	TextStorySet object to copy values from
				*/
				TextStorySet(const TextStorySet& ss_);

				/**
					\brief	Destructs a TextStorySet object
					\author	GW
					\date	08/2014
				*/
				virtual ~TextStorySet();
			
				/**
					\brief	Overloaded assignment operator for TextStorySet objects
					\author	GW
					\date	08/2014
				*/
				virtual TextStorySet& operator=(const TextStorySet& rhs_);
				
				/**
					\brief		Tests whether a given TextStorySet object is the same as another
					\author		GW
					\date		08/2014

					\param		rhs_	TextStorySet to compare against (righthand side of equality operator)
					\returns	true for the target and rhs_ being the same text story set, false otherwise
				*/
				virtual bool operator==(const TextStorySet& rhs_) const;
				
				/**
					\brief		Tests whether a given TextStorySet object is not the same as another
					\author		GW
					\date		08/2014

					\param		rhs_	TextStorySet to compare against (righthand side of inequality operator)
					\returns	true for the target and rhs_ being different text story sets, false otherwise
				*/
				virtual bool operator!=(const TextStorySet& rhs_) const;

				/**
					\brief		Gets the ATE story set ref around which the target object is wrapped
					\author		GW
					\date		08/2014
					
					\returns	ATE::IStoriesRef for the target object

					\note		Generally, you shouldn't need to call this method. Only use it if you know what you're
								doing. If a specific piece of functionality provided by Illustrator is not handled by
								this class (or related classes), then it should probably be added to the hdi_core library.
				*/
				virtual std::unique_ptr<ATE::IStories> ateStorySet() const;

				/**
					\brief		Gets whether the target TextStorySet object is empty
					\author		GW
					\date		08/2014
					
					\returns	true if the target TextStorySet object is empty, false otherwise
				*/
				virtual bool isEmpty() const;
				
				/**
					\brief		Retrieves the number of members of this set
					\author		GW
					\date		08/2014
					
					\returns	The number of members, or -1 for error
				*/
				virtual int32_t size() const;
				
				/**
					\brief		Reports whether a member story has the input focus
					\author		GW
					\date		08/2014
					
					\returns	true if a story in this collection is currently being edited

					\note		Only one story can have the focus. It can lose focus if the user selects text outside it, or clicks outside the text of the story.
				*/
				virtual bool hasFocus() const;
				
				/**
					\brief		Retrieves the member story that has input focus
					\author		GW
					\date		08/2014
					
					\returns	The story currently being edited, or a NULL object if no text is being edited
				*/
				virtual std::unique_ptr<TextStory> focusedStory() const;
				
				/**
					\brief	Sets the input focus to a member story, deselecting everything in the document
					\author	GW
					\date	08/2014
					
					\param	story_	Story to focus
				*/
				virtual void setFocusedStory(const TextStory& story_);
				
				/**
					\brief	Removes input focus from the story currently being edited, deselecting everything in the document
					\author	GW
					\date	08/2014
				*/
				virtual void loseFocus();
				
				/**
					\brief		Retrieves the set of text ranges in these stories
					\author		GW
					\date		08/2014
					
					\returns	The text range set object
				*/
				virtual std::unique_ptr<TextRangeSet> textRanges() const;
								
				/**
					\brief		Retrieves the set of text ranges for selected text in these stories
					\author		GW
					\date		08/2014
					
					\returns	The text range set object
				*/
				virtual std::unique_ptr<TextRangeSet> selections() const;
				
				/**
					\brief		Retrieves a vector of paragraphs contained by the member stories
					\author		GW
					\date		08/2014
					
					\returns	The paragraph vector object
				*/
				virtual TextParaVector paragraphs() const;
				
				/**
					\brief		Retrieves a vector of words contained by the member stories
					\author		GW
					\date		08/2014
					
					\returns	The words vector object
				*/
				virtual TextWordVector words() const;
				
				/**
					\brief		Retrieves a vector of runs (ranges) contained by the member stories
					\author		GW
					\date		08/2014
					
					\returns	The ranges vector object
				*/
				virtual TextRangeVector runs() const;
				
				/**
					\brief		Retrieves a member of this set by position index
					\author		GW
					\date		08/2014
					
					\param		index_	The 0-based position index
					\returns	The member story, or NULL for error
				*/
				virtual std::unique_ptr<TextStory> story(const int32_t index_) const;
				
				/**
					\brief	Suspends reflow calculation for the target stories. Speeds up calls that cause reflow, such as
							TextRange insertion methods
					\author	GW
					\date	08/2014
					
					\note	Use resumeReflow() to restore normal reflow calculation
				*/
				virtual void suspendReflow();

				/**
					\brief	Resumes normal reflow calculation after a call to suspendReflow()
					\author	GW
					\date	08/2014
				*/
				virtual void resumeReflow();


			private:
				friend aip::TextStorySet* __accessImpl(const TextStorySet&);
				friend TextStorySet __accessCtor(const aip::TextStorySet&);

				/**
					\brief	Private implementation object
				*/
				aip::TextStorySet* __impl;

				/**
					\brief	Internal use only
					\author	GW
					\date	08/2014
				*/
				TextStorySet(const aip::TextStorySet&);
		};
		
		typedef std::unique_ptr<TextStorySet> TextStorySetUP;
		typedef std::shared_ptr<TextStorySet> TextStorySetSP;
		typedef std::weak_ptr<TextStorySet> TextStorySetWP;
		
		extern aip::TextStorySet* __accessImpl(const TextStorySet&);
		extern TextStorySet __accessCtor(const aip::TextStorySet&);
	}
}

#endif
// HDI_CORE_AIP_MODE

#endif
// __HDI_CORE_TEXT_STORY_SET__
