/**
	\file
	\brief		Header file for Illustrator timer manipulation
	\copyright	Hot Door, Inc. 2010-2025
*/

#ifndef __HDI_CORE_TIMER__
#define __HDI_CORE_TIMER__

#include "hdicoreTypes.h"
	
namespace hdi
{
	#if defined(HDI_CORE_AIP_MODE)
		namespace aip
		{
			class Timer;
		}
	#endif
	// HDI_CORE_AIP_MODE

	namespace core
	{
		class Callback;
		
		/**
			\brief		Deals with timer registration and callbacks
			\details	Timers will fire over and over at the set delay until deactivate() is called (though they can
						be fired again later after activate() is called).
		*/
		class Timer
		{
			public:
				/**
					\brief	Constructs an empty Timer object
					\author	GW
					\date	08/2013
					
					\note	To test if a Timer object is empty, call isEmpty() on it
					\note	Empty Timer objects do not relate to any Illustrator timer; they are designed to be
							"receivers" of some other Timer object via the overloaded assignment operator. Empty
							Timer objects are useless until such time (though it is safe to call any of their
							methods).
				*/
				Timer();
			
				/**
					\brief	Constructs a new Timer object from an existing Timer object (copy constructor)
					\author	GW
					\date	08/2013

					\param	t_	Existing Timer object
				*/
				Timer(const Timer& t_);

				/**
					\brief	Constructs a Timer object to fire a responder (target and action) in a given number of
							seconds
					\author	GW
					\date	08/2013

					\param	name_		Unique name for the timer, as UTF-8
					\param	delay_		Delay in seconds before the responder should fire
					\param	responder_	Target and action to fire after delay_ seconds
				*/
				Timer(const std::string& name_, const double delay_, const Callback& responder_);

				/**
					\brief	Destructs a Timer object, deactivating the timer first
					\author	GW
					\date	08/2013
				*/
				virtual ~Timer();
			
				/**
					\brief		Assigns one Timer object to another
					\author		GW
					\date		08/2013

					\param		rhs_	Existing Timer object to copy values from
					\returns	The target Timer object, but with its values updated to match that of the rhs_
								argument
				*/
				virtual Timer& operator=(const Timer& rhs_);
				
				/**
					\brief		Tests whether a given Timer object is the same as another
					\author		GW
					\date		08/2013

					\param		rhs_	Timer to compare against (righthand side of equality operator)
					\returns	true for the target and rhs_ being the same timer, false otherwise
				*/
				virtual bool operator==(const Timer& rhs_) const;
				
				/**
					\brief		Tests whether a given Timer object is not the same as another
					\author		GW
					\date		08/2013

					\param		rhs_	Timer to compare against (righthand side of inequality operator)
					\returns	true for the target and rhs_ being different timers, false otherwise
				*/
				virtual bool operator!=(const Timer& rhs_) const;

				/**
					\brief	Destroys the timer, converting the target object to an empty Timer object (see the default
							constructor for more info on empty Timer objects)
					\author	GW
					\date	03/2014

					\pre	If you have registered the timer with Dispatcher::registerTimer(), then be sure to
							unregister the timer before calling the destroy() method.

					\note	This method must be called for any existing timers by the time the plugin is shutting
							down. If you no longer retain any of your own references to a timer then the lib will call
							this method for you as it is shutting down.
				*/
				virtual void destroy();

				#if defined(HDI_CORE_AIP_MODE)
					/**
						\brief		Gets the timer handle around which the target object is wrapped
						\author		GW
						\date		11/2013
						
						\returns	AITimerHandle for this object

						\note		Generally, you shouldn't need to call this method. Only use it if you know what you're
									doing. If a specific piece of functionality provided by Illustrator is not handled by
									this class (or related classes), then it should probably be added to the hdi_core library.
					*/
					virtual AITimerHandle aiTimerHandle() const;
				#endif
				// HDI_CORE_AIP_MODE

				/**
					\brief		Gets whether the target Timer object is empty (constructed with the default ctor)
					\author		GW
					\date		08/2013
					
					\returns	true if the target Timer object is empty, false otherwise
				*/
				virtual bool isEmpty() const;
				
				/**
					\brief		Gets the unique timer name
					\author		GW
					\date		08/2013

					\returns	Name of the timer, as UTF-8
				*/
				virtual std::string name() const;

				/**
					\brief		Gets whether the timer is active
					\author		GW
					\date		08/2013

					\returns	true if the timer is enabled, false otherwise
				*/
				virtual bool active() const;

				/**
					\brief	Sets whether the timer is active
					\author	GW
					\date	08/2013

					\param	active_		true to enable the timer, false to disable

					\note	This method simply calls activate() or deactivate() as appropriate
				*/
				virtual void setActive(const bool active_);

				/**
					\brief	Activates (enables) the timer
					\author	GW
					\date	08/2013
				*/
				virtual void activate();

				/**
					\brief	Deactivates (disables) the timer, preventing it from firing
					\author	GW
					\date	08/2013
				*/
				virtual void deactivate();

				/**
					\brief		Gets the delay in seconds for the target Timer object
					\author		GW
					\date		08/2013
					
					\returns	The number of seconds between firings of the timer callback
				*/
				virtual double delay() const;
				
				/**
					\brief	Sets the delay in seconds for the target Timer object
					\author	GW
					\date	08/2013
					
					\param	delay_	New delay, in seconds, between firings of the timer callback
					
					\note	The new delay is not guaranteed to be respected until after the next time the timer callback
							is fired.
				*/
				virtual void setDelay(const double delay_);

				/**
					\brief		Gets the registered callback for the timer
					\author		GW
					\date		08/2013

					\returns	Currently registered timer callback, or NULL if none
				*/
				virtual Callback* const callback() const;
				
				/**
					\brief	Sets the callback for the timer
					\author	GW
					\date	08/2013

					\param	cb_		New callback to be performed when the timer fires
				*/
				virtual void setCallback(const Callback& cb_);


			private:
				/**
					\brief	Private implementation data
				*/
				void* __data;
				
				#if defined(HDI_CORE_AIP_MODE)
					friend aip::Timer* __accessImpl(const Timer&);
				
					/**
						\brief	Internal use only
						\author	GW
						\date	08/2013
					*/
					void* __impl() const;
				#endif
				// HDI_CORE_AIP_MODE
		};
		
		typedef std::unique_ptr<Timer> TimerUP;
		typedef std::shared_ptr<Timer> TimerSP;
		typedef std::weak_ptr<Timer> TimerWP;
		
		#if defined(HDI_CORE_AIP_MODE)
			extern aip::Timer* __accessImpl(const Timer&);
		#endif
		// HDI_CORE_AIP_MODE
	}
}

#endif
// __HDI_CORE_TIMER__
