/**
	\file
	\brief		Header file for Illustrator tool manipulation
	\copyright	Hot Door, Inc. 2010-2025
*/

#ifndef __HDI_CORE_TOOL__
#define __HDI_CORE_TOOL__

#if defined(HDI_CORE_AIP_MODE)

#include <vector>

#include "hdicoreArtboardPoint.h"
#include "hdicoreTypes.h"

namespace hdi
{
	namespace aip
	{
		class BaseTool;
	}

	namespace core
	{
		class BuiltinTool;
		class Callback;
		class HitData;
		class ModalDialog;

		/**
			\brief	Base class for all tools; stores hit data, cursor loc, dialogs, etc.; receives tool-related messages
		*/
		class Tool
		{
			public:
				typedef std::vector<ArtboardPoint> ArtboardPointVector;
			
				/**
					\brief	Controls various options for a tool
				*/
				enum Options
				{
					NoOptions						= 0,
					TracksCursorOption				= 1 << 0,
					DoesntAutoScrollOption			= 1 << 1,
					BufferedDraggingOption			= 1 << 2,
					MaintainsEditContextOption		= 1 << 3,
					WantsToEditTextOption			= 1 << 4,
					WantsToChangeDiameterOption		= 1 << 5,
					WantsAlternateSelectionOption	= 1 << 6,
					HiddenOption					= 1 << 7,
					DoesntAllowSoftSelectionOption	= 1 << 8,
					DefaultOptions					= (TracksCursorOption | WantsToChangeDiameterOption)
				};

				/**
					\brief	Constructs an empty Tool object
					\author	GW
					\date	09/2013
					
					\note	To test if a Tool object is empty, call isEmpty() on it
					\note	Empty Tool objects do not relate to any actual Illustrator tool; they are designed to
							be "receivers" of some other Tool object via the overloaded assignment operator. Empty Tool
							objects are useless until such time (though it is safe to call any of their methods)
				*/
				Tool();
			
				/**
					\brief	Constructs a new Tool object from an existing Tool object (copy constructor)
					\author	GW
					\date	09/2013

					\param	tool_	Existing Tool object
				*/
				Tool(const Tool& tool_);

				/**
					\brief	Constructs a new Tool object, optionally in the same set/group as one of your own tools
					\author	GW
					\date	09/2013

					\param	sameGroupAs_	Existing reference Tool object for the group the new Tool object belongs in;
											pass an empty Tool object to force a new group
					\param	sameSetAs_		Existing reference Tool object for the set the new Tool object belongs in
											(sets are inside groups); pass NULL for a new set
					\param	title_			Tool title (must be unique), as UTF-8
					\param	tip_			Tool tip (shown when the tool is rolled over in the toolbar), as UTF-8
					\param	iconType_		Type of resource to which the icon IDs relate
					\param	lightIconID_	PNGI or SVG resource ID for the light interface icon
					\param	darkIconID_		PNGI or SVG resource ID for the dark interface icon
					\param	options_		Bitwise OR of Options enum values, controlling behavior of the tool
					
					\note	If the sameGroupAs_ argument is empty, then sameSetAs_ will be ignored.
					\note	Illustrator requires lightIconID_ and darkIconID_ to have a "sibling" resource that is
							lightIconID_+1 and darkIconID_+1 in value (respectively), if using PNGI. Additionally,
							the name of the PNGI resource for lightIconID_+1 and darkIconID_+1 must be same name as
							lightIconID_ and darkIconID_ but suffixed with "@2x". The reason is that Illustrator
							(partially) supports 200% HiDPI resolution displays, and this is their solution for supporting
							higher resolution images. If you do not follow this advice, you will receive a mysterious
							'CANT' error from Illustrator in a modal dialog at startup and your tool(s) will not be loaded.
							Lastly, 150% resolution images can be provided at lightIconID_+2 and darkIconID_+2 with names
							suffixed with "@3to2x".
				*/
				Tool(
					const Tool& sameGroupAs_,
					const Tool* const sameSetAs_,
					const std::string& title_,
					const std::string& tip_,
					const IconResourceType iconType_,
					const int16_t lightIconID_,
					const int16_t darkIconID_,
					const Options options_ = DefaultOptions
				);

				/**
					\brief	Constructs a new Tool object in the same set/group as a built-in Illustrator tool
					\author	GW
					\date	09/2013

					\param	sameGroupAs_	Existing reference BuiltinTool object for the group the new Tool object
											belongs in; pass an empty BuiltinTool object to force a new group
					\param	sameSetAs_		Existing reference BuiltinTool object for the set the new Tool object
											belongs in (sets are inside groups); pass NULL for a new set
					\param	title_			Tool title (must be unique), as UTF-8
					\param	tip_			Tool tip (shown when the tool is rolled over in the toolbar), as UTF-8
					\param	iconType_		Type of resource to which the icon IDs relate
					\param	lightIconID_	PNGI or SVG resource ID for the light interface icon
					\param	darkIconID_		PNGI or SVG resource ID for the dark interface icon
					\param	options_		Bitwise OR of Options enum values, controlling behavior of the tool
					
					\note	If the sameGroupAs_ argument is empty, then sameSetAs_ will be ignored.
					\note	Illustrator requires lightIconID_ and darkIconID_ to have a "sibling" resource that is
							lightIconID_+1 and darkIconID_+1 in value (respectively), if using PNGI. Additionally,
							the name of the PNGI resource for lightIconID_+1 and darkIconID_+1 must be same name as
							lightIconID_ and darkIconID_ but suffixed with "@2x". The reason is that Illustrator
							(partially) supports 200% HiDPI resolution displays, and this is their solution for supporting
							higher resolution images. If you do not follow this advice, you will receive a mysterious
							'CANT' error from Illustrator in a modal dialog at startup and your tool(s) will not be loaded.
							Lastly, 150% resolution images can be provided at lightIconID_+2 and darkIconID_+2 with names
							suffixed with "@3to2x".
				*/
				Tool(
					const BuiltinTool& sameGroupAs_,
					const BuiltinTool* const sameSetAs_,
					const std::string& title_,
					const std::string& tip_,
					const IconResourceType iconType_,
					const int16_t lightIconID_,
					const int16_t darkIconID_,
					const Options options_ = DefaultOptions
				);

				/**
					\brief	Constructs a new Tool object in the same set/group as some third-party tool (via handles)
					\author	GW
					\date	02/2016

					\param	sameGroupAs_	Tool number for the existing group the new Tool object belongs in; pass -2
											for a new group
					\param	sameSetAs_		Tool number for the existing set the new Tool object belongs in (sets are
											inside groups); pass -2 for a new set
					\param	title_			Tool title (must be unique), as UTF-8
					\param	tip_			Tool tip (shown when the tool is rolled over in the toolbar), as UTF-8
					\param	iconType_		Type of resource to which the icon IDs relate
					\param	lightIconID_	PNGI or SVG resource ID for the light interface icon
					\param	darkIconID_		PNGI or SVG resource ID for the dark interface icon
					\param	options_		Bitwise OR of Options enum values, controlling behavior of the tool
					
					\note	Illustrator itself uses a value of -2 to mean "no tool".
					\note	If the sameGroupAs_ argument is -2, then sameSetAs_ will be ignored.
					\note	Illustrator requires lightIconID_ and darkIconID_ to have a "sibling" resource that is
							lightIconID_+1 and darkIconID_+1 in value (respectively), if using PNGI. Additionally,
							the name of the PNGI resource for lightIconID_+1 and darkIconID_+1 must be same name as
							lightIconID_ and darkIconID_ but suffixed with "@2x". The reason is that Illustrator 17+
							(partially) supports 200% HiDPI resolution displays, and this is their solution for supporting
							higher resolution images. If you do not follow this advice, you will receive a mysterious
							'CANT' error from Illustrator in a modal dialog at startup and your tool(s) will not be loaded.
							Lastly, 150% resolution images can be provided at lightIconID_+2 and darkIconID_+2 with names
							suffixed with "@3to2x".
				*/
				explicit Tool(
					const int16_t sameGroupAs_,
					const int16_t sameSetAs_,
					const std::string& title_,
					const std::string& tip_,
					const IconResourceType iconType_,
					const int16_t lightIconID_,
					const int16_t darkIconID_,
					const Options options_ = DefaultOptions
				);

				/**
					\brief	Destructs a Tool object
					\author	GW
					\date	09/2013
				*/
				virtual ~Tool();

				/**
					\brief	Destroys the tool, converting the target object to an empty Tool object (see the default
							constructor for more info on empty Tool objects)
					\author	GW
					\date	12/2013

					\note	This method must be called for any existing tools by the time the plugin is shutting down.
							If you no longer retain any of your own references to a tool then the lib will call this
							method for you as it is shutting down.
				*/
				virtual void destroy();
				
				/**
					\brief		Assigns one Tool object to another
					\author		GW
					\date		09/2013

					\param		rhs_	Existing Tool object to copy values from
					\returns	The target Tool object, but with its values updated to match that of the rhs_ argument
				*/
				virtual Tool& operator=(const Tool& rhs_);
				
				/**
					\brief		Tests whether a given Tool object is the same as another
					\author		GW
					\date		08/2013

					\param		rhs_	Tool to compare against (righthand side of equality operator)
					\returns	true for the target and rhs_ being the same tool, false otherwise
				*/
				virtual bool operator==(const Tool& rhs_) const;
				
				/**
					\brief		Tests whether a given Tool object is not the same as another
					\author		GW
					\date		08/2013

					\param		rhs_	Tool to compare against (righthand side of inequality operator)
					\returns	true for the target and rhs_ being different tools, false otherwise
				*/
				virtual bool operator!=(const Tool& rhs_) const;

				/**
					\brief		Gets the tool handle around which the target object is wrapped
					\author		GW
					\date		11/2013
					
					\returns	AIToolHandle for this object

					\note		Generally, you shouldn't need to call this method. Only use it if you know what you're
								doing. If a specific piece of functionality provided by Illustrator is not handled by
								this class (or related classes), then it should probably be added to the hdi_core library.
				*/
				virtual AIToolHandle aiToolHandle() const;

				/**
					\brief		Gets the tool's number
					\author		GW
					\date		08/2015

					\returns	The unique number for the tool, assigned by Illustrator
					
					\note		Illustrator uses a value of -2 for "no tool", so this is used to indicate an error or
								empty Tool object (hence no underlying number)
				*/
				virtual int16_t number() const;

				/**
					\brief		Gets whether the target Tool object is empty (constructed with the default ctor)
					\author		GW
					\date		09/2013
					
					\returns	true if the target Tool object is empty, false otherwise
				*/
				virtual bool isEmpty() const;
				
				/**
					\brief		Gets whether the target Tool object is the currently selected tool
					\author		GW
					\date		07/2014
					
					\returns	true if the tool is selected, false otherwise
				*/
				virtual bool selected() const;
				
				/**
					\brief	Sets the target Tool object as the currently selected tool
					\author	GW
					\date	09/2014
				*/
				virtual void setSelected();

				/**
					\brief		Gets the options set for the tool
					\author		GW
					\date		09/2013

					\returns	A bitwise OR of Options enum values
				*/
				virtual Options options() const;

				/**
					\brief	Sets the options for the tool
					\author	GW
					\date	09/2013

					\param	options_	A bitwise OR of new Options enum values
				*/
				virtual void setOptions(const Options options_);

				/**
					\brief		Gets the tool's title
					\author		GW
					\date		09/2013

					\returns	The title for the tool, as UTF-8
				*/
				virtual std::string title() const;

				/**
					\brief	Sets the tool's title
					\author	GW
					\date	09/2013

					\param	title_	New title for the tool, as UTF-8
				*/
				virtual void setTitle(const std::string& title_);

				/**
					\brief		Gets the tool's tip string
					\author		GW
					\date		09/2013

					\returns	The rollover tip for the tool, as UTF-8
				*/
				virtual std::string tip() const;

				/**
					\brief	Sets the tool's tip string
					\author	GW
					\date	09/2013

					\param	tip_	New rollover tip for the tool, as UTF-8
				*/
				virtual void setTip(const std::string& tip_);
			
				/**
					\brief		Gets which type of icon resource the tool was originally constructed with
					\author		GW
					\date		09/2018
				 
					\returns	The construction-time icon resource type
				*/
				virtual IconResourceType iconType() const;
				
				/**
					\brief		Gets the light UI icon ID as set when the tool was created
					\author		GW
					\date		02/2014
					
					\returns	The PNGI resource ID for the light interface icon for the target tool
				*/
				virtual int16_t lightIconID() const;
				
				/**
					\brief	Sets the light UI icon ID
					\author	GW
					\date	04/2015
					
					\param	imgID_	New PNGI resource ID for the light interface icon for the target tool
				*/
				virtual void setLightIconID(const int16_t imgID_);
				
				/**
					\brief		Gets the dark UI icon ID as set when the tool was created
					\author		GW
					\date		02/2014
					
					\returns	The PNGI resource ID for the dark interface icon for the target tool
				*/
				virtual int16_t darkIconID() const;
				
				/**
					\brief	Sets the dark UI icon ID
					\author	GW
					\date	04/2015
					
					\param	imgID_	New PNGI resource ID for the dark interface icon for the target tool
				*/
				virtual void setDarkIconID(const int16_t imgID_);

				/**
					\brief		Gets the current location of the cursor
					\author		GW
					\date		09/2013

					\returns	The last known cursor location for the tool

					\note		The cursor location is automatically set when a mouse event occurs
				*/
				virtual ArtboardPoint cursorLoc() const;

				/**
					\brief	Stores the current location of the cursor
					\author	GW
					\date	09/2013

					\param	loc_	Current cursor location

					\note	Typically, this method shouldn't need to be called. The cursor loc for the current (active)
							tool is automatically set when a mouse event occurs.
				*/
				virtual void setCursorLoc(const ArtboardPoint& loc_);

				/**
					\brief		Gets the mouse down points for the target tool
					\author		GW
					\date		09/2013

					\returns	A vector of points where the user performed a mouse down
				*/
				virtual ArtboardPointVector mouseDownLocs() const;

				/**
					\brief		Gets the mouse up points for the target tool
					\author		GW
					\date		09/2013

					\returns	A vector of points where the user performed a mouse up
				*/
				virtual ArtboardPointVector mouseUpLocs() const;

				/**
					\brief		Gets hit data for a given point in the mouse down location vector
					\author		GW
					\date		09/2013

					\param		index_		Index of the point in the mouse down vector
					\param		request_	Type of hit request to perform
					\param		tolerance_	Point radius around mouse down location in which a hit is valid
					\returns	The HitData object for the specified mouse down location
				*/
				virtual std::unique_ptr<HitData> mouseDownHitData(
					const uint32_t index_,
					const HitRequest request_,
					const double tolerance_ = 2.0
				) const;

				/**
					\brief		Gets hit data for a given point in the mouse up location vector
					\author		GW
					\date		09/2013

					\param		index_		Index of the point in the mouse up vector
					\param		request_	Type of hit request to perform
					\param		tolerance_	Point radius around mouse up location in which a hit is valid
					\returns	The HitData object for the specified mouse up location
				*/
				virtual std::unique_ptr<HitData> mouseUpHitData(
					const uint32_t index_,
					const HitRequest request_,
					const double tolerance_ = 2.0
				) const;

				/**
					\brief		Gets hit data for the current cursor location
					\author		GW
					\date		09/2013

					\param		request_	Type of hit request to perform
					\param		tolerance_	Point radius around cursor location in which a hit is valid
					\returns	The HitData object for the cursor location
				*/
				virtual std::unique_ptr<HitData> cursorHitData(
					const HitRequest request_,
					const double tolerance_ = 2.0
				) const;

				/**
					\brief		Gets the registered callback for the tool selection event
					\author		GW
					\date		09/2013
					
					\returns	Callback for selection, or NULL if none
				*/
				virtual Callback* const selectionCallback() const;

				/**
					\brief	Sets the callback for the tool selection event
					\author	GW
					\date	09/2013
					
					\param	cb_		The new selection callback
				*/
				virtual void setSelectionCallback(const Callback& cb_);

				/**
					\brief		Gets the registered callback for the tool reselection event
					\author		GW
					\date		09/2013
					
					\returns	Callback for reselection, or NULL if none
				*/
				virtual Callback* const reselectionCallback() const;

				/**
					\brief	Sets the callback for the tool reselection event
					\author	GW
					\date	09/2013
					
					\param	cb_		The new reselection callback
				*/
				virtual void setReselectionCallback(const Callback& cb_);

				/**
					\brief		Gets the registered callback for the tool deselection event
					\author		GW
					\date		09/2013
					
					\returns	Callback for deselection, or NULL if none
				*/
				virtual Callback* const deselectionCallback() const;

				/**
					\brief	Sets the callback for the tool deselection event
					\author	GW
					\date	09/2013
					
					\param	cb_		The new deselection callback
				*/
				virtual void setDeselectionCallback(const Callback& cb_);

				/**
					\brief		Gets the registered callback for the tool tracking event (mouse move)
					\author		GW
					\date		09/2013
					
					\returns	Callback for tracking, or NULL if none
				*/
				virtual Callback* const trackCallback() const;

				/**
					\brief	Sets the callback for the tool tracking event (mouse move)
					\author	GW
					\date	09/2013
					
					\param	cb_		The new tracking callback
				*/
				virtual void setTrackCallback(const Callback& cb_);

				/**
					\brief		Gets the registered callback for the mouse down event
					\author		GW
					\date		09/2013
					
					\returns	Callback for mouse down, or NULL if none
				*/
				virtual Callback* const mouseDownCallback() const;

				/**
					\brief	Sets the callback for the mouse down event
					\author	GW
					\date	09/2013
					
					\param	cb_		The new mouse down callback
				*/
				virtual void setMouseDownCallback(const Callback& cb_);

				/**
					\brief		Gets the registered callback for the drag event
					\author		GW
					\date		09/2013
					
					\returns	Callback for dragging, or NULL if none
				*/
				virtual Callback* const dragCallback() const;

				/**
					\brief	Sets the callback for the drag event
					\author	GW
					\date	09/2013
					
					\param	cb_		The new dragging callback
				*/
				virtual void setDragCallback(const Callback& cb_);

				/**
					\brief		Gets the registered callback for the mouse up event
					\author		GW
					\date		09/2013
					
					\returns	Callback for mouse up, or NULL if none
				*/
				virtual Callback* const mouseUpCallback() const;

				/**
					\brief	Sets the callback for the mouse up event
					\author	GW
					\date	09/2013
					
					\param	cb_		The new mouse up callback
				*/
				virtual void setMouseUpCallback(const Callback& cb_);

				/**
					\brief		Gets the registered callback for the decrease diameter event ('[' key)
					\author		GW
					\date		09/2013
					
					\returns	Callback for decreasing diameter, or NULL if none
				*/
				virtual Callback* const decreaseDiameterCallback() const;

				/**
					\brief	Sets the callback for the decrease diameter event
					\author	GW
					\date	09/2013
					
					\param	cb_		The new decrease diameter callback
				*/
				virtual void setDecreaseDiameterCallback(const Callback& cb_);

				/**
					\brief		Gets the registered callback for the increase diameter event (']' key)
					\author		GW
					\date		09/2013
					
					\returns	Callback for increasing diameter, or NULL if none
				*/
				virtual Callback* const increaseDiameterCallback() const;

				/**
					\brief	Sets the callback for the increase diameter event
					\author	GW
					\date	09/2013
					
					\param	cb_		The new increase diameter callback
				*/
				virtual void setIncreaseDiameterCallback(const Callback& cb_);

				/**
					\brief		Gets the registered callback for the tool edit event (when its icon is double-clicked)
					\author		GW
					\date		09/2013
					
					\returns	Callback for tool edit, or NULL if none
				*/
				virtual Callback* const editCallback() const;

				/**
					\brief	Sets the callback for the tool edit event
					\author	GW
					\date	09/2013
					
					\param	cb_		The new edit callback
				*/
				virtual void setEditCallback(const Callback& cb_);

				/**
					\brief	Clears the mouse down and mouse up location containers
					\author	GW
					\date	09/2013

					\note	Since tool implementations often vary (e.g. a tool might require exactly two clicks instead
							of just one click), it is up to the tool creator to call this method when appropriate.
				*/
				virtual void resetCursorLocs();

				/**
					\brief		Gets whether the tool has been dragged
					\author		GW
					\date		09/2013

					\returns	true if a mouse drag event was received between mouse down and mouse up events

					\note		The state of the dragged bit is reset with each mouse down event.
				*/
				virtual bool wasDragged() const;

				/**
					\brief		Determines whether the shift key is down
					\author		GW
					\date		09/2013

					\returns	true if the shift key was down in the last tool event

					\note		The return value is only valid if the last message the plugin received was a tool-related
								event, and only if the event relates to the target Tool.
				*/
				virtual bool shiftKeyDown() const;

				/**
					\brief		Determines whether the command key is down
					\author		GW
					\date		09/2013

					\returns	true if the command key was down in the last tool event

					\note		Windows machines do not have a command key. If this method is called on a Windows
								machine, it will return the same value as the controlKeyDown() method (because the
								control key is the closest command key analog on Windows).
					\note		The return value is only valid if the last message the plugin received was a tool-related
								event, and only if the event relates to the target Tool.
				*/
				virtual bool commandKeyDown() const;

				/**
					\brief		Determines whether the option key is down
					\author		GW
					\date		09/2013

					\returns	true if the option key was down in the last tool event

					\note		The option key on Mac is akin to the alt key on Windows.
					\note		The return value is only valid if the last message the plugin received was a tool-related
								event, and only if the event relates to the target Tool.
				*/
				virtual bool optionKeyDown() const;

				/**
					\brief		Determines whether the control key is down
					\author		GW
					\date		09/2013

					\returns	true if the control key was down in the last tool event

					\note		The return value is only valid if the last message the plugin received was a tool-related
								event, and only if the event relates to the target Tool.
				*/
				virtual bool controlKeyDown() const;

				/**
					\brief		Gets the modal input dialog for the target tool
					\author		GW
					\date		10/2013
					
					\returns	A pointer to the modal input dialog for the tool, or NULL if none/error
					
					\note		It is common for tools to have modal input dialogs that are shown when the mouse button
								comes up; however, this process is not automated for the creator of the tool as not all
								tools show an input dialog at the same point in the mouse event timeline. The Tool class
								merely offers a convenient place to store said dialog object.
				*/
				virtual ModalDialog* const inputDialog() const;
				
				/**
					\brief	Sets the modal input dialog for the target tool
					\author	GW
					\date	10/2013
					
					\param	dialog_		New modal input dialog for the tool
					
					\note	It is common for tools to have modal input dialogs that are shown when the mouse button
							comes up; however, this process is not automated for the creator of the tool as not all
							tools show an input dialog at the same point in the mouse event timeline. The Tool class
							merely offers a convenient place to store said dialog object.
				*/
				virtual void setInputDialog(const ModalDialog& dialog_);

				/**
					\brief		Gets the modal prefs dialog for the target tool
					\author		GW
					\date		10/2013
					
					\returns	A pointer to the modal prefs dialog for the tool, or NULL if none/error
					
					\note		It is common for tools to have modal prefs dialogs that are shown when the tool is
								double-clicked in the Illustrator toolbox; this process is automated for the creator of
								the tool unless setAutoShowPrefsDialog(false) is called.
				*/
				virtual ModalDialog* const prefsDialog() const;
				
				/**
					\brief	Sets the modal prefs dialog for the target tool
					\author	GW
					\date	10/2013
					
					\param	dialog_		New modal prefs dialog for the tool
					
					\note	It is common for tools to have modal prefs dialogs that are shown when the tool is double-
							clicked in the Illustrator toolbox; this process is automated for the creator of the tool
							unless setAutoShowPrefsDialog(false) is called.
				*/
				virtual void setPrefsDialog(const ModalDialog& dialog_);

				/**
					\brief		Gets whether the tool prefs dialog should automatically be shown when the tool is
								double-clicked in the Illustrator toolbox
					\author		GW
					\date		10/2013
					
					\returns	true if the dialog will automatically be shown, false otherwise
				*/
				virtual bool autoShowPrefsDialog() const;
				
				/**
					\brief	Sets whether the tool prefs dialog should automatically be shown when the tool is double-
							clicked in the Illustrator toolbox
					\author	GW
					\date	10/2013
					
					\param	show_	true to show the prefs dialog automatically
				*/
				virtual void setAutoShowPrefsDialog(const bool show_);


			private:
				friend aip::BaseTool* __accessImpl(const Tool&);
				friend Tool __accessCtorTool(const void*);

				/**
					\brief	Private implementation data
				*/
				void* __data;
				
				/**
					\brief	Internal use only
					\author	GW
					\date	08/2013
				*/
				void* __impl() const;

				/**
					\brief	Internal use only
					\author	GW
					\date	08/2013
				*/
				Tool(const void*);
		};
		
		typedef std::unique_ptr<Tool> ToolUP;
		typedef std::shared_ptr<Tool> ToolSP;
		typedef std::weak_ptr<Tool> ToolWP;
		
		extern aip::BaseTool* __accessImpl(const Tool&);
		extern Tool __accessCtorTool(const void*);
	}
}

hdi::core::Tool::Options operator|(
	const hdi::core::Tool::Options lhs_,
	const hdi::core::Tool::Options rhs_
);

hdi::core::Tool::Options& operator|=(
	hdi::core::Tool::Options& lhs__,
	const hdi::core::Tool::Options rhs_
);

#endif
// HDI_CORE_AIP_MODE

#endif
// __HDI_CORE_TOOL__
