/**
	\file
	\brief		Header file for creating tree-like entry objects with multiple columns, to be added to Widget subclasses
				(that support them)
	\copyright	Hot Door, Inc. 2010-2025
*/

#ifndef __HDI_CORE_TREE_COLUMN_ENTRY__
#define __HDI_CORE_TREE_COLUMN_ENTRY__

#include <vector>

#include "hdicoreEntryValue.h"
#include "hdicoreMacros.h"

namespace hdi
{
	namespace pui
	{
		class TreeColumnEntryData;
	}
	
	namespace core
	{
		/**
			\brief	Allows hierarchical column views to store more data than simply strings and indices
		*/
		class TreeColumnEntry
		{
			public:
				typedef std::vector<EntryValue> ValueVector;
				typedef std::vector<bool> EditableVector;
				typedef std::vector< void* > DataVector;
			
				/**
					\brief	When you want a separator in a container widget, create a TreeColumnEntry object with
							this as its string value
				*/
				static const char* separatorValue;
			
				/**
					\brief	Constructs an empty TreeColumnEntry object
					\author	GW
					\date	01/2015
					
					\note	To test if a TreeColumnEntry object is empty, call isEmpty() on it
					\note	Empty TreeColumnEntry objects do not relate to any entry in any widget; they are
							designed to be "receivers" of some other TreeColumnEntry object via the overloaded
							assignment operator. Empty TreeColumnEntry objects are useless until such time (though
							it is safe to call any of their methods).
				*/
				TreeColumnEntry();
			
				/**
					\brief	Constructs a new TreeColumnEntry object from an existing TreeColumnEntry object (copy
							constructor)
					\author	GW
					\date	01/2015

					\param	e_	Existing TreeColumnEntry object
				*/
				TreeColumnEntry(const TreeColumnEntry& e_);

				/**
					\brief	Constructs a TreeColumnEntry object with some custom data
					\author	GW
					\date	01/2015

					\param	id_			Entry ID string, as UTF-8 (must form a unique ID path in the target container)
					\param	values_		Entry value objects, one for each column
					\param	userData_	Data to store in each column of the entry
					\param	enabled_	Whether the entry should be enabled
					\param	expanded_	true to expand the entry, false to collapse
					\param	editable_	Whether each cell in the entry should be inline editable
				*/
				TreeColumnEntry(
					const std::string& id_,
					const ValueVector& values_,
					const DataVector& userData_,
					const bool enabled_,
					const bool expanded_,
					const EditableVector& editable_
				);

				/**
					\brief	Destructs a TreeColumnEntry object
					\author	GW
					\date	01/2015
				*/
				virtual ~TreeColumnEntry();
				
				/**
					\brief		Assigns one TreeColumnEntry object to another
					\author		GW
					\date		01/2015

					\param		rhs_	Existing TreeColumnEntry object to copy values from
					\returns	The target TreeColumnEntry object, but with its values updated to match that of the
								rhs_ argument
				*/
				virtual TreeColumnEntry& operator=(const TreeColumnEntry& rhs_);

				/**
					\brief		Gets whether the target TreeColumnEntry object is empty (constructed with the default
								ctor)
					\author		GW
					\date		01/2015
					
					\returns	true if the target TreeColumnEntry object is empty, false otherwise
				*/
				virtual bool isEmpty() const;

				/**
					\brief		Gets the full ID path of the target entry
					\author		GW
					\date		12/2022

					\returns	The the ID path for the target, from top-most to the target
					
					\warning	Only valid after the entry and its ancestors have been added to a HierarchicalWidget, and
								can change if the entries are e.g. reorganized and the widget is updated.
				*/
				virtual std::vector<std::string> idPath() const;

				/**
					\brief		Gets the ID string of an entry
					\author		GW
					\date		01/2015

					\returns	The unique ID of the entry, as UTF-8
				*/
				virtual std::string entryID() const;
				
				/**
					\brief	Sets the ID string of an entry
					\author	GW
					\date	01/2015

					\param	id_		New entry ID string, as UTF-8

					\note	It is up to the caller to ensure that the ID path is not in use with other entries within the
							same container widget.
				*/
				virtual void setEntryID(const std::string& id_);
				
				/**
					\brief		Gets the value objects of an entry
					\author		GW
					\date		01/2015

					\returns	The entry's value objects
				*/
				virtual ValueVector values() const;
				
				/**
					\brief	Sets the value objects of an entry
					\author	GW
					\date	01/2015

					\param	values_		New entry value objects
				*/
				virtual void setValues(const ValueVector& values_);

				/**
					\brief		Gets whether the entry should be enabled (selectable by the user in the list)
					\author		GW
					\date		01/2015
					
					\returns	true for enabled, false otherwise
				*/
				virtual bool enabled() const;

				/**
					\brief	Sets whether the entry should be enabled (selectable by the user in the list)
					\author	GW
					\date	01/2015
					
					\param	enabled_	true for enabled, false otherwise
				*/
				virtual void setEnabled(const bool enabled_);

				/**
					\brief		Gets whether the entry is expanded (i.e. its children are visible)
					\author		GW
					\date		02/2015
					
					\returns	true for expanded, false for collapsed
				*/
				virtual bool expanded() const;

				/**
					\brief	Sets whether the entry is expanded (i.e. its children are visible)
					\author	GW
					\date	02/2015
					
					\param	exp_	true for expanded, false for collapsed
				*/
				virtual void setExpanded(const bool exp_);
				
				/**
					\brief		Gets the editability of each cell in the entry (whether the user can edit it inline)
					\author		GW
					\date		07/2022
					
					\returns	true for enabled, false otherwise
				*/
				virtual EditableVector editable() const;

				/**
					\brief	Sets the editability of each cell in the entry (whether the user can edit it inline)
					\author	GW
					\date	07/2022
					
					\param	enabled_	true for enabled, false otherwise
				*/
				virtual void setEditable(const EditableVector& editable_);

				/**
					\brief		Gets the custom data of an entry
					\author		GW
					\date		01/2015

					\returns	The data stored inside the target entry object
				*/
				virtual DataVector userData() const;
				
				/**
					\brief	Sets the custom data of an entry
					\author	GW
					\date	01/2015

					\param	data_	New custom data to store
				*/
				virtual void setUserData(const DataVector& data_);

				/**
					\brief		Gets whether the entry has a child with the given ID string
					\author		GW
					\date		12/2022
				
					\param		id_		ID of the child to search for
					\returns	true if found, false otherwise
					
					\note		This method performs an O(n) operation, and only searches direct children (i.e. not a
								full-subtree search).
				*/
				virtual bool hasChild(const std::string& id_) const;
				
				/**
					\brief		Get the child entry with the given ID string, if it exists within the target
					\author		GW
					\date		12/2022
				
					\param		id_		ID of the child to search for
					\returns	The found child entry, or NULL if not found
					
					\note		This method performs an O(n) operation, and only searches direct children (i.e. not a
								full-subtree search).
				*/
				virtual std::unique_ptr<TreeColumnEntry> childWithID(const std::string& id_) const;
				
				/**
					\brief		Gets the count of child entries in the target entry
					\author		GW
					\date		01/2015
					
					\returns	The count of child entries
				*/
				virtual uint32_t childCount() const;
				
				/**
					\brief		Gets a specific child of the target entry
					\author		GW
					\date		01/2015
					
					\param		index_	Index of the desired child entry
					\returns	The child at the given index, or NULL for error
				*/
				virtual std::unique_ptr<TreeColumnEntry> childAtIndex(const uint32_t index_) const;
				
				/**
					\brief		Pushes a new entry onto the end of the list of children of the target entry
					\author		GW
					\date		01/2015
					
					\param		child_	New child to add to the end of the list of children
					\returns	true if the child could be pushed, false otherwise
				*/
				virtual bool pushChild(const TreeColumnEntry& child_);
				
				/**
					\brief		Pops an existing entry off the end of the list of children of the target entry
					\author		GW
					\date		01/2015
					
					\param		child__		Return-by-reference for the child entry that was removed
					\returns	true if the child could be acquired and popped, false otherwise
				*/
				virtual bool popChild(TreeColumnEntry& child__);
				
				/**
					\brief		Inserts an entry into the list of child entries at a given position
					\author		GW
					\date		12/2022

					\param		id_		Existing entry immediately before the position at which the new entry will be
										inserted, as UTF-8; pass "" to insert at the beginning
					\param		child_	New entry object to insert into the list of children
					\returns	true if the entry is inserted successfully, false otherwise

					\note		If the entry for id_ doesn't exist in the list, the new child will not be inserted.
					\note		If the passed child is already present in the list (based on its ID), it will not be
								inserted.
				*/
				virtual bool insertChild(const std::string& id_, const TreeColumnEntry& child_);
				
				/**
					\brief		Removes a specific entry from the list of child entries
					\author		GW
					\date		12/2022

					\param		id_			ID of the entry to be removed, as UTF-8
					\param		entry__		Return-by-reference for the removed child, so the caller can inspect which
											one was removed
					\returns	true if the child was removed and acquired, false otherwise
				*/
				virtual bool removeChild(const std::string& id_, TreeColumnEntry& child__);
				
				/**
					\brief		Tests whether a given TreeColumnEntry object is the same as another
					\author		GW
					\date		01/2015

					\param		rhs_	TreeColumnEntry to compare against (righthand side of equality operator)
					\returns	true for the target and rhs_ being the same entry, false otherwise
				*/
				virtual bool operator==(const TreeColumnEntry& rhs_) const;
				
				/**
					\brief		Tests whether a given TreeColumnEntry object is not the same as another
					\author		GW
					\date		01/2015

					\param		rhs_	TreeColumnEntry to compare against (righthand side of inequality operator)
					\returns	true for the target and rhs_ being different entries, false otherwise
				*/
				virtual bool operator!=(const TreeColumnEntry& rhs_) const;


			private:
				friend pui::TreeColumnEntryData* __accessImpl(const TreeColumnEntry&);

				/**
					\brief	Private implementation data
				*/
				void* __data;
				
				/**
					\brief	Internal use only
					\author	GW
					\date	09/2013
				*/
				void* __impl() const;
		};
		
		typedef std::unique_ptr<TreeColumnEntry> TreeColumnEntryUP;
		typedef std::shared_ptr<TreeColumnEntry> TreeColumnEntrySP;
		typedef std::weak_ptr<TreeColumnEntry> TreeColumnEntryWP;
		
		extern pui::TreeColumnEntryData* __accessImpl(const TreeColumnEntry&);
	}
}

#endif
// __HDI_CORE_TREE_COLUMN_ENTRY__
