/**
	\file
	\brief		Header file for a class to create and manipulate plugin windows
	\copyright	Hot Door, Inc. 2010-2025
*/

#ifndef __HDI_CORE_WINDOW__
#define __HDI_CORE_WINDOW__

#include <vector>

#include "hdicoreTypes.h"

namespace hdi
{
	namespace core
	{
		class TabOrder;
		class Widget;
		class WidgetGroup;

		/**
			\brief	Used as a base class for modal dialogs, panels, and control bars
		*/
		class Window
		{
			public:
				typedef std::vector< std::shared_ptr<Widget> > WidgetVector;
			
				/**
					\brief	Destructs a Window object
					\author	GW
					\date	09/2013
				*/
				virtual ~Window();
				
				/**
					\brief		Convenience method to clone a Window object, taking its subclass type into account for
								proper memory management
					\author		GW
					\date		10/2013

					\returns	A pointer to the new Window object
					
					\warning	The caller must manage the memory for the returned Window object.
				*/
				virtual Window* clone() const = 0;

				/**
					\brief		Gets whether the target Window object is empty (constructed with the default ctor)
					\author		GW
					\date		09/2013
					
					\returns	true if the target Window object is empty, false otherwise
				*/
				virtual bool isEmpty() const;

				/**
					\brief		Gets the type of window
					\author		GW
					\date		09/2013

					\returns	The type of the window

					\note		This is useful for knowing how to cast a Window object to its proper subclass
				*/
				virtual WindowType windowType();

				/**
					\brief		Tests whether a given Window object is the same as another
					\author		GW
					\date		09/2013

					\param		rhs_	Window to compare against (righthand side of equality operator)
					\returns	true for the target and rhs_ being the same UI window, false otherwise
				*/
				virtual bool operator==(const Window& rhs_) const;

				/**
					\brief		Tests whether a given Window object is not the same as another
					\author		GW
					\date		09/2013

					\param		rhs_	Window to compare against (righthand side of inequality operator)
					\returns	true for the target and rhs_ being different UI windows, false otherwise
				*/
				virtual bool operator!=(const Window& rhs_) const;

				/**
					\brief		Gets whether the window is visible
					\author		GW
					\date		09/2013

					\returns	true if the window is visible, false otherwise
				*/
				virtual bool visible() const = 0;
				
				/**
					\brief	Shows the window
					\author	GW
					\date	09/2013
				*/
				virtual void show() = 0;

				/**
					\brief		Gets a list of all widgets contained by the target window
					\author		GW
					\date		09/2013

					\returns	A list of all widgets contained by the window
				*/
				virtual const WidgetVector widgets() const;

				/**
					\brief		Adds a widget (label, button, text field, etc.) to the window
					\author		GW
					\date		09/2013

					\param		widget_		Widget to place in the window
					\returns	true if the widget could be added, false otherwise
				*/
				virtual bool addWidget(const Widget& widget_);

				/**
					\brief		Removes a widget (label, button, text field, etc.) from the window
					\author		GW
					\date		09/2013

					\param		widget_		Widget to remove from the window
					\returns	true if the widget was removed, false otherwise
				*/
				virtual bool removeWidget(const Widget& widget_);

				/**
					\brief		Adds a widget group to the window
					\author		GW
					\date		02/2015

					\param		wg_		Widget group to place in the window
					\returns	true if the widgets could be added, false otherwise
				*/
				virtual bool addWidgetGroup(const WidgetGroup& wg_);

				/**
					\brief		Removes a widget group from the window
					\author		GW
					\date		02/2015

					\param		wg_		Widget group to remove from the window
					\returns	true if the widgets were removed, false otherwise
				*/
				virtual bool removeWidgetGroup(const WidgetGroup& wg_);

				/**
					\brief		Gets the tab order object for the window
					\author		GW
					\date		10/2013
					
					\returns	A pointer to the TabOrder object for the target window (all Window objects have one)
				*/
				virtual TabOrder* const tabOrder() const;

				/**
					\brief	Forces the widgets in the modal dialog to update/redraw
					\author	GW
					\date	09/2013
				*/
				virtual void update() const = 0;

				/**
					\brief		Gets the factor by which the UI is scaled for the given window (i.e. for HiDPI support)
					\author		GW
					\date		02/2017
					
					\returns	The scaling factor for the target window; 1.0 means e.g. @1x image variants should be
								used, 2.0 means @2x image variants should be used, etc.
					
					\note		On Windows, it is possible to have a variety of non-round scaling factors (e.g. 1.25 or
								1.5); Adobe optionally respects these scaling factors.
				*/
				virtual double scalingFactor() const;


			protected:
				/**
					\brief	Private implementation data
				*/
				void* _data;
			
				/**
					\brief	Constructs an empty Window object
					\author	GW
					\date	10/2013
					
					\note	To test if a Window object is empty, call isEmpty() on it
					\note	Empty Window objects do not relate to any actual UI window; they are designed to be
							"receivers" of some other Window object via the overloaded assignment operator. Empty Window
							objects are useless until such time (though it is safe to call any of their methods).
				*/
				Window();
			
				/**
					\brief	Constructs a new Window object from an existing Window object (copy constructor)
					\author	GW
					\date	10/2013

					\param	w_	Existing Window object
					
					\note	This constructor should not be called directly; subclasses should "overload" it and call
							another ctor.
				*/
				Window(const Window& w_);
			
				/**
					\brief	Constructs a Window object
					\author	GW
					\date	09/2013

					\param	type_	Type of window that is being constructed (indicative of the Window subclass)
				*/
				Window(const WindowType type_);
				
				/**
					\brief	Internal use only
					\author	GW
					\date	12/2014
				*/
				virtual Window& operator=(const Window&);
		};
		
		typedef std::unique_ptr<Window> WindowUP;
		typedef std::shared_ptr<Window> WindowSP;
		typedef std::weak_ptr<Window> WindowWP;
	}
}

#endif
// __HDI_CORE_WINDOW__
