/**
	\file
	\brief		Header file for plugin alerts, confirmation dialogs, etc.
	\copyright	Hot Door, Inc. 2010-2025
*/

#ifndef __HDI_CORE_ALERTS__
#define __HDI_CORE_ALERTS__

#include "hdicoreTypes.h"

namespace hdi
{
	namespace core
	{
		class Callback;
	
		/**
			\brief	Organizes all basic alerting functions into one namespace
		*/
		namespace alerts
		{
			/**
				\brief	Describes the choice the user made when dismissing an alert
				\note	message() and error() do not return an AlertResponse, because they both only have one
						button.
				\note	confirm() will only return either NoButtonChosen or YesButtonChosen; however, if alerts have
						been disabled with setState(), it will always return CancelButtonChosen.
			*/
			enum Response
			{
				CancelButtonChosen	= 1,
				NoButtonChosen		= 2,
				YesButtonChosen		= 3
			};

			/**
				\brief		Allows alerts to be (temporarily) disabled
				\author		GW
				\date		08/2013

				\param		enabled_	true to enable alerts, false to disable
				\returns	The previous enable/disable setting

				\note		When alerts are disabled, the message(), error(), confirm(), and question()
							functions will return early, thereby preventing the alert UI from ever being presented to
							the user. The beep() function will return early, preventing a beep sound from being made.
			*/
			bool setState(const bool enabled_);

			/**
				\brief	Shows a simple platform message dialog, with just an "OK" button and a message/warning icon
				\author	GW
				\date	08/2013

				\param	message_	Message string to show to the user in the dialog, as UTF-8

				\note	The dialog is "blocking" and execution will continue immediately after the call to this function
			*/
			void message(const std::string& message_);

			/**
				\brief	Shows a simple platform error dialog, with just an "OK" button and an error icon
				\author	GW
				\date	08/2013

				\param	error_	Error string to show to the user in the dialog, as UTF-8

				\note	The dialog is "blocking" and execution will continue immediately after the call to this function
			*/
			void error(const std::string& error_);

			/**
				\brief		Shows a confirmation dialog with default affirmative and negative buttons ("Yes" and "No") 
				\author		GW
				\date		08/2013

				\param		question_	Question to pose to the user, as UTF-8
				\returns	The choice the user made within the dialog (either YesButtonChosen or NoButtonChosen)

				\note		The dialog is "blocking" and execution will continue immediately after the call to this
							function.
			*/
			Response confirm(const std::string& question_);

			/**
				\brief		Shows a confirmation dialog with affirmative and negative buttons with custom titles
				\author		GW
				\date		08/2013

				\param		question_	Question to pose to the user, as UTF-8
				\param		yesBtnStr_	Title for the affirmative button, as UTF-8
				\param		noBtnStr_	Title for the negative button, as UTF-8
				\returns	The choice the user made within the dialog (either YesButtonChosen or NoButtonChosen)

				\note		The dialog is "blocking" and execution will continue immediately after the call to this
							function.
			*/
			Response confirm(
				const std::string& question_,
				const std::string& yesBtnStr_,
				const std::string& noBtnStr_
			);

			/**
				\brief		Shows a confirmation dialog with default affirmative and negative buttons ("Yes" and "No") 
				\author		GW
				\date		08/2013

				\param		question_	Question to pose to the user, as UTF-8
				\param		yesCb_		Action to perform on the user having chosen "Yes"
				\param		noCb_		Action to perform on the user having chosen "No"
				\returns	The choice the user made within the dialog (either YesButtonChosen or NoButtonChosen)

				\note		The dialog is "blocking" and, immediately after the call to this function and to any passed
							callbacks, execution will continue
			*/
			Response confirm(const std::string& question_, const Callback& yesCb_, const Callback& noCb_);

			/**
				\brief		Shows a confirmation dialog with affirmative and negative buttons with custom titles
				\author		GW
				\date		08/2013

				\param		question_	Question to pose to the user, as UTF-8
				\param		yesBtnStr_	Title for the affirmative button, as UTF-8
				\param		yesCb_		Action to perform on the user having chosen "Yes"
				\param		noBtnStr_	Title for the negative button, as UTF-8
				\param		noCb_		Action to perform on the user having chosen "No"
				\returns	The choice the user made within the dialog (either YesButtonChosen or NoButtonChosen)

				\note		The dialog is "blocking" and, immediately after the call to this function and to any passed
							callbacks, execution will continue
			*/
			Response confirm(
				const std::string& question_,
				const std::string& yesBtnStr_,
				const Callback& yesCb_,
				const std::string& noBtnStr_,
				const Callback& noCb_
			);

			/**
				\brief		Shows a question dialog with default affirmative, negative, and dismissal buttons ("Yes",
							"No", and "Cancel")
				\author		GW
				\date		08/2013

				\param		question_	Question to pose to the user, as UTF-8
				\returns	The choice the user made within the dialog

				\note		The dialog is "blocking" and execution will continue immediately after the call to this
							function.
			*/
			Response question(const std::string& question_);

			/**
				\brief		Shows a question dialog with affirmative, negative, and dismissal buttons with custom titles
				\author		GW
				\date		08/2013

				\param		question_		Question to pose to the user, as UTF-8
				\param		yesBtnStr_		Title for the affirmative button, as UTF-8
				\param		noBtnStr_		Title for the negative button, as UTF-8
				\param		cancelBtnStr_	Title for the dismissal button, as UTF-8
				\returns	The choice the user made within the dialog

				\note		The dialog is "blocking" and execution will continue immediately after the call to this
							function.
			*/
			Response question(
				const std::string& question_,
				const std::string& yesBtnStr_,
				const std::string& noBtnStr_,
				const std::string& cancelBtnStr_
			);

			/**
				\brief		Shows a question dialog with default affirmative, negative, and dismissal buttons ("Yes",
							"No", and "Cancel")
				\author		GW
				\date		08/2013

				\param		question_	Question to pose to the user, as UTF-8
				\param		yesCb_		Action to perform on the user having chosen "Yes"
				\param		noCb_		Action to perform on the user having chosen "No"
				\param		cancelCb_	Action to perform on the user having chosen "Cancel"
				\returns	The choice the user made within the dialog

				\note		The dialog is "blocking" and, immediately after the call to this function and to any passed
							callbacks, execution will continue
			*/
			Response question(
				const std::string& question_,
				const Callback& yesCb_,
				const Callback& noCb_,
				const Callback& cancelCb_
			);

			/**
				\brief		Shows a question dialog with affirmative, negative, and dismissal buttons with custom titles
				\author		GW
				\date		08/2013

				\param		question_		Question to pose to the user, as UTF-8
				\param		yesBtnStr_		Title for the affirmative button, as UTF-8
				\param		yesCb_			Action to perform on the user having chosen "Yes"
				\param		noBtnStr_		Title for the negative button, as UTF-8
				\param		noCb_			Action to perform on the user having chosen "No"
				\param		cancelBtnStr_	Title for the dismissal button, as UTF-8
				\param		cancelCb_		Action to perform on the user having chosen "Cancel"
				\returns	The choice the user made within the dialog

				\note		The dialog is "blocking" and, immediately after the call to this function and to any passed
							callbacks, execution will continue
			*/
			Response question(
				const std::string& question_,
				const std::string& yesBtnStr_,
				const Callback& yesCb_,
				const std::string& noBtnStr_,
				const Callback& noCb_,
				const std::string& cancelBtnStr_,
				const Callback& cancelCb_
			);

			/**
				\brief	Causes a platform beep to occur
				\author	GW
				\date	08/2013
			*/
			void beep();
		}
	}
}

#endif
// __HDI_CORE_ALERTS__
