/**
	\file
	\brief		Header file for describing angles
	\copyright	Hot Door, Inc. 2010-2025
*/

#ifndef __HDI_CORE_ANGLE__
#define __HDI_CORE_ANGLE__

#include "hdicoreConstants.h"
#include "hdicoreTypes.h"

namespace hdi
{
	namespace geo
	{
		class Angle;
	}

	namespace core
	{
		/**
			\brief	Describes an angle, and allows for easy conversion between various units
		*/
		class Angle
		{
			public:
				/**
					\brief	Units supported for angle calculations
				*/
				enum Units
				{
					Radians		= 1,
					Degrees		= 2,
					Gradians	= 3,
					Turns		= 4
				};

				/**
					\brief	Default Angle constructor; sets the angle to 0.0 radians
					\author	GW
					\date	08/2013
				*/
				Angle();

				/**
					\brief	Angle constructor; sets the angle to the provided value
					\author	GW
					\date	08/2013

					\param	value_		Value for the new Angle object
					\param	units_		Units for the value_ object
				*/
				Angle(const double value_, const Units units_ = Radians);
				
				/**
					\brief	Angle copy constructor
					\author	GW
					\date	08/2013

					\param	angle_	Existing Angle object, whose values will be copied
				*/
				Angle(const Angle& angle_);
				
				/**
					\brief	Angle destructor
					\author	GW
					\date	08/2013
				*/
				virtual ~Angle();

				/**
					\brief		Checks for two angles being within a given tolerance of one another
					\author		GW
					\date		08/2013

					\param		compare_	Angle to compare against the target object
					\param		tol_		Allowable tolerance between the two objects, for them to be considered "equal"
					\returns	true if the angles are "equal" within the given tolerance
				*/
				virtual bool equalWithinTol(const Angle& compare_, const Angle& tol_) const;

				/**
					\brief		Compares two angles for equality
					\author		GW
					\date		08/2013

					\param		rhs_	Righthand side of the == operator; the angle to compare against
					\returns	true if both angles' values are binary equal to each other, or false otherwise
				*/
				virtual bool operator==(const Angle& rhs_) const;

				/**
					\brief		Compares two angles for inequality
					\author		GW
					\date		08/2013

					\param		rhs_	Righthand side of the != operator; the angle to compare against
					\returns	true if the angles' values are unequal to each other, or false otherwise
				*/
				virtual bool operator!=(const Angle& rhs_) const;
				
				/**
					\brief		Checks if the target angle's value is less than the provided angle's
					\author		GW
					\date		08/2013

					\param		rhs_	Righthand side of the < operator; the angle to compare against
					\returns	true if the target angle's value is less than the value for rhs_, once their ranges are
								restricted to [0,2pi]
				*/
				virtual bool operator<(const Angle& rhs_) const;
				
				/**
					\brief		Checks if the target angle's value is less than or equal to the provided angle's
					\author		GW
					\date		08/2013

					\param		rhs_	Righthand side of the <= operator; the angle to compare against
					\returns	true if the target angle's value is less than or equal to the value for rhs_, once their
								ranges are restricted to [0,2pi]
				*/
				virtual bool operator<=(const Angle& rhs_) const;
				
				/**
					\brief		Checks if the target angle's value is greater than the provided angle's
					\author		GW
					\date		08/2013

					\param		rhs_	Righthand side of the > operator; the angle to compare against
					\returns	true if the target angle's value is greater than the value for rhs_, once their ranges
								are restricted to [0,2pi]
				*/
				virtual bool operator>(const Angle& rhs_) const;
				
				/**
					\brief		Checks if the target angle's value is greater than or equal to the provided angle's
					\author		GW
					\date		08/2013

					\param		rhs_	Righthand side of the >= operator; the angle to compare against
					\returns	true if the target angle's value is greater than or equal to the value for rhs_, once
								their ranges are restricted to [0,2pi]
				*/
				virtual bool operator>=(const Angle& rhs_) const;
				
				/**
					\brief		Flips the sign of the target angle
					\author		GW
					\date		08/2013

					\returns	A copy of the target Angle object, but with the sign of its value flipped
				*/
				virtual Angle operator-() const;

				/**
					\brief		Allows two Angle objects to be added together with the + operator
					\author		GW
					\date		08/2013

					\param		rhs_	Righthand side of the + operator; the angle to add to the target
					\returns	A copy of the target angle, but with the value of rhs_ added to it
				*/
				virtual Angle operator+(const Angle& rhs_) const;

				/**
					\brief		Allows one Angle object to be substracted from another with the - operator
					\author		GW
					\date		08/2013

					\param		rhs_	Righthand side of the - operator; the angle to subtract from the target
					\returns	A copy of the target angle, but with the value of rhs_ subtracted from it
				*/
				virtual Angle operator-(const Angle& rhs_) const;

				/**
					\brief		Allows two Angle objects to be multipled together with the * operator
					\author		GW
					\date		08/2013

					\param		rhs_	Righthand side of the * operator; the angle to multiply by the target
					\returns	A copy of the target angle, but with the value of rhs_ multiplied by it
				*/
				virtual Angle operator*(const Angle& rhs_) const;

				/**
					\brief		Allows one Angle object to be divided by another with the / operator
					\author		GW
					\date		08/2013

					\param		rhs_	Righthand side of the / operator; the angle to divide into the target
					\returns	A copy of the target angle, but with the value of rhs_ divided into it
				*/
				virtual Angle operator/(const Angle& rhs_) const;

				/**
					\brief		Allows one Angle object to be assigned from another
					\author		GW
					\date		08/2013

					\param		rhs_	Righthand side of the = operator; the object to copy a value from
					\returns	The target Angle object, but with its value updated to match that of the rhs_ argument
				*/
				virtual Angle& operator=(const Angle& rhs_);

				/**
					\brief		Allows two Angle objects to be added together with the + operator, assigning the
								lefthand object
					\author		GW
					\date		08/2013

					\param		rhs_	Righthand side of the + operator; the angle to add to the target
					\returns	The target Angle object, but with the value of rhs_ added to it
				*/
				virtual Angle& operator+=(const Angle& rhs_);

				/**
					\brief		Allows one Angle object to be substracted from another with the - operator, assigning
								the lefthand object
					\author		GW
					\date		08/2013

					\param		rhs_	Righthand side of the - operator; the angle to subtract from the target
					\returns	The target Angle object, but with the value of rhs_ subtracted from it
				*/
				virtual Angle& operator-=(const Angle& rhs_);

				/**
					\brief		Allows two Angle objects to be multipled together with the * operator, assigning the
								lefthand object
					\author		GW
					\date		08/2013

					\param		rhs_	Righthand side of the * operator; the angle to multiply by the target
					\returns	The target Angle object, but with the value of rhs_ multiplied by it
				*/
				virtual Angle& operator*=(const Angle& rhs_);

				/**
					\brief		Allows one Angle object to be divided by another with the / operator, assigning the
								lefthand object
					\author		GW
					\date		08/2013

					\param		rhs_	Righthand side of the / operator; the angle to divide into the target
					\returns	The target Angle object, but with the value of rhs_ divided into it
				*/
				virtual Angle& operator/=(const Angle& rhs_);

				/**
					\brief		Allows an Angle object to be multipled by a constant
					\author		GW
					\date		10/2013

					\param		rhs_	Righthand side of the * operator; the constant to multiply against the target
					\returns	A copy of the target angle, but with its value multiplied by the constant
				*/
				virtual Angle operator*(const double rhs_) const;

				/**
					\brief		Allows an Angle object to be divided by a constant
					\author		GW
					\date		10/2013

					\param		rhs_	Righthand side of the / operator; the constant to divide into the target
					\returns	A copy of the target angle, but with its value updated by dividing the constant into it
				*/
				virtual Angle operator/(const double rhs_) const;

				/**
					\brief		Allows an Angle object to be multipled by a constant, assigning the lefthand object
					\author		GW
					\date		10/2013

					\param		rhs_	Righthand side of the * operator; the constant to multiply against the target
					\returns	The target Angle object, but with its value multiplied by the constant
				*/
				virtual Angle& operator*=(const double rhs_);

				/**
					\brief		Allows an Angle object to be divided by a constant, assigning the lefthand object
					\author		GW
					\date		10/2013

					\param		rhs_	Righthand side of the / operator; the constant to divide into the target
					\returns	The target Angle object, but with its value updated by dividing the constant into it
				*/
				virtual Angle& operator/=(const double rhs_);

				/**
					\brief		Gets the value of the target object in radians
					\author		GW
					\date		08/2013

					\returns	The value of the target Angle object in radians
				*/
				virtual double asRadians() const;

				/**
					\brief		Gets the value of the target object in degrees
					\author		GW
					\date		08/2013

					\returns	The value of the target Angle object in degrees
				*/
				virtual double asDegrees() const;

				/**
					\brief		Gets the value of the target object in gradians
					\author		GW
					\date		08/2013

					\returns	The value of the target Angle object in gradians
				*/
				virtual double asGradians() const;

				/**
					\brief		Gets the value of the target object in turns
					\author		GW
					\date		08/2013

					\returns	The value of the target Angle object in turns
				*/
				virtual double asTurns() const;

				/**
					\brief		Gets the value of the target object in the given units
					\author		GW
					\date		08/2013

					\param		units_	Desired units for the return value
					\returns	The value of the target Angle object in the provided units
				*/
				virtual double valueInUnits(const Units units_ = Radians) const;

				/**
					\brief		Sets the value of the target object in the given units
					\author		GW
					\date		08/2013

					\param		value_	Value, in "units_" units, to set
					\param		units_	Units for the value_ argument
				*/
				virtual void setValueInUnits(const double value_, const Units units_ = Radians);

				/**
					\brief		Changes the range of the angle value to be in [0,2pi]
					\author		GW
					\date		08/2013

					\returns	An equivalent angle to the target, but forced within range [0,2pi]
				*/
				virtual Angle range0To2Pi() const;

				/**
					\brief		Changes the range of the angle value to be in [-pi,pi]
					\author		GW
					\date		08/2013

					\returns	An equivalent angle to the target, but forced within range [-pi,pi]
				*/
				virtual Angle rangeNegPiToPi() const;

				/**
					\brief		Constrains an angle to the nearest 45-degree (pi/4) interval
					\author		GW
					\date		08/2013

					\returns	The nearest 45-degree interval for the target Angle object
				*/
				virtual Angle nearest45() const;

				/**
					\brief		Constrains the target Angle object's value to the range [0,pi]
					\author		GW
					\date		08/2013

					\returns	A copy of the target Angle object, but with its value constrained to [0,pi]
					
					\note		This method essentially takes any angle in the range (pi,2pi) and "flips" it across the
								polar coordinate system to point in the opposite direction; i.e. 3pi/2 becomes pi/2.
				*/
				virtual Angle constrain0ToPi() const;
				
				/**
					\brief		Constrains the target Angle object's value to the range [pi/2,3pi/2]
					\author		GW
					\date		08/2013

					\returns	A copy of the target Angle object, but with its value constrained to [pi/2,3pi/2]
					
					\note		This method essentially takes any angle in the range (-pi/2,pi/2) and "flips" it across
								the polar coordinate system to point in the opposite direction; i.e. 0 becomes pi.
				*/
				virtual Angle constrainPiOver2To3PiOver2() const;

				/**
					\brief		Constrains the target Angle object's value to the range [pi,2pi]
					\author		GW
					\date		08/2013

					\returns	A copy of the target Angle object, but with its value constrained to [pi,2pi]
					
					\note		This method essentially takes any angle in the range (0,pi) and "flips" it across the
								polar coordinate system to point in the opposite direction; i.e. pi/2 becomes 3pi/2.
				*/
				virtual Angle constrainPiTo2Pi() const;
				
				/**
					\brief		Constrains the target Angle object's value to the range [-pi/2,pi/2]
					\author		GW
					\date		08/2013

					\returns	A copy of the target Angle object, but with its value constrained to [-pi/2,pi/2]
					
					\note		This method essentially takes any angle in the range (pi/2,3pi/2) and "flips" it across
								the polar coordinate system to point in the opposite direction; i.e. pi becomes 0.
				*/
				virtual Angle constrainNegPiOver2ToPiOver2() const;
				
				/**
					\brief		Gets the bisection of the target Angle with a provided Angle on their most acute "side"
					\author		GW
					\date		08/2013

					\param		angle2_		Second angle
					\returns	An Angle object forming the acute bisection of the target and the provided Angle
				*/
				virtual Angle bisect(const Angle& angle2_) const;
		
				/**
					\brief		Gets the delta between the target Angle and a provided Angle object
					\author		GW
					\date		08/2013

					\param		angle2_		Second angle
					\returns	An Angle object representing the delta between the target and the provided Angle

					\note		This method always returns the smallest of the two possible deltas; if you want the
								larger, subtract the returned Angle from 2*pi
				*/
				virtual Angle delta(const Angle& angle2_) const;

				/**
					\brief		Tests if the target Angle lies in the first quadrant, were it from point (0,0)
					\author		GW
					\date		08/2013
					
					\returns	true if the Angle is in quadrant one, false otherwise
				*/
				virtual bool firstQuadrant() const;

				/**
					\brief		Tests if the target Angle lies in the second quadrant, were it from point (0,0)
					\author		GW
					\date		08/2013
					
					\returns	true if the Angle is in quadrant two, false otherwise
				*/
				virtual bool secondQuadrant() const;

				/**
					\brief		Tests if the target Angle lies in the third quadrant, were it from point (0,0)
					\author		GW
					\date		08/2013
					
					\returns	true if the Angle is in quadrant three, false otherwise
				*/
				virtual bool thirdQuadrant() const;

				/**
					\brief		Tests if the target Angle lies in the fourth quadrant, were it from point (0,0)
					\author		GW
					\date		08/2013
					
					\returns	true if the Angle is in quadrant four, false otherwise
				*/
				virtual bool fourthQuadrant() const;

				/**
					\brief		Gets the quadrant number for the target Angle, were it from point (0,0)
					\author		GW
					\date		08/2013

					\returns	The quadrant number for the angle, or 0 for error (which should never happen...)
				*/
				virtual int16_t quadrantNumber() const;

				/**
					\brief		"Safely" computes the sine of an angle
					\author		GW
					\date		08/2013

					\param		tol_	Tolerance around zero, whereupon the returned value will be rounded to 0.0
					\returns	Sine of the target angle, modified to 0.0 exactly if original value is within tolerance

					\note		Due to floating-point precision limitations, it can be pretty bad if a sine value is
								slightly off from zero. For example, an "erroneously" negative value can cause program 
								errors due to having a negative sign (instead of positive). If a value is very close to
								zero, it is simply converted to exactly 0.0.
				*/
				virtual double sin(const double tol_ = 10000.0 * maxDoublePrecision) const;

				/**
					\brief		"Safely" computes the cosine of an angle
					\author		GW
					\date		08/2013

					\param		tol_	Tolerance around zero, whereupon the returned value will be rounded to 0.0
					\returns	Cosine of angle_ argument, modified to 0.0 exactly if original value is within tolerance

					\note		Due to floating-point precision limitations, it can be pretty bad if a cosine value is
								slightly off from zero. For example, an "erroneously" negative value can cause program 
								errors due to having a negative sign (instead of positive). If a value is very close to
								zero, it is simply converted to exactly 0.0.
				*/
				virtual double cos(const double tol_ = 10000.0 * maxDoublePrecision) const;

				/**
					\brief		"Safely" computes the tangent of an angle
					\author		GW
					\date		08/2013

					\param		tol_	Tolerance around zero, whereupon the returned value will be rounded to 0.0
					\returns	Tangent of angle_ argument, modified to 0.0 exactly if original value is within tolerance

					\note		Due to floating-point precision limitations, it can be pretty bad if a tangent value is
								slightly off from zero. For example, an "erroneously" negative value can cause program 
								errors due to having a negative sign (instead of positive). If a value is very close to
								zero, it is simply converted to exactly 0.0.
				*/
				virtual double tan(const double tol_ = 10000.0 * maxDoublePrecision) const;

				/**
					\brief		"Safely" computes the angle for a given sine value
					\author		GW
					\date		08/2013

					\param		val_	Sine value, whose corresponding angle is required
					\param		tol_	Tolerance around zero, whereupon the returned value will be rounded to 0.0
					\returns	Arc sine of val_ argument, modified to 0.0 exactly if original value is within tol_

					\note		Due to floating-point precision limitations, it can be pretty bad if a sine value is
								slightly off from zero. For example, an "erroneously" negative value can cause program 
								errors due to having a negative sign (instead of positive). If a value is very close to
								zero, it is simply converted to exactly 0.0.
				*/
				static Angle asin(const double val_, const double tol_ = 10000.0 * maxDoublePrecision);

				/**
					\brief		"Safely" computes the angle for a given cosine value
					\author		GW
					\date		08/2013

					\param		val_	Cosine value, whose corresponding angle is required
					\param		tol_	Tolerance around zero, whereupon the returned value will be rounded to 0.0
					\returns	Arc cosine of val_ argument, modified to 0.0 exactly if original value is within tol_

					\note		Due to floating-point precision limitations, it can be pretty bad if a cosine value is
								slightly off from zero. For example, an "erroneously" negative value can cause program 
								errors due to having a negative sign (instead of positive). If a value is very close to
								zero, it is simply converted to exactly 0.0.
				*/
				static Angle acos(const double val_, const double tol_ = 10000.0 * maxDoublePrecision);

				/**
					\brief		"Safely" computes the angle for a given tangent value
					\author		GW
					\date		08/2013

					\param		val_	Tangent value, whose corresponding angle is required
					\param		tol_	Tolerance around zero, whereupon the returned value will be rounded to 0.0
					\returns	Arc tangent of val_ argument, modified to 0.0 exactly if original value is within tol_

					\note		Due to floating-point precision limitations, it can be pretty bad if a tangent value is
								slightly off from zero. For example, an "erroneously" negative value can cause program 
								errors due to having a negative sign (instead of positive). If a value is very close to
								zero, it is simply converted to exactly 0.0.
				*/
				static Angle atan(const double val_, const double tol_ = 10000.0 * maxDoublePrecision);
				
				/**
					\brief		Convenience method to construct an Angle object for zero radians, degrees, etc.
					\author		GW
					\date		08/2013
					
					\returns	An Angle object for 0.0 radians, degrees, etc.
				*/
				static Angle Zero();
				
				/**
					\brief		Convenience method to construct an Angle object for pi radians
					\author		GW
					\date		08/2013
					
					\returns	An Angle object for pi radians
				*/
				static Angle Pi();
				
				/**
					\brief		Convenience method to construct an Angle object for 2*pi radians
					\author		GW
					\date		08/2013
					
					\returns	An Angle object for 2*pi radians
				*/
				static Angle TwoPi();


			private:
				friend geo::Angle* __accessImpl(const Angle&);
				friend Angle __accessCtor(const geo::Angle&);

				/**
					\brief	Private implementation object
				*/
				geo::Angle* __impl;
				
				/**
					\brief	Internal use only
					\author	GW
					\date	08/2013
				*/
				Angle(const geo::Angle&);
		};
		
		typedef std::unique_ptr<Angle> AngleUP;
		typedef std::shared_ptr<Angle> AngleSP;
		typedef std::weak_ptr<Angle> AngleWP;
		
		extern geo::Angle* __accessImpl(const Angle&);
		extern Angle __accessCtor(const geo::Angle&);
	}
}

/**
	\brief		Allows a constant to be multipled by an Angle object
	\author		GW
	\date		10/2013

	\param		lhs_	Lefthand side of the * operator; the constant to multiply against the angle
	\param		rhs_	Righthand side of the * operator; the angle to multiply against the constant
	\returns	A copy of rhs_, but with its value multiplied by the constant
*/
hdi::core::Angle operator*(const double lhs_, const hdi::core::Angle& rhs_);

#endif
// __HDI_CORE_ANGLE__
