/**
	\file
	\brief		Header file for general Illustrator art manipulation
	\copyright	Hot Door, Inc. 2010-2025
*/

#ifndef __HDI_CORE_ART__
#define __HDI_CORE_ART__

#if defined(HDI_CORE_AIP_MODE)

#include "hdicoreArtboardRect.h"
#include "hdicoreTypes.h"

namespace hdi
{
	namespace aip
	{
		class Art;
	}

	namespace core
	{
		namespace ai
		{
			class Dictionary;
		}
		
		class Artboard;
		class ArtStyle;
		class CustomArt;
		class Layer;
		class MaskArt;
		class MeshArt;
		class PathArt;
		class Pattern;
		class PlacedArt;
		class RasterArt;
		class Symbol;
		class TextArt;
		class UID;
	
		/**
			\brief		Handles general art-related functionality
			\details	This class handles general art functionality (e.g. tranforms, metadata, duplication, reordering,
						siblings, etc.); more specific actions can be taken on specific art types (e.g. paths, text, etc.)
						by accessing the appropriate method in this class (e.g. path(), text(), etc.), but be sure to
						verify the type of the art first by calling artType()
		*/
		class Art
		{
			public:
				typedef std::vector< std::shared_ptr<Artboard> > ArtboardVector;
				typedef std::vector< std::shared_ptr<Art> > ArtVector;
				typedef std::vector<MatchArtSpec> MatchArtSpecVector;
			
				/**
					\brief	Allows for querying various attributes of some art's bounds
				*/
				enum BoundsOptions
				{
					// Includes hidden and unpainted objects if they are not guides. Ignores direction handles.
					VisibleBounds					= 1 << 0,
					
					// When off, NoStrokeBounds, NoExtendedBounds, and ExcludeUnpaintedObjectBounds can be combined to
					// ignore certain aspects of the visual bounds.
					ControlBounds					= 1 << 1,
					
					// Strokes, effects, and other appearance attributes that may make an object extend past the bezier
					// outlines are ignored.
					NoStrokeBounds					= 1 << 2,
					
					// Implies NoStrokeBounds, but additionally excludes text characters for text on a path or area
					// text, measuring only the path.
					NoExtendedBounds				= 1 << 3,
					
					// Use the setting of the "Use Preview Bounds" user preference to control whether the strokes and
					// other appearance attributes are included. Cannot be used with NoStrokeBounds (NoStrokeBounds will
					// take precedence).
					StrokeDependsOnPref				= 1 << 4,
					
					// Exclude hidden objects. Can be set with ControlBounds or VisibleBounds.
					ExcludeHiddenObjectBounds		= 1 << 5,
					
					// Excludes objects with no fill and no stroke.
					ExcludeUnpaintedObjectBounds	= 1 << 6,
					
					// Excludes guides. Can be set with ControlBounds or VisibleBounds.
					ExcludeGuideBounds				= 1 << 7
				};
				
				/**
					\brief	Controls how art should be expanded by the expand() method
				*/
				enum ExpandOptions
				{
					ExpandPluginArt			= 1 << 0,
					ExpandText				= 1 << 1,
					ExpandStroke			= 1 << 2,
					ExpandPattern			= 1 << 3,
					ExpandGradientToMesh	= 1 << 4,
					ExpandGradientToPaths	= 1 << 5,
					ExpandSymbolInstances	= 1 << 6,
					ExpandOneAtATime		= 1 << 7,
					ExpandShowProgressBar	= 1 << 8,
					ExpandLockedObjects		= 1 << 9
				};
				
				/**
					\brief	Controls how art should be transformed by the transform() method
				*/
				enum TransformOptions
				{
					// No transform
					TransformOptionNone				= 0,
					
					// Apply transform to art objects (almost always used)
					TransformOptionObjects			= 1 << 0,
					
					// Apply transform to gradient fills
					TransformOptionFillGradients	= 1 << 1,
					
					// Apply transform to pattern fills
					TransformOptionFillPatterns		= 1 << 2,
					
					// Apply transform to gradient strokes
					TransformOptionStrokeGradients	= 1 << 3,
					
					// Apply transform to pattern strokes
					TransformOptionStrokePatterns	= 1 << 4,
					
					// Apply transform to any opacity masks (if they are set as linked)
					TransformOptionLinkedMasks		= 1 << 5,
					
					// Apply the transformation recursively
					TransformOptionChildren			= 1 << 6,
					
					// Apply the transformation only to selected segments of path art
					TransformOptionSelectedSegsOnly	= 1 << 7
				};

				/**
					\brief	Allows for iteration through the siblings of a piece of source art, a la std::vector iteration

					\code
						// Here's an example, iterating through the children of someGroup:
						hdi::core::ArtAP child0 = someGroup->firstChild();
						for(hdi::core::Art::Iterator iter = child0->begin(); !iter.atEnd(); ++iter)
						{
							// do stuff to iter.art()
						}

						// Reverse iteration is similar:
						hdi::core::ArtAP childN = someGroup->lastChild();
						for(hdi::core::Art::Iterator iter = childN->begin(); !iter.atEnd(); --iter)
						{
							// do stuff to iter.art()
						}
					\endcode
				*/
				class Iterator
				{
					public:
						/**
							\brief	Constructs an Iterator object from an existing Iterator object (copy constructor)
							\author	GW
							\date	10/2013

							\param	iter_	Existing Iterator object
						*/
						Iterator(const Iterator& iter_);
					
						/**
							\brief	Destructs an Iterator object
							\author	GW
							\date	08/2013
						*/
						~Iterator();
						
						/**
							\brief		Assigns one Iterator object to another
							\author		GW
							\date		10/2013

							\param		rhs_	Existing Iterator object to copy values from
							\returns	The target Iterator object, but with its values updated to match that of the
										rhs_ argument
						*/
						Iterator& operator=(const Iterator& rhs_);
						
						/**
							\brief		Tests whether a given Iterator object is the same as another
							\author		GW
							\date		10/2013

							\param		rhs_	Iterator to compare against (righthand side of equality operator)
							\returns	true for the target and rhs_ having the same state, false otherwise
						*/
						bool operator==(const Iterator& rhs_) const;
						
						/**
							\brief		Tests whether a given Iterator object is not the same as another
							\author		GW
							\date		10/2013

							\param		rhs_	Iterator to compare against (righthand side of inequality operator)
							\returns	true for the target and rhs_ having differing states, false otherwise
						*/
						bool operator!=(const Iterator& rhs_) const;

						/**
							\brief		Prefix increment operator for forward iteration
							\author		GW
							\date		08/2013
							
							\returns	A reference to the iterator after its position was changed
						*/
						Iterator& operator++();

						/**
							\brief		Postfix increment operator for forward iteration
							\author		GW
							\date		08/2013
							
							\returns	A copy of the original iterator before its position was changed
						*/
						Iterator operator++(int);

						/**
							\brief		Prefix decrement operator for reverse iteration
							\author		GW
							\date		08/2013
							
							\returns	A reference to the iterator after its position was changed
						*/
						Iterator& operator--();

						/**
							\brief		Postfix decrement operator for reverse iteration
							\author		GW
							\date		08/2013
							
							\returns	A copy of the original iterator before its position was changed
						*/
						Iterator operator--(int);

						/**
							\brief		Determines whether iteration should stop; works well as the conditional in a
										for() loop
							\author		GW
							\date		08/2013
							
							\returns	true if at the end of the list of siblings, false otherwise
						*/
						bool atEnd() const;

						/**
							\brief		Gets the current art pointed to by the iterator
							\author		GW
							\date		08/2013

							\returns	The current piece of art, or an empty Art object if at the end (see default Art
										constructor for more information)
						*/
						std::unique_ptr<Art> art() const;


					private:
						// Only Art can construct a new Iterator object
						friend class Art;

						/**
							\brief	Private implementation data
						*/
						void* __data;
				
						/**
							\brief	Empty Iterator objects cannot be constructed
							\author	GW
							\date	08/2013
						*/
						Iterator();
					
						/**
							\brief	Creates a new iterator that starts at some target art
							\author	GW
							\date	08/2013

							\param	art_	Initial art whose parent group will be iterated through; iteration starts at
											art_
						*/
						explicit Iterator(const Art& art_);
				};
		
				typedef std::unique_ptr<Iterator> IteratorUP;
				typedef std::shared_ptr<Iterator> IteratorSP;
				typedef std::weak_ptr<Iterator> IteratorWP;
			
				/**
					\brief	Constructs an empty Art object
					\author	GW
					\date	08/2013
					
					\note	To test if an Art object is empty, call isEmpty() on it
					\note	Empty Art objects do not relate to any art on the Illustrator artboard; they are designed to
							be "receivers" of some other Art object via the overloaded assignment operator. Empty Art
							objects are useless until such time (though it is safe to call any of their methods)
				*/
				Art();
			
				/**
					\brief	Constructs a new Art object from an existing Art object (copy constructor)
					\author	GW
					\date	08/2013

					\param	art_	Existing Art object
				*/
				Art(const Art& art_);
				
				/**
					\brief	Constructs a new Art object from an AIArtHandle
					\author	GW
					\date	08/2015
					
					\param	aiArt_	Raw Illustrator AIArtHandle
				*/
				Art(const AIArtHandle aiArt_);
				
				/**
					\brief		Constructs a new Art object from an AISafeArtHandle
					\author		GW
					\date		07/2017
					
					\param		aiArt_	Raw Illustrator AISafeArtHandle
				*/
				Art(const AISafeArtHandle aiArt_);
		
				/**
					\brief		Constructs a new Art object from a UUID string
					\author		GW
					\date		07/2017
					
					\param		uuid_	Raw Illustrator art UUID
				*/
				Art(const std::string& uuid_);

				/**
					\brief		Constructs a new Art object for a newly created piece of Illustrator art
					\author		GW
					\date		08/2013

					\param		type_	Type of art to create
					\param		order_	Order in which to create the art
					\param		prep_	Prepositional piece of art to which order_ will apply (or NULL if not needed)
					
					\warning	This method does not assume responsibility of the memory for the prep_ argument.
				*/
				Art(const ArtType type_, const PaintOrder order_, const Art* const prep_ = NULL);
				
				/**
					\brief		Constructs a new Art object for a newly created piece of Illustrator text art (type of
								ArtTypeText)
					\author		GW
					\date		09/2014

					\param		anchor_	Start point of the baseline of the first character
					\param		dir_	Text orientation direction
					\param		order_	Order in which to create the art
					\param		prep_	Prepositional piece of art to which order_ will apply (or NULL if not needed)
					
					\warning	This method does not assume responsibility of the memory for the prep_ argument.
				*/
				Art(
					const ArtboardPoint& anchor_,
					const TextOrientation dir_,
					const PaintOrder order_,
					const Art* const prep_ = NULL
				);

				/**
					\brief	Art object destructor
					\author	GW
					\date	08/2013
				*/
				virtual ~Art();
				
				/**
					\brief		Assigns one Art object to another
					\author		GW
					\date		08/2013

					\param		rhs_	Existing Art object to copy values from
					\returns	The target Art object, but with its values updated to match that of the rhs_ argument
				*/
				virtual Art& operator=(const Art& rhs_);

				/**
					\brief		Removes the art from the artboard
					\author		GW
					\date		08/2013
					
					\returns	true for successful disposal, false otherwise

					\note		After calling dispose(), this object is basically useless! The actual art is gone, so
								obviously none of these methods will work after that.
				*/
				virtual bool dispose();

				/**
					\brief		Removes all the children art from the parent group or compound path
					\author		GW
					\date		07/2017
					
					\returns	true for successful disposal, false otherwise
				*/
				virtual bool disposeChildren();

				/**
					\brief		Duplicates the target piece of art on the artboard
					\author		GW
					\date		08/2013

					\param		order_	Order in which to create the art
					\param		prep_	Prepositional piece of art to which order_ will apply (or NULL if not needed)
					\returns	Art object representing the duplicated piece of art, or NULL for none

					\warning	This method does not assume responsibility of the memory for the prep_ argument.
				*/
				virtual std::unique_ptr<Art> duplicate(const PaintOrder order_, const Art* const prep_ = NULL) const;

				/**
					\brief		Gets the art handle around which the target object is wrapped
					\author		GW
					\date		11/2013
					
					\returns	AIArtHandle for this object

					\note		Generally, you shouldn't need to call this method. Only use it if you know what you're
								doing. If a specific piece of functionality provided by Illustrator is not handled by
								this class (or related classes), then it should probably be added to the hdi_core library.
				*/
				virtual AIArtHandle aiArtHandle() const;
				
				/**
					\brief		Gets the "safe" art handle around which the target object is wrapped
					\author		GW
					\date		07/2017
					
					\returns	AISafeArtHandle for this object
					
					\note		Generally, you shouldn't need to call this method. Only use it if you know what you're
								doing. If a specific piece of functionality provided by Illustrator is not handled by
								this class (or related classes), then it should probably be added to the library.
				*/
				virtual AISafeArtHandle aiSafeArtHandle() const;
		
				/**
					\brief		Accessor for the UUID string for the Art object
					\author		GW
					\date		07/2017
					
					\returns	The object's UUID string, or "" for none/error
				*/
				virtual std::string uuid() const;

				/**
					\brief		Gets whether the target Art object is empty (constructed with the default ctor)
					\author		GW
					\date		08/2013
					
					\returns	true if the target Art object is empty, false otherwise
				*/
				virtual bool isEmpty() const;
				
				/**
					\brief		Tests the validity of the target piece of art by searching for it in the document
					\author		GW
					\date		08/2013
					
					\returns	true for valid art, false otherwise
				*/
				virtual bool valid() const;
				
				/**
					\brief		Tests whether a given Art object is the same as another
					\author		GW
					\date		08/2013

					\param		rhs_	Art to compare against (righthand side of equality operator)
					\returns	true for the target and rhs_ being the same piece of art, false otherwise
				*/
				virtual bool operator==(const Art& rhs_) const;
				
				/**
					\brief		Tests whether a given Art object is not the same as another
					\author		GW
					\date		08/2013

					\param		rhs_	Art to compare against (righthand side of inequality operator)
					\returns	true for the target and rhs_ being different pieces of art, false otherwise
				*/
				virtual bool operator!=(const Art& rhs_) const;

				/**
					\brief		Gets the type of the art
					\author		GW
					\date		08/2013
					
					\returns	The type of the target art, or ArtTypeUnknown if the art is invalid
				*/
				virtual ArtType artType() const;
				
				/**
					\brief		Gets the UID for the target art
					\author		GW
					\date		08/2013
					
					\returns	The target art's UID object, from which the target piece of art can be uniquely acquired
								at a later time (provided it exists)
				*/
				virtual std::unique_ptr<UID> uid() const;

				/**
					\brief		Gets a CustomArt object for this Art object
					\author		GW
					\date		07/2017
					
					\returns	A CustomArt object unique to this piece of art, for further manipulating custom user-editable
								and user-viewable attributes of the art

					\note		The CustomArt class has all custom art-related methods.
				*/
				virtual CustomArt* const customArt() const;

				/**
					\brief		Gets a MaskArt object for this Art object
					\author		GW
					\date		08/2014
					
					\returns	A MaskArt object unique to this piece of art, for further manipulating mask-related
								attributes of the art

					\note		The MaskArt class has all mask-related methods.
				*/
				virtual MaskArt* const mask() const;

				/**
					\brief		Gets a MeshArt object for this Art object
					\author		GW
					\date		08/2014
					
					\returns	A MeshArt object unique to this piece of art, for further manipulating mesh-related
								attributes of the art

					\note		The MeshArt class has all mesh-related methods.
				 	\note		This method is "dumb" in that it will return a MeshArt object even if the target art is
								not really a mesh. It's up to the caller to check the type before blindly calling this
								method.
				*/
				virtual MeshArt* const mesh() const;

				/**
					\brief		Gets a PathArt object for this Art object
					\author		GW
					\date		08/2013
					
					\returns	A PathArt object unique to this piece of art, for further manipulating path-related
								attributes of the art

					\note		The PathArt class has all path-related methods, e.g. segment counts, path length, path
								intersection detection, etc.
				 	\note		This method is "dumb" in that it will return a PathArt object even if the target art is
								not really a path. It's up to the caller to check the type before blindly calling this
								method.
				*/
				virtual PathArt* const path() const;

				/**
					\brief		Gets a PlacedArt object for this Art object
					\author		GW
					\date		08/2014
					
					\returns	A PlacedArt object unique to this piece of art, for further manipulating placed art-
								related attributes of the art

					\note		The PlacedArt class has all placed art-related methods.
				 	\note		This method is "dumb" in that it will return a PlacedArt object even if the target art
								is not really placed art. It's up to the caller to check the type before blindly calling
								this method.
				*/
				virtual PlacedArt* const placed() const;

				/**
					\brief		Gets a RasterArt object for this Art object
					\author		GW
					\date		08/2014
					
					\returns	A RasterArt object unique to this piece of art, for further manipulating raster-related
								attributes of the art

					\note		The RasterArt class has all path-related methods.
				 	\note		This method is "dumb" in that it will return a RasterArt object even if the target art
								is not really raster art. It's up to the caller to check the type before blindly calling
								this method.
				*/
				virtual RasterArt* const raster() const;

				/**
					\brief		Gets a TextArt object for this Art object
					\author		GW
					\date		08/2013
					
					\returns	A TextArt object unique to this piece of art, for further manipulating text-related
								attributes of the art

					\note		The TextArt class has all text-related methods, e.g. text type, orientation, text
								ranges, etc.
					\note		This method is "dumb" in that it will return a TextArt object even if the target art is
								not really a text frame. It's up to the caller to check the type before blindly calling
								this method.
				*/
				virtual TextArt* const text() const;

				/**
					\brief	Marks an Art object for redrawing by incrementing its internal modification time stamp
					\author	GW
					\date	11/2016
				*/
				virtual void markDirty();

				/**
					\brief	Marks an Art object that previously would have been redrawn as NOT needing redrawing
					\author	GW
					\date	07/2017
				*/
				virtual void markClean();

				/**
					\brief	Sets the art object and all its children (in the case of a container) to be selected
					\author	GW
					\date	08/2013

					\note	Illustrator doesn't always select art properly, and generally seems to have trouble
							selecting during mouse drag and track events. If you need to select some newly created art,
							it's best to do it in response to a mouse up event.
					\note	Sometimes calling sync() on the current document can aid in art selection woes when
							Illustrator "misbehaves".
				*/
				virtual void select();

				/**
					\brief	Sets the art object and all its children (in the case of a container) to be deselected
					\author	GW
					\date	08/2013

					\note	Illustrator doesn't always deselect art properly, and generally seems to have trouble
							deselecting art that has had select() called on it.
					\note	Sometimes calling sync() on the current document can aid in art deselection woes when
							Illustrator "misbehaves".
				*/
				virtual void deselect();
				
				/**
					\brief		Gets whether the art is selected
					\author		GW
					\date		08/2013
					
					\returns	true for the art being selected, false otherwise
					
					\note		Container art (e.g. a group) is considered to be selected even when only one child is
								selected. To determine if a container is fully selected, call fullySelected() instead.
				*/
				virtual bool selected() const;
				
				/**
					\brief		Gets whether the art is fully selected
					\author		GW
					\date		08/2013
					
					\returns	true for the art being fully selected, false otherwise
					
					\note		Container art (e.g. a group) is considered to be fully selected when all its children
								are selected. Other types of art will always return true for fullySelected() if the
								selected() method returns true, and vice versa.
				*/
				virtual bool fullySelected() const;

				/**
					\brief	Sets whether the art is selected
					\author	GW
					\date	08/2013

					\param	select_		true to select, false otherwise

					\note	This method simply calls select() or deselect() as appropriate
					\note	Container art (e.g. a group) will be fully selected, meaning that all its children will also
							be selected.
				*/
				virtual void setSelected(const bool select_);
				
				/**
					\brief		Gets whether the art is editable
					\author		GW
					\date		08/2013
					
					\returns	true for the art being editable (unlocked), false otherwise
					
					\note		Always returns the opposite of locked() method.
				*/
				virtual bool editable() const;

				/**
					\brief	Sets whether the art is editable
					\author	GW
					\date	08/2013

					\param	editable_	true to allow editing, false otherwise

					\note	Always sets the art to be locked.
				*/
				virtual void setEditable(const bool editable_);
				
				/**
					\brief		Gets whether the art is locked
					\author		GW
					\date		08/2013
					
					\returns	true for the art being locked (uneditable), false otherwise
					
					\note		Always returns the opposite of editable() method.
				*/
				virtual bool locked() const;

				/**
					\brief	Sets whether the art is locked
					\author	GW
					\date	08/2013

					\param	lock_	true to lock, false otherwise

					\note	Always sets the art to be uneditable.
				*/
				virtual void setLocked(const bool lock_);
				
				/**
					\brief		Gets whether the art is hidden
					\author		GW
					\date		08/2013
					
					\returns	true for art being hidden, false otherwise
				*/
				virtual bool hidden() const;

				/**
					\brief	Sets whether the art is hidden
					\author	GW
					\date	08/2013

					\param	hidden_		true to hide, false otherwise
				*/
				virtual void setHidden(const bool hidden_);
				
				/**
					\brief		Gets whether any ancestor of the art is selected
					\author		GW
					\date		08/2013
					
					\returns	true for the art being selected, false otherwise
				*/
				virtual bool anyAncestorSelected() const;
				
				/**
					\brief		Gets whether any ancestor of the art is locked
					\author		GW
					\date		08/2013
					
					\returns	true for the art being locked (uneditable), false otherwise
				*/
				virtual bool anyAncestorLocked() const;
				
				/**
					\brief		Gets whether any ancestor of the art is hidden
					\author		GW
					\date		08/2013
					
					\returns	true for art being hidden, false otherwise
				*/
				virtual bool anyAncestorHidden() const;

				/**
					\brief		Gets a bitwise OR'd enum value of all currently set art attributes
					\author		GW
					\date		12/2015
					
					\returns	All art attributes that are currently true bitwise OR'd into a single enum value
				*/
				virtual ArtAttribute allAttributes() const;

				/**
					\brief	Sets a bitwise OR'd enum value for all art attributes
					\author	GW
					\date	12/2015
					
					\param	attrs_	All art attributes that should be true bitwise OR'd into a single enum value
				*/
				virtual void setAllAttributes(const ArtAttribute attrs_);

				/**
					\brief		Gets whether a specific art attribute is true
					\author		GW
					\date		08/2013

					\param		attr_	The attribute of interest
					\returns	true if the provided attribute is enabled/true/set, false otherwise
				*/
				virtual bool attribute(const ArtAttribute attr_) const;

				/**
					\brief	Sets an art attribute for the target art (e.g. hidden, locked, etc.)
					\author	GW
					\date	08/2013

					\param	attr_		The attribute of interest
					\param	enable_		Whether to enable the attribute (e.g. true for ArtAttrSelected to make it
										selected, false to deselect)
				*/
				virtual void setAttribute(const ArtAttribute attr_, const bool enable_);

				/**
					\brief		Gets the built-in art data dictionary object for the target art
					\author		GW
					\date		08/2013
					
					\returns	A Dictionary object for the art's unique data dictionary
				*/
				virtual std::unique_ptr<ai::Dictionary> dictionary() const;

				/**
					\brief		Gets the bounds rect for the target art, including the stroke
					\details	This is a convenience method for calling bounds(VisibleBounds | ExcludeGuideBounds)
					\author		GW
					\date		08/2013
					
					\returns	A rect object describing the art's bounds, including the stroke width
				*/
				virtual ArtboardRect strokeBounds() const;

				/**
					\brief		Gets the bounds rect for the target art, excluding the stroke
					\details	This is a convenience method for calling
								bounds(VisibleBounds | NoStrokeBounds | ExcludeGuideBounds)
					\author		GW
					\date		08/2013
					
					\returns	A rect object describing the art's bounds, excluding the stroke width
				*/
				virtual ArtboardRect noStrokeBounds() const;

				/**
					\brief		Gets the bounds rect for the target art
					\author		GW
					\date		08/2013
					
					\param		opts_	A bitwise OR of the bounds in question; see BoundsOptions
					\returns	A rect object describing the art's bounds, taking the provided options into account
				*/
				virtual ArtboardRect bounds(const BoundsOptions opts_) const;

				/**
					\brief	Updates the bounds rect for the target art by updating Illustrator caches (regarding
							bounding boxes)
					\author	GW
					\date	08/2013
				*/
				virtual void updateBounds();

				/**
					\brief		Gets the bounds that would be the result of a transform
					\author		GW
					\date		08/2013

					\param		transform_	Transform to "virtually" perform before acquiring bounds
					\param		opts_		Flags for which attributes to take into consideration during bounds
											calculations (bitwise OR of BoundsOptions enum)
					\returns	A rect object describing the art's bounds, were it to be transformed by the given matrix
				*/
				virtual ArtboardRect transformBounds(const TransformMatrix& transform_, const BoundsOptions opts_) const;

				/**
					\brief		Checks whether any linked objects (linked images or placed objects) contained in the
								subtree of a given container need updating, and updates them if needed.
					\author		GW
					\date		01/2016
					
					\param		force_	Whether to force updating of objects regardless of whether they have changed
					\returns	true if any objects were updated, false otherwise
				*/
				virtual bool updateLinks(const bool force_ = false);

				/**
					\brief		Gets the top-left x-value of the art
					\author		GW
					\date		08/2013
					
					\returns	The x-position of the art, excluding the stroke
				*/
				virtual double x() const;

				/**
					\brief		Gets the top-left y-value of the art
					\author		GW
					\date		08/2013
					
					\returns	The y-position of the art, excluding the stroke
				*/
				virtual double y() const;

				/**
					\brief		Gets the width of the bounds of the art
					\author		GW
					\date		08/2013
					
					\returns	The width of the art, excluding the stroke
				*/
				virtual double width() const;

				/**
					\brief		Gets the height of the bounds of the art
					\author		GW
					\date		08/2013
					
					\returns	The height of the art, excluding the stroke
				*/
				virtual double height() const;

				/**
					\brief		Gets the perimeter of the art
					\author		GW
					\date		08/2013
					
					\returns	The perimeter of the art, excluding the stroke
				*/
				virtual double perimeter() const;

				/**
					\brief		Gets the area of the art
					\author		GW
					\date		08/2013
					
					\returns	The area of the art, in square points, excluding the stroke
				*/
				virtual double area() const;

				/**
					\brief		Gets whether the center point is visible for the target art
					\author		GW
					\date		08/2013
					
					\returns	true for a visible center point, false otherwise
				*/
				virtual bool centerPointVisible() const;

				/**
					\brief	Sets whether the center point is visible for the target art
					\author	GW
					\date	08/2013

					\param	visible_	Whether the center point is visible
				*/
				virtual void setCenterPointVisible(const bool visible_);

				/**
					\brief		Gets the "most overlapping" artboard for the target art
					\author		GW
					\date		08/2014
					
					\returns	The artboard that is most overlapped with the bounds of the target piece of art (if
								an artboard could be inferred and acquired)
					
					\note		Since Illustrator artwork is not hierarchically contained within an artboard (such
								as with layers), this method merely infers the placement of a piece of art on a
								given artboard by searching for whichever artboard whose bounds most overlaps the 
								bounds of the target piece of art. If you are interested in all the artboards that
								touch the target piece of art, call allArtboards() instead.
				*/
				virtual std::unique_ptr<Artboard> artboard() const;
				
				/**
					\brief		Gets all the artboards that "touch" the target art
					\author		GW
					\date		08/2014
					
					\returns	A vector of artboards whose bounds intersect the bounds of the target piece of art
								in any way
				*/
				virtual ArtboardVector allArtboards() const;

				/**
					\brief		Gets the parent layer for the target art
					\author		GW
					\date		08/2013
					
					\returns	The layer that contains the target piece of art
				*/
				virtual std::unique_ptr<Layer> layer() const;

				/**
					\brief		Gets the count of the children for the target art
					\author		GW
					\date		08/2013
					
					\returns	The count of children that the art contains, or 0 if it is not a container type
					
					\note		Illustrator provides no way to directly access the child count, so they must be manually
								iterated through. As such, the complexity of this method is O(n).
				*/
				virtual uint32_t childCount() const;

				/**
					\brief		Gets the parent art object for the target art
					\author		GW
					\date		08/2013
					
					\returns	The target's parent Art object
				*/
				virtual std::unique_ptr<Art> parent() const;

				/**
					\brief		Gets the first child art object for the target art
					\author		GW
					\date		08/2013
					
					\returns	The container's first child Art object
				*/
				virtual std::unique_ptr<Art> firstChild() const;

				/**
					\brief		Gets the last child art object for the target art
					\author		GW
					\date		08/2013
					
					\returns	The container's last child Art object
				*/
				virtual std::unique_ptr<Art> lastChild() const;

				/**
					\brief		Gets the next sibling art object for the target art
					\author		GW
					\date		08/2013
					
					\returns	The target's sibling Art object
				*/
				virtual std::unique_ptr<Art> sibling() const;

				/**
					\brief		Gets the previous sibling art object for the target art
					\author		GW
					\date		08/2013
					
					\returns	The container's prior sibling Art object
				*/
				virtual std::unique_ptr<Art> priorSibling() const;

				/**
					\brief		Gets the original symbol definition for an instance of a symbol
					\author		GW
					\date		08/2013
					
					\returns	The symbol instance's source group Art object
				*/
				virtual std::unique_ptr<Symbol> symbol() const;
				
				/**
					\brief		Sets the original symbol definition object for an instance of a symbol
					\author		GW
					\date		11/2013
					
					\param		sym_	The symbol art instance's definition
					\returns	true if the symbol instance's defintion was set, false otherwise
				*/
				virtual bool setSymbol(const Symbol& sym_);
				
				/**
					\brief		Gets a vector of art in the art tree, rooted at the target, whose specs match those given
					\author		GW
					\date		11/2014

					\param		specs_	Vector of match specifications
					\returns	An artwork vector whose entries match the provided specifications

					\note		This method takes a vector of specs so the caller can specify a multitude of artwork
								properties, thereby acquiring a vector of artwork that (currently) exhibit all of the
								provided properties.
				*/
				virtual ArtVector matchingChildren(const MatchArtSpecVector& specs_) const;

				/**
					\brief		Gets whether the target art object has a fill
					\author		GW
					\date		08/2013
					
					\returns	true for filled, false otherwise
				*/
				virtual bool hasFill() const;

				/**
					\brief		Gets whether the target art object has a stroke
					\author		GW
					\date		08/2013
					
					\returns	true for stroked, false otherwise
				*/
				virtual bool hasStroke() const;

				/**
					\brief		Gets whether the target art object is a group that corresponds to a layer or sublayer
					\author		GW
					\date		08/2013
					
					\returns	true if the target is a layer group, false otherwise
				*/
				virtual bool isLayerGroup() const;
				
				/**
					\brief		Gets whether the target art object is a group that has clip mask children
					\author		GW
					\date		11/2014
					
					\returns	true if the art object is a clip group, false otherwise
				*/
				virtual bool isClipGroup() const;
			
				/**
					\brief	Sets whether the target art object is a group that has clip mask children
					\author	GW
					\date	07/2017
					
					\param	clip_	true to set the art object to be a clip group, false otherwise
					
					\note	This method will fail if the art is not of type ArtTypeGroup.
				*/
				virtual void setIsClipGroup(const bool clip_);

				/**
					\brief		Gets whether the target art object is a piece of dimension art
					\author		GW
					\date		10/2023
					
					\returns	true if a dimension, false otherwise
					
					\note		Always returns false for Illustrator versions below 28.1
				*/
				virtual bool isDimension() const;

				/**
					\brief		Gets the art's style
					\author		GW
					\date		08/2013
					
					\returns	The target art's styling
				*/
				virtual std::unique_ptr<ArtStyle> style() const;

				/**
					\brief	Sets the art's style
					\author	GW
					\date	08/2013

					\param	style_	New style for the art
				*/
				virtual void setStyle(const ArtStyle& style_);

				/**
					\brief		Gets whether the target art is part of the styled art of another art object
					\author		GW
					\date		08/2013
					
					\returns	true if the art is styled, false otherwise
				*/
				virtual bool isStyled() const;

				/**
					\brief		If the art has a style, this acquires its styled representation
					\author		GW
					\date		09/2021
					
					\returns	The target's styled Art object, if any
				*/
				virtual std::unique_ptr<Art> styledArt() const;

				/**
					\brief	If the art has a style, this flattens the styled representation and replaces the source
							art with said representation
					\author	GW
					\date	09/2021
				*/
				virtual void flattenStyledArt();

				/**
					\brief		Retrieves the current blending mode of an art object, which applies to the object as a
								whole
					\author		GW
					\date		08/2014
					
					\returns	The blending mode enum value
				*/
				virtual BlendMode blendMode() const;

				/**
					\brief	Sets the current blending mode of an art object, which applies to the object as a whole
					\author	GW
					\date	08/2014
					
					\param	mode_	New blending mode enum value
				*/
				virtual void setBlendMode(const BlendMode mode_);

				/**
					\brief		Retrieves the current opacity value applied to an art object
					\author		GW
					\date		08/2014
					
					\returns	The opacity value, in the range [0,1] where 0 is completely transparent and 1 is
								completely opaque
				*/
				virtual double opacity() const;
				
				/**
					\brief	Sets the current opacity value applied to an art object
					\author	GW
					\date	08/2014
					
					\param	o_	New opacity value, in the range [0,1]
				*/
				virtual void setOpacity(const double o_);

				/**
					\brief		Retrieves the current knockout state of an art object
					\author		GW
					\date		08/2014
					
					\returns	The knockout enum value
					
					\note		If the state is KnockoutInherited, the on/off state is inherited from the object's
								parent, which can be acquired with inheritedKnockout()
				*/
				virtual BlendKnockout knockout() const;

				/**
					\brief		Retrieves the inherited knockout state (on or off) of an art object
					\author		GW
					\date		08/2014
					
					\returns	The inherited knockout enum value
				*/
				virtual BlendKnockout inheritedKnockout() const;

				/**
					\brief	Sets the current knockout state of an art object
					\author	GW
					\date	08/2014
					
					\param	ko_		New knockout enum value
				*/
				virtual void setKnockout(const BlendKnockout ko_);

				/**
					\brief		Reorders the target art in Illustrator's drawing order
					\author		GW
					\date		08/2013

					\param		order_	Order in which to place the art
					\param		prep_	Prepositional piece of art to which order_ will apply (or NULL if not needed)

					\warning	This method does not assume responsibility of the memory for the prep_ argument.
				*/
				virtual void reorder(const PaintOrder order_, const Art* const prep_ = NULL);

				/**
					\brief	Expands the target art object
					\author	GW
					\date	08/2013

					\param	opts_	Controls how the expansion is done
					\param	steps_	For artwork with gradient fills, this is the number of art pieces to create to
									approximate the gradient

					\note	Expansion will usually cause the art to change types (into a group), because it's possible
							for expansion to result in multiple art objects. For example, if you have some path art with
							a wide stroke and expand it, you may have to get the first child of the resulting group to
							get at the new path art itself.
				*/
				virtual void expand(const ExpandOptions opts_, const uint32_t steps_);

				/**
					\brief	Rotates the target art by a given angle around its center point
					\author	GW
					\date	08/2013

					\param	angle_	Angle to rotate the art around its center point
					\param	deep_	Whether the rotation transform is recursively applied to child objects
				*/
				virtual void rotate(const Angle& angle_, const bool deep_ = false);

				/**
					\brief	Rotates the target art by a given angle around a given pre-defined pin point
					\author	GW
					\date	08/2013

					\param	angle_	Angle to rotate the art
					\param	pinPt_	"Pin point" around which the art will be rotated
					\param	deep_	Whether the rotation transform is recursively applied to child objects
				*/
				virtual void rotate(const Angle& angle_, const BoundsPinPointType pinPt_, const bool deep_ = false);

				/**
					\brief	Rotates the target art by a given angle around a given pin point
					\author	GW
					\date	08/2013

					\param	angle_		Angle to rotate the art
					\param	origin_		New "origin" around which the art will be rotated
					\param	deep_		Whether the rotation transform is recursively applied to child objects
				*/
				virtual void rotate(const Angle& angle_, const ArtboardPoint& origin_, const bool deep_ = false);

				/**
					\brief	Scales the art by a given scaler from its center point
					\author	GW
					\date	08/2013

					\param	scaler_		Multiplier for scaling (e.g. 0.75 is 75% of the current size)
					\param	deep_		Whether the scale transform is recursively applied to child objects
				*/
				virtual void scale(const double scaler_, const bool deep_ = false);

				/**
					\brief	Scales the art by a given scaler from a preset pin point
					\author	GW
					\date	08/2013

					\param	scaler_		Multiplier for scaling (e.g. 0.75 is 75% of the current size)
					\param	pinPt_		"Pin point" from which the art will be scaled
					\param	deep_		Whether the scale transform is recursively applied to child objects
				*/
				virtual void scale(const double scaler_, const BoundsPinPointType pinPt_, const bool deep_ = false);

				/**
					\brief	Scales the art by a given scaler from a given pin point
					\author	GW
					\date	08/2013

					\param	scaler_		Multiplier for scaling (e.g. 0.75 is 75% of the current size)
					\param	origin_		"Origin" from which the art will be scaled
					\param	deep_		Whether the scale transform is recursively applied to child objects
				*/
				virtual void scale(const double scaler_, const ArtboardPoint& origin_, const bool deep_ = false);

				/**
					\brief	Scales the art by givens scaler from its center point
					\author	GW
					\date	08/2013

					\param	xScaler_	Multiplier for scaling horizontally (e.g. 0.75 is 75% of the current size)
					\param	yScaler_	Multiplier for scaling vertically (e.g. 0.75 is 75% of the current size)
					\param	deep_		Whether the scale transform is recursively applied to child objects
				*/
				virtual void scale(const double xScaler_, const double yScaler_, const bool deep_ = false);

				/**
					\brief	Scales the art by given scalers from a preset pin point
					\author	GW
					\date	08/2013

					\param	xScaler_	Multiplier for scaling horizontally (e.g. 0.75 is 75% of the current size)
					\param	yScaler_	Multiplier for scaling vertically (e.g. 0.75 is 75% of the current size)
					\param	pinPt_		"Pin point" from which the art will be scaled
					\param	deep_		Whether the scale transform is recursively applied to child objects
				*/
				virtual void scale(
					const double xScaler_,
					const double yScaler_,
					const BoundsPinPointType pinPt_,
					const bool deep_ = false
				);

				/**
					\brief	Scales the art by given scalers from a given pin point
					\author	GW
					\date	08/2013

					\param	xScaler_	Multiplier for scaling horizontally (e.g. 0.75 is 75% of the current size)
					\param	yScaler_	Multiplier for scaling vertically (e.g. 0.75 is 75% of the current size)
					\param	origin_		New "origin" from which the art will be scaled
					\param	deep_		Whether the scale transform is recursively applied to child objects
				*/
				virtual void scale(
					const double xScaler_,
					const double yScaler_,
					const ArtboardPoint& origin_,
					const bool deep_ = false
				);

				/**
					\brief	Offsets (moves) the art by given x-axis and y-axis values
					\author	GW
					\date	08/2013

					\param	x_		Amount to offset along the x-axis
					\param	y_		Amount to offset along the y-axis
					\param	deep_	Whether the offset transform is recursively applied to child objects
				*/
				virtual void offset(const double x_, const double y_, const bool deep_ = false);

				/**
					\brief	Moves the art to a given point, placing its pin point (pinPt_ arg) at a particular pre-
							defined point
					\author	GW
					\date	08/2013

					\param	newOrigin_	New "origin" for the art
					\param	pinPt_		"Pin point" which will be moved to the new "origin"
					\param	deep_		Whether the move transform is recursively applied to child objects
				*/
				virtual void move(
					const ArtboardPoint& newOrigin_,
					const BoundsPinPointType pinPt_,
					const bool deep_ = false
				);

				/**
					\brief	Moves the art to a given point, placing its pin point at another arbitrary point (pinPt_ arg)
					\author	GW
					\date	08/2013

					\param	newOrigin_	New "origin" for the art
					\param	origin_		"Pin point", at an arbitrary position, which will be moved to the new "origin"
					\param	deep_		Whether the move transform is recursively applied to child objects
				*/
				virtual void move(
					const ArtboardPoint& newOrigin_,
					const ArtboardPoint& origin_,
					const bool deep_ = false
				);

				/**
					\brief	Performs an arbitrary transformation on the target art
					\author	GW
					\date	08/2013

					\param	matrix_		Transformation matrix to apply for the transform
					\param	deep_		Whether the transform is recursively applied to child objects
				*/
				virtual void transform(const TransformMatrix& matrix_, const bool deep_ = false);

				/**
					\brief	Performs an arbitrary transformation on the target art with the given options
					\author	GW
					\date	03/2016

					\param	matrix_		Transformation matrix to apply for the transform
					\param	options_	TransformOptions enum values bitwise OR'd together
				*/
				virtual void transformWithOptions(const TransformMatrix& matrix_, const TransformOptions options_);

				/**
					\brief	Performs a transformation of the size of strokes on the target art
					\author	GW
					\date	07/2017

					\param	scaler_		Amount to scale the strokes, as a multiplier (e.g. 2.0 means 200%)
					\param	children_	Whether to transform strokes of all children too
				*/
				virtual void transformStrokes(const double scaler_, const bool children_);

				/**
					\brief		Gets whether the art, or one of its parents, can be isolated
					\author		GW
					\date		08/2013
					
					\returns	true if the art can be isolated, false otherwise
					
					\note		Typically, Illustrator does not allow "leaf art" (e.g. paths) to be isolated; however,
								this method looks up the parent lineage to see if a parent can be isolated.
					\note		When isolating a piece of art, the isolate() method will look up the parent lineage and
								automatically isolate the first parent that can be isolated.
				*/
				virtual bool canIsolate() const;

				/**
					\brief		Gets whether the art, or one of its parents, is currently isolated
					\author		GW
					\date		08/2013
					
					\returns	true if the art is currently isolated, false otherwise
					
					\note		Typically, Illustrator does not allow "leaf art" (e.g. paths) to be isolated; however,
								this method looks up the parent lineage to see if a parent is currently isolated.
				*/
				virtual bool isIsolated() const;

				/**
					\brief		Isolates the art or one of its parents
					\author		GW
					\date		08/2013
					
					\param		hideOtherArt_	true to make non-isolated invisible, false to dim it instead
					\returns	true for successful isolation, false otherwise
					
					\note		When isolating a piece of art, this will look up the parent lineage and automatically
								isolate the first parent that can be isolated (as Illustrator does not allow "leaf art",
								such as paths, to be isolated).
				*/
				virtual bool isolate(const bool hideOtherArt_);

				/**
					\brief		Gets an iterator for the art in the target's parent group, starting with the target
					\author		GW
					\date		08/2013
					
					\returns	An ArtIterator object, for iterating through the target's siblings
					
					\code
						// Here's an example, iterating through all children of someArt:
						hdi::core::Art child0;
						someArt.firstChild(child0);
						for(hdi::core::Art::Iterator iter = child0.begin(); !iter.atEnd(); ++iter)
						{
							// Do stuff with iter.art() ...
						}
						
						// Here's another example, reverse iterating through all children of someArt:
						hdi::core::Art childN;
						someArt.lastChild(childN);
						for(hdi::core::Art::Iterator iter = childN.begin(); !iter.atEnd(); --iter)
						{
							// Do more stuff with iter.art() ...
						}
					\endcode
				*/
				Iterator begin() const;


			private:
				friend aip::Art* __accessImpl(const Art&);
				friend Art __accessCtor(aip::Art*&);

				/**
					\brief	Private implementation data
				*/
				void* __data;
				
				/**
					\brief	Internal use only
					\author	GW
					\date	08/2013
				*/
				void* __impl() const;

				/**
					\brief	Internal use only
					\author	GW
					\date	08/2013
				*/
				Art(aip::Art*&);
		};
		
		typedef std::unique_ptr<Art> ArtUP;
		typedef std::shared_ptr<Art> ArtSP;
		typedef std::weak_ptr<Art> ArtWP;
		
		extern aip::Art* __accessImpl(const Art&);
		extern Art __accessCtor(aip::Art*&);
	}
}

hdi::core::Art::BoundsOptions operator|(
	const hdi::core::Art::BoundsOptions lhs_,
	const hdi::core::Art::BoundsOptions rhs_
);

hdi::core::Art::BoundsOptions& operator|=(
	hdi::core::Art::BoundsOptions& lhs__,
	const hdi::core::Art::BoundsOptions rhs_
);

hdi::core::Art::ExpandOptions operator|(
	const hdi::core::Art::ExpandOptions lhs_,
	const hdi::core::Art::ExpandOptions rhs_
);

hdi::core::Art::ExpandOptions& operator|=(
	hdi::core::Art::ExpandOptions& lhs__,
	const hdi::core::Art::ExpandOptions rhs_
);

hdi::core::Art::TransformOptions operator|(
	const hdi::core::Art::TransformOptions lhs_,
	const hdi::core::Art::TransformOptions rhs_
);

hdi::core::Art::TransformOptions& operator|=(
	hdi::core::Art::TransformOptions& lhs__,
	const hdi::core::Art::TransformOptions rhs_
);

#endif
// HDI_CORE_AIP_MODE

#endif
// __HDI_CORE_ART__
