/**
	\file
	\brief		Header file for art color manipulation
	\copyright	Hot Door, Inc. 2010-2025
*/

#ifndef __HDI_CORE_ART_COLOR__
#define __HDI_CORE_ART_COLOR__

#include "hdicoreAngle.h"
#include "hdicoreSize.h"

#if defined(HDI_CORE_AIP_MODE)
	#include "hdicoreArtboardPoint.h"
	#include "hdicoreTransformMatrix.h"
#endif
	
namespace hdi
{
	#if defined(HDI_CORE_AIP_MODE)
		namespace aip
		{
			class ArtColor;
		}
	#endif

	namespace core
	{
		class Color;
		
		#if defined(HDI_CORE_AIP_MODE)
			class Gradient;
			class Pattern;
		#endif

		/**
			\brief		Describes the color of art on the artboard
			\details	Contains color data, in a variety of formats, and color metadata (e.g. names). Can be grayscale,
						RGB, Lab, HSB, or CMYK.
		*/
		class ArtColor
		{
			public:
				/**
					\brief	Describes the format/type of an ArtColor object
				*/
				enum Type
				{
					TypeUnknown			= 0,
					
					// Basic colors
					TypeNone			= 1 << 0,
					TypeGray			= 1 << 1,
					TypeRGB				= 1 << 2,
					TypeCMYK			= 1 << 3,
					TypeLab				= 1 << 4,
					TypeHSB				= 1 << 5,	// Same as HSV
					
					#if defined(HDI_CORE_AIP_MODE)
						// "Named" colors
						TypeNamedRGB		= 1 << 10,
						TypeNamedCMYK		= 1 << 11,
						TypeNamedLab		= 1 << 12,
					
						// Other
						TypePattern			= 1 << 20,
						TypeGradient		= 1 << 21,
					#endif
					// HDI_CORE_AIP_MODE
					
					// Helpers
					AnonymousTypes	= (TypeGray | TypeRGB | TypeCMYK | TypeLab | TypeHSB),
					
					#if defined(HDI_CORE_AIP_MODE)
						NamedTypes		= (TypeNamedRGB | TypeNamedCMYK | TypeNamedLab)
					#endif
				};

				#if defined(HDI_CORE_AIP_MODE)
					/**
						\brief	Option flags for use when creating a named color
					*/
					enum NamedFlags
					{
						NoFlag					= 0,
						SpotColorFlag			= 1 << 0,
						RegistrationColorFlag	= 1 << 1
					};
				#endif
				// HDI_CORE_AIP_MODE
				
				/**
					\brief	Stores all the possible representations of a color-based ArtColor in all supported color
							models; the original model is indicated by the master member
				*/
				class Channels
				{
					public:
						/**
							\brief	Describes a set of RGB channels
						*/
						struct RGB
						{
							/**
								\brief	Stores the RGB color red channel, in the range [0.0,1.0]
							*/
							double red;
						
							/**
								\brief	Stores the RGB color green channel, in the range [0.0,1.0]
							*/
							double green;
						
							/**
								\brief	Stores the RGB color blue channel, in the range [0.0,1.0]
							*/
							double blue;
							
							/**
								\brief	Constructs an RGB object with default values
								\author	GW
								\date	07/2014
							*/
							(RGB)();

							/**
								\brief	Constructs a new RGB object from an existing RGB
								\author	GW
								\date	07/2014
								
								\param	obj_	Existing RGB object to copy values from
							*/
							(RGB)(const RGB& obj_);

							/**
								\brief	Constructs a new RGB object with the given values
								\author	GW
								\date	07/2014
								
								\param	r_	(See red member description)
								\param	g_	(See green member description)
								\param	b_	(See blue member description)
							*/
							(RGB)(const double r_, const double g_, const double b_);

							/**
								\brief	Destructs an RGB object
								\author	GW
								\date	07/2014
							*/
							virtual (~RGB)();

							/**
								\brief		RGB assignment operator
								\author		GW
								\date		07/2014
								
								\param		rhs_	Existing RGB object to copy values from (righthand side of the
													assignment operator)
								\returns	The target RGB object, but with its values updated
							*/
							RGB& operator=(const RGB& rhs_);
							
							/**
								\brief		Tests whether a given RGB object is the same as another
								\author		GW
								\date		03/2015

								\param		rhs_	RGB to compare against (righthand side of equality operator)
								\returns	true for the target and rhs_ being the same colors, false otherwise
							*/
							bool operator==(const RGB& rhs_) const;
							
							/**
								\brief		Tests whether a given RGB object is not the same as another
								\author		GW
								\date		03/2015

								\param		rhs_	RGB to compare against (righthand side of inequality operator)
								\returns	true for the target and rhs_ being different colors, false otherwise
							*/
							bool operator!=(const RGB& rhs_) const;
						};
		
						typedef std::unique_ptr<RGB> RGBUP;
						typedef std::shared_ptr<RGB> RGBSP;
						typedef std::weak_ptr<RGB> RGBWP;
						
						/**
							\brief	Describes a set of CMYK channels
						*/
						struct CMYK
						{
							/**
								\brief	Stores the CMYK color cyan channel, in the range [0.0,1.0]
							*/
							double cyan;
						
							/**
								\brief	Stores the CMYK color magenta channel, in the range [0.0,1.0]
							*/
							double magenta;
						
							/**
								\brief	Stores the CMYK color yellow channel, in the range [0.0,1.0]
							*/
							double yellow;
						
							/**
								\brief	Stores the CMYK color black channel, in the range [0.0,1.0]
							*/
							double black;
							
							/**
								\brief	Constructs a CMYK object with default values
								\author	GW
								\date	07/2014
							*/
							(CMYK)();

							/**
								\brief	Constructs a new CMYK object from an existing CMYK
								\author	GW
								\date	07/2014
								
								\param	obj_	Existing CMYK object to copy values from
							*/
							(CMYK)(const CMYK& obj_);

							/**
								\brief	Constructs a new CMYK object with the given values
								\author	GW
								\date	07/2014
								
								\param	c_	(See cyan member description)
								\param	m_	(See magenta member description)
								\param	y_	(See yellow member description)
								\param	k_	(See black member description)
							*/
							(CMYK)(const double c_, const double m_, const double y_, const double k_);

							/**
								\brief	Destructs a CMYK object
								\author	GW
								\date	07/2014
							*/
							virtual (~CMYK)();

							/**
								\brief		CMYK assignment operator
								\author		GW
								\date		07/2014
								
								\param		rhs_	Existing CMYK object to copy values from (righthand side of the
													assignment operator)
								\returns	The target CMYK object, but with its values updated
							*/
							CMYK& operator=(const CMYK& rhs_);
							
							/**
								\brief		Tests whether a given CMYK object is the same as another
								\author		GW
								\date		03/2015

								\param		rhs_	CMYK to compare against (righthand side of equality operator)
								\returns	true for the target and rhs_ being the same colors, false otherwise
							*/
							bool operator==(const CMYK& rhs_) const;
							
							/**
								\brief		Tests whether a given CMYK object is not the same as another
								\author		GW
								\date		03/2015

								\param		rhs_	CMYK to compare against (righthand side of inequality operator)
								\returns	true for the target and rhs_ being different colors, false otherwise
							*/
							bool operator!=(const CMYK& rhs_) const;
						};
		
						typedef std::unique_ptr<CMYK> CMYKUP;
						typedef std::shared_ptr<CMYK> CMYKSP;
						typedef std::weak_ptr<CMYK> CMYKWP;
					
						/**
							\brief	Describes a set of Lab channels
						*/
						struct Lab
						{
							/**
								\brief	Stores the Lab color lightness channel, in the range [0.0,1.0]
							*/
							double lightness;
						
							/**
								\brief	Stores the Lab color a-dimension channel, in the range [-1.0,1.0]
							*/
							double a;
						
							/**
								\brief	Stores the Lab color b-dimension channel, in the range [-1.0,1.0]
							*/
							double b;
							
							/**
								\brief	Constructs a Lab object with default values
								\author	GW
								\date	07/2014
							*/
							Lab();

							/**
								\brief	Constructs a new Lab object from an existing Lab
								\author	GW
								\date	07/2014
								
								\param	obj_	Existing Lab object to copy values from
							*/
							Lab(const Lab& obj_);

							/**
								\brief	Constructs a new Lab object with the given values
								\author	GW
								\date	07/2014
								
								\param	l_	(See lightness member description)
								\param	a_	(See a member description)
								\param	b_	(See b member description)
							*/
							Lab(const double l_, const double a_, const double b_);

							/**
								\brief	Destructs a Lab object
								\author	GW
								\date	07/2014
							*/
							virtual ~Lab();

							/**
								\brief		Lab assignment operator
								\author		GW
								\date		07/2014
								
								\param		rhs_	Existing Lab object to copy values from (righthand side of the
													assignment operator)
								\returns	The target Lab object, but with its values updated
							*/
							Lab& operator=(const Lab& rhs_);
							
							/**
								\brief		Tests whether a given Lab object is the same as another
								\author		GW
								\date		03/2015

								\param		rhs_	Lab to compare against (righthand side of equality operator)
								\returns	true for the target and rhs_ being the same colors, false otherwise
							*/
							bool operator==(const Lab& rhs_) const;
							
							/**
								\brief		Tests whether a given Lab object is not the same as another
								\author		GW
								\date		03/2015

								\param		rhs_	Lab to compare against (righthand side of inequality operator)
								\returns	true for the target and rhs_ being different colors, false otherwise
							*/
							bool operator!=(const Lab& rhs_) const;
						};
		
						typedef std::unique_ptr<Lab> LabUP;
						typedef std::shared_ptr<Lab> LabSP;
						typedef std::weak_ptr<Lab> LabWP;
					
						/**
							\brief	Describes a set of HSB channels
						*/
						struct HSB
						{
							/**
								\brief	Stores the HSB color hue channel
							*/
							Angle hue;
						
							/**
								\brief	Stores the HSB color saturation channel, in the range [0.0,1.0]
							*/
							double saturation;
						
							/**
								\brief	Stores the HSB color brightness channel, in the range [0.0,1.0]
							*/
							double brightness;
							
							/**
								\brief	Constructs a HSB object with default values
								\author	GW
								\date	03/2015
							*/
							HSB();

							/**
								\brief	Constructs a new HSB object from an existing HSB
								\author	GW
								\date	03/2015
								
								\param	obj_	Existing HSB object to copy values from
							*/
							HSB(const HSB& obj_);

							/**
								\brief	Constructs a new HSB object with the given values
								\author	GW
								\date	03/2015
								
								\param	h_	(See hue member description)
								\param	s_	(See saturation member description)
								\param	b_	(See brightness member description)
							*/
							HSB(const Angle& h_, const double s_, const double b_);

							/**
								\brief	Destructs a HSB object
								\author	GW
								\date	03/2015
							*/
							virtual ~HSB();

							/**
								\brief		HSB assignment operator
								\author		GW
								\date		03/2015
								
								\param		rhs_	Existing Lab object to copy values from (righthand side of the
													assignment operator)
								\returns	The target HSB object, but with its values updated
							*/
							HSB& operator=(const HSB& rhs_);
							
							/**
								\brief		Tests whether a given HSB object is the same as another
								\author		GW
								\date		03/2015

								\param		rhs_	HSB to compare against (righthand side of equality operator)
								\returns	true for the target and rhs_ being the same colors, false otherwise
							*/
							bool operator==(const HSB& rhs_) const;
							
							/**
								\brief		Tests whether a given HSB object is not the same as another
								\author		GW
								\date		03/2015

								\param		rhs_	HSB to compare against (righthand side of inequality operator)
								\returns	true for the target and rhs_ being different colors, false otherwise
							*/
							bool operator!=(const HSB& rhs_) const;
						};
		
						typedef std::unique_ptr<HSB> HSBUP;
						typedef std::shared_ptr<HSB> HSBSP;
						typedef std::weak_ptr<HSB> HSBWP;
						
						/**
							\brief	Stores the "master" type of channel data stored in the target object. The gray,
									rgb, cmyk, or lab member that matches this member will have the most accurate values
									as all other members were derived via color conversion.
						*/
						Type master;
					
						/**
							\brief	Stores the grayscale color gray channel (0.0 is white, 1.0 is black)
						*/
						double gray;
						
						/**
							\brief	Stores the RGB channels
						*/
						RGB rgb;
						
						/**
							\brief	Stores the CMYK channels
						*/
						CMYK cmyk;
						
						/**
							\brief	Stores the Lab channels
						*/
						Lab lab;
						
						/**
							\brief	Stores the HSB channels
						*/
						HSB hsb;

						/**
							\brief	Constructs a new Channels object with default values
							\author	GW
							\date	07/2014
						*/
						Channels();

						/**
							\brief	Constructs a new Channels object from an existing Channels
							\author	GW
							\date	07/2014
							
							\param	obj_	Existing Channels object to copy values from
						*/
						Channels(const Channels& obj_);

						/**
							\brief	Constructs a new Channels object with the given values
							\author	GW
							\date	07/2014
							
							\param	master_		(See master member description)
							\param	gray_		(See gray member description)
							\param	rgb_		(See rgb member description)
							\param	cmyk_		(See cmyk member description)
							\param	lab_		(See lab member description)
							\param	hsb_		(See hsb member description)
						*/
						Channels(
							const Type master_,
							const double gray_,
							const RGB& rgb_,
							const CMYK& cmyk_,
							const Lab& lab_,
							const HSB& hsb_
						);

						/**
							\brief	Destructs a Channels object
							\author	GW
							\date	07/2014
						*/
						virtual ~Channels();

						/**
							\brief		Channels assignment operator
							\author		GW
							\date		12/2013
							
							\param		rhs_	Existing Channels object to copy values from (righthand side of the
												assignment operator)
							\returns	The target Channels object, but with its values updated
						*/
						Channels& operator=(const Channels& rhs_);
				};
		
				typedef std::unique_ptr<Channels> ChannelsUP;
				typedef std::shared_ptr<Channels> ChannelsSP;
				typedef std::weak_ptr<Channels> ChannelsWP;

				#if defined(HDI_CORE_AIP_MODE)
					/**
						\brief	Describes a gradient that has been applied to some art
					*/
					class GradientStyle
					{
						public:
							/**
								\brief	Gradient to which the style applies. The lifetime of the object is only as long as
										the GradientStyle which contains it (so make sure to use the copy constructor if you
										need to store the Gradient any longer).
							*/
							core::Gradient* gradient;

							/**
								\brief	For a linear gradient, the origin of the gradient in page coordinates, which is 0 on
										the gradient ramp. For a radial gradient, the center of the containing circle.
							*/
							core::ArtboardPoint origin;

							/**
								\brief	For a linear gradient, the angle of the gradient in degrees from the X axis. The
										ramp extends from the origin at this angle. Positive values are counterclockwise.
										Not used for radial gradients, but the value is preserved in case the gradient is
										changed back to linear.
							*/
							core::Angle angle;

							/**
								\brief	For a linear gradient, the distance in points over which the gradient ramp is
										applied. The ramp is scaled so that 100% of its value is this length. For a radial
										gradient, the radius of the containing circle.
							*/
							double length;

							/**
								\brief	The accumulated transformations of the gradient. It is not necessarily the same as
										the transformation matrix of the object containing the gradient. When a gradient is
										first applied to an object, the value is set to the identity matrix. When the user
										transforms the object, the user transformation matrix is concatenated to the
										gradient instance's matrix.
							*/
							core::TransformMatrix matrix;

							/**
								\brief	For a radial gradient, the angle to the blend highlight point. Not used for linear
										gradients.
							*/
							core::Angle highlightAngle;

							/**
								\brief	For a radial gradient, the distance of the hilight from the origin expressed as a
										fraction of the radius (a value between 0 and 1). Not used for linear gradients.
							*/
							double highlightLength;

							/**
								\brief	Constructs a new GradientStyle object with default values
								\author	GW
								\date	12/2013
							*/
							GradientStyle();

							/**
								\brief	Constructs a new GradientStyle object from an existing GradientStyle
								\author	GW
								\date	12/2013
								
								\param	grad_	Existing GradientStyle object to copy values from
							*/
							GradientStyle(const GradientStyle& grad_);

							/**
								\brief	Constructs a new GradientStyle object with the given values
								\author	GW
								\date	12/2013
								
								\param	gradient_		(See gradient member description)
								\param	origin_			(See origin member description)
								\param	angle_			(See angle member description)
								\param	length_			(See length member description)
								\param	matrix_			(See matrix member description)
								\param	hiliteAngle_	(See highlightAngle member description)
								\param	hiliteLength_	(See highlightLength member description)
							*/
							GradientStyle(
								const core::Gradient& gradient_,
								const core::ArtboardPoint& origin_,
								const core::Angle& angle_,
								const double length_,
								const core::TransformMatrix& matrix_,
								const core::Angle& hiliteAngle_,
								const double hiliteLength_
							);

							/**
								\brief	Destructs a GradientStyle object
								\author	GW
								\date	12/2013
							*/
							virtual ~GradientStyle();

							/**
								\brief		GradientStyle assignment operator
								\author		GW
								\date		12/2013
								
								\param		rhs_	Existing GradientStyle object to copy values from (righthand side of the
													assignment operator)
								\returns	The target GradientStyle object, but with its values updated
							*/
							GradientStyle& operator=(const GradientStyle& rhs_);
					};
			
					typedef std::unique_ptr<GradientStyle> GradientStyleUP;
					typedef std::shared_ptr<GradientStyle> GradientStyleSP;
					typedef std::weak_ptr<GradientStyle> GradientStyleWP;

					/**
						\brief	Describes a pattern that has been applied to some art
					*/
					class PatternStyle
					{
						public:
							/**
								\brief	Pattern to which the style applies. The lifetime of the object is only as long as
										the PatternStyle which contains it (so make sure to use the copy constructor if you
										need to store the Pattern any longer).
							*/
							core::Pattern* pattern;

							/**
								\brief	Distance to translate the unscaled prototype before filling, measured from the ruler
										origin
							*/
							double shiftDistance;

							/**
								\brief	Angle to translate the unscaled prototype before filling
							*/
							core::Angle shiftAngle;

							/**
								\brief	Fraction to scale the prototype along the x-axis before filling
							*/
							double scaleX;

							/**
								\brief	Fraction to scale the prototype along the y-axis before filling
							*/
							double scaleY;

							/**
								\brief	Angle to rotate the prototype before filling
							*/
							core::Angle rotation;

							/**
								\brief	When true, the prototype is reflected before filling
							*/
							bool reflect;

							/**
								\brief	Axis around which to reflect
							*/
							core::Angle reflectAngle;

							/**
								\brief	Angle to slant the shear by
							*/
							core::Angle shearAngle;

							/**
								\brief	Axis to shear with respect to
							*/
							core::Angle shearAxis;

							/**
								\brief	Additional transformation arising from manipulating the path
							*/
							core::TransformMatrix transform;

							/**
								\brief	Constructs a new PatternStyle object with default values
								\author	GW
								\date	12/2013
							*/
							PatternStyle();

							/**
								\brief	Constructs a new PatternStyle object from an existing PatternStyle
								\author	GW
								\date	12/2013
								
								\param	patt_	Existing PatternStyle object to copy values from
							*/
							PatternStyle(const PatternStyle& patt_);

							/**
								\brief	Constructs a new PatternStyle object with the given values
								\author	GW
								\date	12/2013
								
								\param	pattern_		(See pattern member description)
								\param	shiftDist_		(See shiftDistance member description)
								\param	shiftAngle_		(See shiftAngle member description)
								\param	scaleX_			(See scaleX member description)
								\param	scaleY_			(See scaleY member description)
								\param	rotation_		(See rotation member description)
								\param	reflect_		(See reflect member description)
								\param	reflectAngle_	(See reflectAngle member description)
								\param	shearAngle_		(See shearAngle member description)
								\param	shearAxis_		(See shearAxis member description)
								\param	transform_		(See transform member description)
							*/
							PatternStyle(
								const core::Pattern& pattern_,
								const double shiftDist_,
								const core::Angle& shiftAngle_,
								const double scaleX_,
								const double scaleY_,
								const core::Angle& rotation_,
								const bool reflect_,
								const core::Angle& reflectAngle_,
								const core::Angle& shearAngle_,
								const core::Angle& shearAxis_,
								const core::TransformMatrix& transform_
							);

							/**
								\brief	Destructs a PatternStyle object
								\author	GW
								\date	12/2013
							*/
							virtual ~PatternStyle();

							/**
								\brief		PatternStyle assignment operator
								\author		GW
								\date		12/2013
								
								\param		rhs_	Existing PatternStyle object to copy values from (righthand side of the
													assignment operator)
								\returns	The target PatternStyle object, but with its values updated
							*/
							PatternStyle& operator=(const PatternStyle& rhs_);
					};
			
					typedef std::unique_ptr<PatternStyle> PatternStyleUP;
					typedef std::shared_ptr<PatternStyle> PatternStyleSP;
					typedef std::weak_ptr<PatternStyle> PatternStyleWP;
				#endif
				// HDI_CORE_AIP_MODE

				/**
					\brief	Constructs an ArtColor object from an existing ArtColor object (copy constructor)
					\author	GW
					\date	08/2013

					\param	c_	Existing ArtColor object
				*/
				ArtColor(const ArtColor& c_);

				/**
					\brief		Constructs an ArtColor object from an existing Color object
					\author		GW
					\date		07/2025

					\param		c_	Existing Color object
					
					\warning	This ignores the opacity field in the Color class.
				*/
				ArtColor(const Color& c_);

				/**
					\brief	Constructs a grayscale ArtColor object from a single color component in the range [0.0,1.0]
					\author	GW
					\date	08/2013

					\param	grayVal_	The grayscale color component, with 0.0 being white and 1.0 being black
				*/
				ArtColor(const double grayVal_);

				/**
					\brief	Constructs an RGB ArtColor object from three color components in the range [0.0,1.0]
					\author	GW
					\date	08/2013

					\param	rVal_	The red color component
					\param	gVal_	The green color component
					\param	bVal_	The blue color component
				*/
				ArtColor(const double rVal_, const double gVal_, const double bVal_);

				/**
					\brief	Constructs a CMYK ArtColor object from four color components in the range [0.0,1.0]
					\author	GW
					\date	08/2013

					\param	cVal_	The cyan color component
					\param	mVal_	The magenta color component
					\param	yVal_	The yellow color component
					\param	kVal_ 	The black color component
				*/
				ArtColor(const double cVal_, const double mVal_, const double yVal_, const double kVal_);
			
				#if defined(HDI_CORE_AIP_MODE)
					/**
						\brief	Constructs an ArtColor object from a GradientStyle
						\author	GW
						\date	11/2016
						
						\param	gs_		The GradientStyle object, from which the ArtColor will be derived
					*/
					ArtColor(const GradientStyle& gs_);
				
					/**
						\brief	Constructs an ArtColor object from a PatternStyle
						\author	GW
						\date	11/2016
						
						\param	ps_		The PatternStyle object, from which the ArtColor will be derived
					*/
					ArtColor(const PatternStyle& ps_);
				#endif
				// HDI_CORE_AIP_MODE

				/**
					\brief	Constructs an ArtColor object that hasn't been assigned
					\author	GW
					\date	08/2013
					
					\note	For Illustrator, this represents a "none" color. For Photoshop, this doesn't represent any
							color at all.
				*/
				ArtColor();

				/**
					\brief	Constructs a Lab ArtColor object from its color components
					\author	GW
					\date	08/2013

					\param	lVal_	The lightness color component, in the range [0.0,1.0]
					\param	aVal_	The a dimension color component, in the range [-1.0,1.0]
					\param	bVal_	The b dimension color component, in the range [-1.0,1.0]
					
					\note	This is a named constructor due to argument type conflicts with that of the RGB constructor
				*/
				static ArtColor FromLab(const double lVal_, const double aVal_, const double bVal_);

				/**
					\brief	Constructs a HSB ArtColor object from its color components
					\author	GW
					\date	03/2015

					\param	hVal_	The hue color component
					\param	sVal_	The saturation color component, in the range [0.0,1.0]
					\param	bVal_	The brightness color component, in the range [0.0,1.0]
				*/
				static ArtColor FromHSB(const Angle& hVal_, const double sVal_, const double bVal_);

				/**
					\brief	Destructs an ArtColor object
					\author	GW
					\date	08/2013
				*/
				virtual ~ArtColor();

				#if defined(HDI_CORE_PSP_MODE)
					/**
						\brief		Gets whether the target ArtColor object is empty (constructed with the default ctor)
						\author		GW
						\date		01/2018
						
						\returns	true if the target ArtColor object is empty, false otherwise
					*/
					virtual bool isEmpty() const;
				#endif
				// HDI_CORE_PSP_MODE

				/**
					\brief		Gets the type/format of the color
					\author		GW
					\date		08/2013
					
					\returns	An enum value to describe the type of color
				*/
				virtual Type colorType() const;

				#if defined(HDI_CORE_AIP_MODE)
					/**
						\brief		Gets the name of a custom color
						\author		GW
						\date		08/2013
						
						\returns	The UTF-8 name of the custom color, otherwise returns "" (empty string)
					*/
					virtual std::string name() const;
				#endif
				// HDI_CORE_AIP_MODE

				/**
					\brief		Gets the target ArtColor object in a grayscale format
					\author		GW
					\date		08/2013
					
					\returns	A copy of the target ArtColor object, but formatted in grayscale
					
					\note		If the type of the target ArtColor is not a basic or "named" color, then black will be
								returned.
				*/
				virtual ArtColor asGray() const;

				/**
					\brief		Gets the target ArtColor object in an RGB format
					\author		GW
					\date		08/2013
					
					\returns	A copy of the target ArtColor object, but formatted in RGB
					
					\note		If the type of the target ArtColor is not a basic or "named" color, then black will be
								returned.
				*/
				virtual ArtColor asRGB() const;

				/**
					\brief		Gets the target ArtColor object in a Lab format
					\author		GW
					\date		08/2013
					
					\returns	A copy of the target ArtColor object, but formatted in Lab
					
					\note		If the type of the target ArtColor is not a basic or "named" color, then black will be
								returned.
				*/
				virtual ArtColor asLab() const;

				/**
					\brief		Gets the target ArtColor object in a HSB format
					\author		GW
					\date		03/2015
					
					\returns	A copy of the target ArtColor object, but formatted in HSB
					
					\note		If the type of the target ArtColor is not a basic or "named" color, then black will be
								returned.
				*/
				virtual ArtColor asHSB() const;

				/**
					\brief		Gets the target ArtColor object in a CMYK format
					\author		GW
					\date		08/2013
					
					\returns	A copy of the target ArtColor object, but formatted in CMYK
					
					\note		If the type of the target ArtColor is not a basic or "named" color, then black will be
								returned.
				*/
				virtual ArtColor asCMYK() const;

				/**
					\brief		Gets the underlying channel data for the target ArtColor
					\author		GW
					\date		07/2014
					
					\returns	A new Channels struct for the target ArtColor; if the type of the art color is Unknown,
								Gradient, or Pattern then the master member will be unknown and all channels will be 0.0
				*/
				virtual Channels channels() const;

				#if defined(HDI_CORE_AIP_MODE)
					/**
						\brief		Gets the underlying gradient style for the target ArtColor
						\author		GW
						\date		12/2013
						
						\returns	A new GradientStyle struct for the target ArtColor; if the type of the art color is not
									"Gradient" then the "gradient" member will be NULL
					*/
					virtual GradientStyle gradient() const;

					/**
						\brief		Gets the underlying pattern for the target ArtColor
						\author		GW
						\date		12/2013
						
						\returns	A new PatternStyle struct for the target ArtColor; if the type of the art color is not
									"Pattern" then the "pattern" member will be NULL
					*/
					virtual PatternStyle pattern() const;

					/**
						\brief		Converts an "anonymous" color to a "named" color
						\author		GW
						\date		08/2013

						\param		name_	New name for the color, as UTF-8
						\param		tint_	How much to tint the color
						\param		flags_	Whether the color is a spot color or registration color
						\returns	true if the conversion occurred, false otherwise
						
						\note		Illustrator only supports named RGB, CMYK, and Lab colors natively. As such, a gray or
									HSB color will be converted to RGB before becoming named. Other types, such as gradients
									and patterns, cannot be named at all.
						\note		Named colors are not automatically added to Illustrator's Swatches panel.
						\note		The names "All", "None", "Cyan", "Magenta", "Yellow", and "Black" are reserved by
									Illustrator.
					*/
					virtual bool makeNamed(
						const std::string& name_,
						const double tint_ = 0.0,
						const NamedFlags flags_ = SpotColorFlag
					);
				
					/**
						\brief		Gets the tint for a "named" color, if any
						\author		GW
						\date		06/2022
						
						\returns	The tint for the color if it is named, 0 otherwise
					*/
					virtual double namedTint() const;

					/**
						\brief		Gets the flags for a "named" color, if any
						\author		GW
						\date		06/2017
						
						\returns	The flags for the color if it is named, NoFlag if none are present or if the color is
									not named
					*/
					virtual NamedFlags namedFlags() const;
				#endif
				// HDI_CORE_AIP_MODE

				/**
					\brief		Allows one ArtColor object to be assigned from another
					\author		GW
					\date		08/2013

					\param		rhs_	Righthand side of the = operator; the object to copy values from
					\returns	The target ArtColor object, but with its value updated to match that of rhs_
				*/
				virtual ArtColor& operator=(const ArtColor& rhs_);

				/**
					\brief		Allows two ArtColor objects to be compared with the == operator
					\author		GW
					\date		08/2013
					
					\param		rhs_	Righthand side of the == operator
					\returns	true if the objects are equal, false otherwise

					\note		Two ArtColor objects are equal when their types and components are equal.
				*/
				virtual bool operator==(const ArtColor& rhs_) const;

				/**
					\brief		Allows two ArtColor objects to be compared with the != operator
					\author		GW
					\date		08/2013
					
					\param		rhs_	Righthand side of the != operator
					\returns	true if the objects are inequal, false otherwise

					\note		Two ArtColor objects are unequal when their types or components are unequal.
				*/
				virtual bool operator!=(const ArtColor& rhs_) const;

				/**
					\brief	Static accessor to return an ArtColor object for pure red
					\author	GW
					\date	08/2013
				*/
				static ArtColor red();

				/**
					\brief	Static accessor to return an ArtColor object for pure orange
					\author	GW
					\date	08/2013
				*/
				static ArtColor orange();

				/**
					\brief	Static accessor to return an ArtColor object for pure yellow
					\author	GW
					\date	08/2013
				*/
				static ArtColor yellow();

				/**
					\brief	Static accessor to return an ArtColor object for pure green
					\author	GW
					\date	08/2013
				*/
				static ArtColor green();

				/**
					\brief	Static accessor to return an ArtColor object for pure cyan
					\author	GW
					\date	08/2013
				*/
				static ArtColor cyan();

				/**
					\brief	Static accessor to return an ArtColor object for pure blue
					\author	GW
					\date	08/2013
				*/
				static ArtColor blue();

				/**
					\brief	Static accessor to return an ArtColor object for pure purple
					\author	GW
					\date	08/2013
				*/
				static ArtColor purple();

				/**
					\brief	Static accessor to return an ArtColor object for pure magenta
					\author	GW
					\date	08/2013
				*/
				static ArtColor magenta();

				/**
					\brief	Static accessor to return an ArtColor object for pure white (0% black)
					\author	GW
					\date	08/2013
				*/
				static ArtColor white();

				/**
					\brief	Static accessor to return an ArtColor object for pure white (25% black)
					\author	GW
					\date	08/2013
				*/
				static ArtColor lightGray();

				/**
					\brief	Static accessor to return an ArtColor object for pure white (50% black)
					\author	GW
					\date	08/2013
				*/
				static ArtColor gray();

				/**
					\brief	Static accessor to return an ArtColor object for pure white (75% black)
					\author	GW
					\date	08/2013
				*/
				static ArtColor darkGray();

				/**
					\brief	Static accessor to return an ArtColor object for pure white (100% black)
					\author	GW
					\date	08/2013
				*/
				static ArtColor black();

				/**
					\brief		Static accessor to return an ArtColor object for "none"
					\details	Illustrator typically interprets a "none" color as completely transparent
					\author		GW
					\date		08/2013
				*/
				static ArtColor none();
			
				#if defined(HDI_CORE_PSP_MODE)
					/**
						\brief		While an ArtColor object can be constructed from a variety of color models and freely
									converted to other color models, the process is not performant due to the overhead
									of constructing C++ objects. This static function allows for much faster conversion
									of individual 8-bit color components.
						\author		GW
						\date		03/2018
					 
						\param		from_	The input color model
						\param		in1_	The first input color component (e.g. cyan)
						\param		in2_	The second input color component (e.g. magenta)
						\param		in3_	The third input color component (e.g. yellow)
						\param		in4_	The fourth input color component (e.g. black)
						\param		to_		The desired output color model
						\param		out1_	Return-by-reference for the first output color component (e.g. red)
						\param		out2_	Return-by-reference for the first output color component (e.g. green)
						\param		out3_	Return-by-reference for the first output color component (e.g. blue)
						\param		out4_	Return-by-reference for the first output color component (e.g. unused for RGB)
						\returns	true for success, false otherwise
					 
					 	\note		While still much faster than constructing ArtColor objects for conversion, it is
					 				worth noting that this method is slower than the underlying color conversion calls
					 				in the Photoshop SDK (because this method transposes some results and fixes various
					 				idiosyncrasies in the native call).
						\note		For any given color format, the input and output arguments should be ordered the
									same as the format initialism (i.e. R, G, B, unused; H, S, B, unused; L, a, b,
									unused; and C, M, Y, K).
						\note		All input and output values are necessarily within the range [0,255]. To convert a
									gray value to [0.0,1.0], divide by 255. To convert CMYK components to [0.0,1.0],
									divide each by 255. To convert Lab components to their respective ranges, divide "L"
									by 255 and subtract 128 from both "a" and "b". To convert HSB components to their
									respective ranges, multiply "H" by 360/255 and divide both "S" and "B" by 255.
					*/
					static bool fastConvert8(
						const Type from_,
						const uint8_t in1_,
						const uint8_t in2_,
						const uint8_t in3_,
						const uint8_t in4_,
						const Type to_,
						uint8_t& out1_,
						uint8_t& out2_,
						uint8_t& out3_,
						uint8_t& out4_
					);
			
					/**
						\brief		While an ArtColor object can be constructed from a variety of color models and freely
									converted to other color models, the process is not performant due to the overhead
									of constructing C++ objects. This static function allows for much faster conversion
									of individual 16-bit color components.
						\author		GW
						\date		03/2018
					 
						\param		from_	The input color model
						\param		in1_	The first input color component (e.g. cyan)
						\param		in2_	The second input color component (e.g. magenta)
						\param		in3_	The third input color component (e.g. yellow)
						\param		in4_	The fourth input color component (e.g. black)
						\param		to_		The desired output color model
						\param		out1_	Return-by-reference for the first output color component (e.g. red)
						\param		out2_	Return-by-reference for the first output color component (e.g. green)
						\param		out3_	Return-by-reference for the first output color component (e.g. blue)
						\param		out4_	Return-by-reference for the first output color component (e.g. unused for RGB)
						\returns	true for success, false otherwise
					 
					 	\note		While still much faster than constructing ArtColor objects for conversion, it is
					 				worth noting that this method is slower than the underlying color conversion calls
					 				in the Photoshop SDK (because this method transposes some results and fixes various
					 				idiosyncrasies in the native call).
						\note		For any given color format, the input and output arguments should be ordered the
									same as the format initialism (i.e. R, G, B, unused; H, S, B, unused; L, a, b,
									unused; and C, M, Y, K).
						\note		All input and output values are necessarily within the range [0,32768]. To convert a
									gray value to [0.0,1.0], divide by 32768. To convert CMYK components to [0.0,1.0],
									divide each by 32768. To convert Lab components to their respective ranges, divide "L"
									by 32768 and subtract 16384 from both "a" and "b". To convert HSB components to their
									respective ranges, multiply "H" by 360/32768 and divide both "S" and "B" by 32768.
					*/
					static bool fastConvert16(
						const Type from_,
						const uint16_t in1_,
						const uint16_t in2_,
						const uint16_t in3_,
						const uint16_t in4_,
						const Type to_,
						uint16_t& out1_,
						uint16_t& out2_,
						uint16_t& out3_,
						uint16_t& out4_
					);
			
					/**
						\brief		While the fastConvert8() method is faster than ArtColor object-based conversion for
									individual color components, this method is faster for converting an entire bitmap
									(conventionally-encoded) from one model to another.
						\author		GW
						\date		03/2018
					 
						\param		from_		The input color model
						\param		to_			The desired output color model
						\param		bitmap__	A pointer to the first pixel in the bitmap for input, and the first
												converted pixel after conversion
						\param		nPixels_	The number of pixels in the bitmap (NOT the number of bytes)
					 
					 	\note		The input and output pixels are expected to be formatted as such, for each supported
					 				color model: RGB is 0RGB, CMYK is simply CMYK, Lab is 0LAB, HSB is 0HSB, and Gray is
					 				000g (lowercase "g" for "gray").
						\note		The pixels are expected to have their values stored according to the convention for
					 				the (input) color model. For example, RGB values are simply (0,R,B,G), but CMYK values
					 				are (255-C,255-M,255-Y,255-K), where each item in the aforementioned sets is an 8-byte
					 				segment of a pixel. If you are passing bitmap data directly from Photoshop to this
					 				method, then the pixels will already be stored as needed.
					*/
					static bool fastConvertBitmap8(
						const Type from_,
						const Type to_,
						uint32_t* const bitmap__,
						const uint32_t nPixels_
					);
			
					/**
						\brief		While the fastConvert16() method is faster than ArtColor object-based conversion for
									individual color components, this method is faster for converting an entire bitmap
									(conventionally-encoded) from one model to another.
						\author		GW
						\date		03/2018
					 
						\param		from_		The input color model
						\param		to_			The desired output color model
						\param		bitmap__	A pointer to the first 64-bit pixel in the bitmap, and the first
												converted pixel after conversion
						\param		nPixels_	The number of pixels in the bitmap (NOT the number of bytes)
					 
					 	\note		The input and output pixels are expected to be formatted as such, for each supported
					 				color model: RGB is 0RGB, CMYK is simply CMYK, Lab is 0LAB, HSB is 0HSB, and Gray is
					 				000g (lowercase "g" for "gray").
						\note		The pixels are expected to have their values stored according to the convention for
					 				the (input) color model. For example, RGB values are simply (0,R,B,G), but CMYK values
					 				are (255-C,255-M,255-Y,255-K), where each item in the aforementioned sets is a 16-byte
					 				segment of a pixel. If you are passing bitmap data directly from Photoshop to this
					 				method, then the pixels will already be stored as needed.
					*/
					static bool fastConvertBitmap16(
						const Type from_,
						const Type to_,
						uint64_t* const bitmap__,
						const uint32_t nPixels_
					);
				#endif
				// HDI_CORE_PSP_MODE


			private:
				#if defined(HDI_CORE_AIP_MODE)
					friend aip::ArtColor* __accessImpl(const ArtColor&);
					friend ArtColor __accessCtor(const aip::ArtColor&);

					/**
						\brief	Private implementation object
					*/
					aip::ArtColor* __impl;

					/**
						\brief	Internal use only
						\author	GW
						\date	08/2013
					*/
					ArtColor(const aip::ArtColor&);
				#elif defined(HDI_CORE_PSP_MODE)
					/**
						\brief	Private implementation object
					*/
					void* __impl;
				#endif
		};
		
		typedef std::unique_ptr<ArtColor> ArtColorUP;
		typedef std::shared_ptr<ArtColor> ArtColorSP;
		typedef std::weak_ptr<ArtColor> ArtColorWP;
		
		#if defined(HDI_CORE_AIP_MODE)
			extern aip::ArtColor* __accessImpl(const ArtColor&);
			extern ArtColor __accessCtor(const aip::ArtColor&);
		#endif
		// HDI_CORE_AIP_MODE
	}
}

#if defined(HDI_CORE_AIP_MODE)
	hdi::core::ArtColor::NamedFlags operator|(
		const hdi::core::ArtColor::NamedFlags lhs_,
		const hdi::core::ArtColor::NamedFlags rhs_
	);

	hdi::core::ArtColor::NamedFlags& operator|=(
		hdi::core::ArtColor::NamedFlags& lhs__,
		const hdi::core::ArtColor::NamedFlags rhs_
	);
#endif
// HDI_CORE_AIP_MODE

#endif
// __HDI_CORE_ART_COLOR__
