/**
	\file
	\brief		Header file for artboard rectangle preference data storage
	\copyright	Hot Door, Inc. 2010-2025
*/

#ifndef __HDI_CORE_ARTBOARD_RECT_PREF__
#define __HDI_CORE_ARTBOARD_RECT_PREF__

#if defined(HDI_CORE_AIP_MODE)

#include "hdicoreArtboardRect.h"
#include "hdicorePrefData.h"

namespace hdi
{
	namespace core
	{
		/**
			\brief	Allows for storage of artboard rects/bounds
		*/
		class ArtboardRectPref : public PrefData
		{
			public:
				/**
					\brief	Constructs an empty ArtboardRectPref object
					\author	GW
					\date	09/2013
					
					\note	To test if a ArtboardRectPref object is empty, call isEmpty() on it
					\note	Empty ArtboardRectPref objects do not relate to any actual preference data; they are
							designed to be "receivers" of some other ArtboardRectPref object via the overloaded
							assignment operator. Empty ArtboardRectPref objects are useless until such time (though it
							is safe to call any of their methods).
				*/
				ArtboardRectPref();
			
				/**
					\brief	Constructs a new ArtboardRectPref object from an existing ArtboardRectPref object (copy
							constructor)
					\author	GW
					\date	09/2013

					\param	abr_	Existing ArtboardRectPref object
				*/
				ArtboardRectPref(const ArtboardRectPref& abr_);

				/**
					\brief	Constructs an ArtboardRectPref object, with new data, to be added to a container at a later
							time
					\author	GW
					\date	09/2013

					\param	value_	Initial rect value
					\param	name_	Name for the data once it has been added to its container, as UTF-8

					\note	The name_ argument can only be "" (omitted) if the object will be added directly to an
							ArrayPref object.
				*/
				ArtboardRectPref(const ArtboardRect& value_, const std::string& name_ = "");

				/**
					\brief	Destructs an ArtboardRectPref object
					\author	GW
					\date	09/2013
				*/
				virtual ~ArtboardRectPref();

				/**
					\brief		Allows one ArtboardRectPref object to be assigned from another
					\author		GW
					\date		09/2013

					\param		rhs_	Righthand side of the = operator; the object to copy values from
					\returns	The target ArtboardRectPref object, but with its value updated to match that of rhs_
				*/
				virtual ArtboardRectPref& operator=(const ArtboardRectPref& rhs_);
				
				/**
					\brief		Convenience method to clone an ArtboardRectPref object on the heap
					\author		GW
					\date		10/2013

					\returns	A pointer to the new ArtboardRectPref object
					
					\note		If you subclass ArtboardRectPref, you MUST overload this method yourself! If you don't
								and/or your clone() method does not return an instance of your ArtboardRectPref
								subclass, you will experience "object slicing" when adding the pref to a container.

					\warning	The caller must manage the memory for the returned ArtboardRectPref object.
				*/
				virtual ArtboardRectPref* clone() const;

				/**
					\brief		Gets the rect for this object
					\author		GW
					\date		09/2013

					\returns	The ArtboardRect object being described by this wrapper
				*/
				virtual ArtboardRect artboardRect() const;

				/**
					\brief		Sets the rect for this object
					\author		GW
					\date		09/2013

					\param		rect_	New rect value
					\returns	true if the rect is successfully updated, or false otherwise
				*/
				virtual bool setArtboardRect(const ArtboardRect& rect_);
		};
		
		typedef std::unique_ptr<ArtboardRectPref> ArtboardRectPrefUP;
		typedef std::shared_ptr<ArtboardRectPref> ArtboardRectPrefSP;
		typedef std::weak_ptr<ArtboardRectPref> ArtboardRectPrefWP;
	}
}

#endif
// HDI_CORE_AIP_MODE

#endif
// __HDI_CORE_ARTBOARD_RECT_PREF__
