/**
	\file
	\brief		Header file for Illustrator path segment point manipulation
	\copyright	Hot Door, Inc. 2010-2025
*/

#ifndef __HDI_CORE_ARTBOARD_SEG_POINT__
#define __HDI_CORE_ARTBOARD_SEG_POINT__

#if defined(HDI_CORE_AIP_MODE)

#include "hdicoreArtboardPoint.h"

namespace hdi
{
	namespace aip
	{
		class PathSegPoint;
	}

	namespace core
	{
		/**
			\brief	Describes a segment point of a bezier path on the Illustrator artboard
		*/
		class ArtboardSegPoint
		{
			public:
				/**
					\brief	Anchor point
				*/
				ArtboardPoint p;

				/**
					\brief	Segment "in" control point
				*/
				ArtboardPoint in;

				/**
					\brief	Segment "out" control point
				*/
				ArtboardPoint out;

				/**
					Whether the seg point is a corner
				*/
				bool corner;

				/**
					\brief	Constructs an ArtboardSegPoint object with the p, in, and out points at (0,0) and corner set
							to false
					\author	GW
					\date	08/2013
				*/
				ArtboardSegPoint();

				/**
					\brief	ArtboardSegPoint copy constructor
					\author	GW
					\date	08/2013

					\param	segPt_	Existing ArtboardSegPoint object to copy values from
				*/
				ArtboardSegPoint(const ArtboardSegPoint& segPt_);

				/**
					\brief	Constructs an ArtboardSegPoint from a given point for the anchor and both control points
					\author	GW
					\date	08/2013

					\param	pt_			Anchor point, control points
					\param	corner_		Whether the segment point is a corner
				*/
				ArtboardSegPoint(const ArtboardPoint& pt_, const bool corner_ = true);

				/**
					\brief	Constructs an ArtboardSegPoint from given points
					\author	GW
					\date	08/2013

					\param	p_			Anchor point
					\param	in_			"In" control point
					\param	out_		"Out" control point
					\param	corner_		Whether the segment point is a corner
				*/
				ArtboardSegPoint(
					const ArtboardPoint& p_,
					const ArtboardPoint& in_,
					const ArtboardPoint& out_,
					const bool corner_ = true
				);

				/**
					\brief	ArtboardSegPoint destructor
					\author	GW
					\date	08/2013
				*/
				virtual ~ArtboardSegPoint();

				/**
					\brief		Sets an ArtboardSegPoint object from another ArtboardSegPoint object
					\author		GW
					\date		08/2013

					\param		rhs_	Righthand side of the = operator; the ArtboardSegPoint to pull values from
					\returns	The target ArtboardSegPoint object, with its values updated
				*/
				virtual ArtboardSegPoint& operator=(const ArtboardSegPoint& rhs_);
				
				/**
					\brief		Tests whether a given ArtboardSegPoint object is the same as another
					\author		GW
					\date		08/2013

					\param		rhs_	ArtboardSegPoint to compare against (righthand side of equality operator)
					\returns	true for the target and rhs_ being the same segment point, false otherwise
				*/
				virtual bool operator==(const ArtboardSegPoint& rhs_) const;
				
				/**
					\brief		Tests whether a given ArtboardSegPoint object is not the same as another
					\author		GW
					\date		08/2013

					\param		rhs_	ArtboardSegPoint to compare against (righthand side of inequality operator)
					\returns	true for the target and rhs_ being different segment points, false otherwise
				*/
				virtual bool operator!=(const ArtboardSegPoint& rhs_) const;

				/**
					\brief		Gets whether the path formed by two segment points is curved
					\author		GW
					\date		08/2013

					\param		segPt2_		Second segment point in the path
					\returns	true if the path formed by the two points is curved, false otherwise
				*/
				virtual bool curved(const ArtboardSegPoint& segPt2_) const;


			private:
				friend aip::PathSegPoint* __accessImpl(const ArtboardSegPoint&);
				friend ArtboardSegPoint __accessCtor(const aip::PathSegPoint&);

				/**
					\brief	Private implementation object
				*/
				aip::PathSegPoint* __impl;

				/**
					\brief	Internal use only
					\author	GW
					\date	08/2013
				*/
				ArtboardSegPoint(const aip::PathSegPoint&);
		};
		
		typedef std::unique_ptr<ArtboardSegPoint> ArtboardSegPointUP;
		typedef std::shared_ptr<ArtboardSegPoint> ArtboardSegPointSP;
		typedef std::weak_ptr<ArtboardSegPoint> ArtboardSegPointWP;
		
		extern aip::PathSegPoint* __accessImpl(const ArtboardSegPoint&);
		extern ArtboardSegPoint __accessCtor(const aip::PathSegPoint&);
	}
}

#endif
// HDI_CORE_AIP_MODE

#endif
// __HDI_CORE_ARTBOARD_SEG_POINT__
