/**
	\file
	\brief		Header file for plugin user interface button widgets
	\copyright	Hot Door, Inc. 2010-2025
*/

#ifndef __HDI_CORE_BUTTON__
#define __HDI_CORE_BUTTON__

#include "hdicoreWidget.h"

namespace hdi
{
	namespace core
	{
		class Callback;

		/**
			\brief	Button widget, which can be either the default button, cancel button, or normal button in a dialog,
					and optionally contain an image
		*/
		class Button : public Widget
		{
			public:
				/**
					\brief	Describes the style of the button (see descriptions for each)
				*/
				enum StyleType
				{
					UnknownStyle	= 0,
					CancelStyle		= 10,	// Indicates the cancel functionality of a window (cancel keyboard shortcut)
					DefaultStyle	= 20,	// Indicates the default functionality of a window (accept keyboard shortcut)
					NormalStyle		= 30	// Just a normal button with no styling
				};

				/**
					\brief	Constructs an empty Button object
					\author	GW
					\date	09/2013
					
					\note	To test if a Button object is empty, call isEmpty() on it
					\note	Empty Button objects do not relate to any actual UI widget; they are designed to be
							"receivers" of some other Button object via the overloaded assignment operator. Empty Button
							objects are useless until such time (though it is safe to call any of their methods).
				*/
				Button();

				/**
					\brief	Constructs a new Button object from an existing Button object (copy constructor)
					\author	GW
					\date	09/2013

					\param	button_		Existing Button object
				*/
				Button(const Button& button_);

				/**
					\brief	Constructs a button using a location and window container type; height is fixed
					\author	GW
					\date	09/2013

					\param	loc_			Top-left location of the button, in 1x resolution coordinates
					\param	title_			Title of the button, as UTF-8
					\param	containerType_	Type of window that will contain the widget
					
					\note	The containerType_ argument is necessary for this constructor because it computes the ideal
							width for the new button from the initial button title string. However, Adobe uses different
							font sizes for different types of windows, and we can't correctly compute the width without
							knowing what font size to use.
				*/
				Button(const Point& loc_, const std::string& title_, const WindowType containerType_);

				/**
					\brief	Constructs a button using a location and width; height is fixed
					\author	GW
					\date	09/2013

					\param	loc_		Top-left location of the button, in 1x resolution coordinates
					\param	title_		Title of the button, as UTF-8
					\param	width_		Width of the button, at 1x resolution
				*/
				Button(const Point& loc_, const std::string& title_, const double width_);
				
				/**
					\brief	Constructs a text button of a specific size
					\author	GW
					\date	09/2013

					\param	frame_	Position and size of the button, in 1x resolution coordinates
					\param	title_	Title of the button, as UTF-8
				*/
				Button(const Rect& frame_, const std::string& title_);
			
				/**
					\brief	Constructs an image button of a certain size
					\author	GW
					\date	09/2013

					\param	frame_		Position and size of the button, in 1x resolution coordinates
					\param	imageIDs_	PNGI resource IDs
				*/
				Button(const Rect& frame_, const ImageWithRolloverIDs& imageIDs_);
			
				/**
					\brief	Constructs an image button of a certain size
					\author	GW
					\date	12/2018

					\param	frame_		Position and size of the button, in 1x resolution coordinates
					\param	svgIDs_		SVG resource IDs
					\param	dims_		1x dimensions to use when rendering the SVG; the default value causes the widget
										frame dimensions to be used instead
				*/
				Button(const Rect& frame_, const SVGWithRolloverIDs& svgIDs_, const Size& dims_ = Size());
				
				/**
					\brief	Button destructor
					\author	GW
					\date	09/2013
				*/
				virtual ~Button();

				/**
					\brief		Allows one Button object to be assigned from another
					\author		GW
					\date		10/2013

					\param		rhs_	Righthand side of the = operator; the object to copy values from
					\returns	The target Button object, but with its value updated to match that of rhs_
				*/
				virtual Button& operator=(const Button& rhs_);
				
				/**
					\brief		Convenience method to clone a Button object on the heap
					\author		GW
					\date		10/2013

					\returns	A pointer to the new Button object
					
					\note		If you subclass Button, you MUST overload this method yourself! If you don't and/or your
								clone() method does not return an instance of your Button subclass, you will experience
								"object slicing" when adding the widget to a window.

					\warning	The caller must manage the memory for the returned Button object.
				*/
				virtual Button* clone() const;
				
				/**
					\brief		Convenience method to duplicate a Button object, creating a new and identical UI element
								to the target (but not belonging to the same containing window)
					\author		GW
					\date		11/2013
					
					\returns	A pointer to the new Button object (and new UI element)
					
					\warning	The caller must manage the memory for the returned Button object.
				*/
				virtual Button* duplicate() const;

				/**
					\brief		Gets the current text value of the button
					\author		GW
					\date		09/2013
					
					\returns	The current text value, as UTF-8

					\note		If the widget is an image button, this returns ""
				*/
				virtual std::string text() const;

				/**
					\brief	Sets the current text value of the button
					\author	GW
					\date	09/2013

					\param	text_	New text value, as UTF-8

					\note	If the widget is an image button, the function will bail early
				*/
				virtual void setText(const std::string& text_);
				
				/**
					\brief		Gets the current button style
					\author		GW
					\date		09/2013
					
					\returns	StyleType enum describing the style
				*/
				virtual StyleType style() const;

				/**
					\brief	Sets the style of the button as the default, cancel, or normal button
					\author	GW
					\date	09/2013

					\param	style_	New button style
				*/
				virtual void setStyle(const StyleType style_);

				/**
					\brief	Simulates a click on the button
					\author	GW
					\date	09/2013
				*/
				virtual void click();
				
				/**
					\brief		Gets the current click callback for the button
					\author		GW
					\date		09/2013

					\returns	A pointer to the currently registered click callback
				*/
				virtual Callback* const clickCallback() const;
				
				/**
					\brief	Sets the click callback for the button
					\author	GW
					\date	09/2013

					\param	callback_	New callback for when the button is clicked
				*/
				virtual void setClickCallback(const Callback& callback_);

				/**
					\brief	Simulates a double-click on the button
					\author	GW
					\date	03/2015
				*/
				virtual void doubleClick();
				
				/**
					\brief		Gets the current double-click callback for the button
					\author		GW
					\date		03/2015

					\returns	A pointer to the currently registered double-click callback
				*/
				virtual Callback* const doubleClickCallback() const;
				
				/**
					\brief	Sets the double-click callback for the button
					\author	GW
					\date	03/2015

					\param	callback_	New callback for when the button is double-clicked
				*/
				virtual void setDoubleClickCallback(const Callback& callback_);
			
				/**
					\brief		Gets the PNGI resource IDs of the widget
					\author		GW
					\date		03/2015

					\returns	The IDs of the PNGI image resources
				*/
				virtual ImageWithRolloverIDs imageIDs() const;
			
				/**
					\brief	Sets the PNGI resource IDs of the widget
					\author	GW
					\date	03/2015

					\param	ids_	New PNGI image resource IDs
				*/
				virtual void setImageIDs(const ImageWithRolloverIDs& ids_);
			
				/**
					\brief		Gets the SVG resource IDs of the widget
					\author		GW
					\date		12/2018

					\returns	The IDs of the SVG image resources
				*/
				virtual SVGWithRolloverIDs svgIDs() const;
			
				/**
					\brief	Sets the SVG resource IDs of the widget
					\author	GW
					\date	12/2018

					\param	ids_	New SVG image resource IDs
				*/
				virtual void setSVGIDs(const SVGWithRolloverIDs& ids_);
				
				/**
					\brief		Gets whether the background of the button should change in appearance upon mouse-over
					\author		GW
					\date		03/2015
					
					\returns	true for showing a rollover effect, false otherwise
				*/
				virtual bool backgroundRollover() const;
				
				/**
					\brief	Sets whether the background of the button should change in appearance upon mouse-over
					\author	GW
					\date	03/2015
					
					\param	ro_		true to show a rollover effect, false otherwise
				*/
				virtual void setBackgroundRollover(const bool ro_);
			
				/**
					\brief		Gets the ideal height for a button, given its title and a known width
					\author		GW
					\date		09/2013

					\param		text_			Button title, as UTF-8
					\param		width_			Known width into which the button must fit
					\param		containerType_	Type of window for the widget (because font sizes differ)
					\returns	Ideal height for the given text to fit into the given width, at 1x resolution
				*/
				static double idealHeight(const std::string& text_, const double width_, const WindowType containerType_);

				/**
					\brief		Gets the ideal width for a button, given its title and a known height
					\author		GW
					\date		09/2013

					\param		text_			Button title, as UTF-8
					\param		height_			Known height into which the button must fit
					\param		containerType_	Type of window for the widget (because font sizes differ)
					\returns	Ideal width for the given text to fit into the given height, at 1x resolution 
				*/
				static double idealWidth(const std::string& text_, const double height_, const WindowType containerType_);
		};
		
		typedef std::unique_ptr<Button> ButtonUP;
		typedef std::shared_ptr<Button> ButtonSP;
		typedef std::weak_ptr<Button> ButtonWP;
	}
}

#endif
// __HDI_CORE_BUTTON__
