/**
	\file
	\brief		Header file for describing plugin UI colors
	\copyright	Hot Door, Inc. 2010-2025
*/

#ifndef __HDI_CORE_COLOR__
#define __HDI_CORE_COLOR__

#include <memory>

namespace hdi
{
	#if defined(HDI_CORE_AIP_MODE)
		namespace aip
		{
			class Color;
		}
	#elif defined(HDI_CORE_PSP_MODE)
		namespace psp
		{
			class Color;
		}
	#endif

	namespace core
	{
		class ArtColor;
	
		/**
			\brief	Describes an RGB color, with opacity, typically for UI purposes
		*/
		class Color
		{
			public:
				/**
					\brief	Red channel, in range [0.0, 1.0]
				*/
				double r;

				/**
					\brief	Green channel, in range [0.0, 1.0]
				*/
				double g;

				/**
					\brief	Blue channel, in range [0.0, 1.0]
				*/
				double b;

				/**
					\brief	Alpha channel, in range [0.0, 1.0]
				*/
				double opacity;

				/**
					\brief	Default Color constructor; makes a black and fully opaque color
					\author	GW
					\date	08/2013
				*/
				Color();

				/**
					\brief	Color constructor, taking an argument for each channel
					\author	GW
					\date	08/2013

					\param	r_	Value for red channel, in range [0.0, 1.0]
					\param	g_	Value for green channel, in range [0.0, 1.0]
					\param	b_	Value for blue channel, in range [0.0, 1.0]
					\param	o_	Value for alpha channel, in range [0.0, 1.0]
				*/
				Color(const double r_, const double g_, const double b_, const double o_ = 1.0);

				/**
					\brief	Color copy constructor
					\author	GW
					\date	08/2013

					\param	c_	Existing Color object to copy values from
				*/
				Color(const Color& c_);

				/**
					\brief		Constructs a Color object from an existing ArtColor object
					\author		GW
					\date		07/2025

					\param		c_	Existing ArtColor object
					
					\warning	This constructor cannot accomodate ArtColor objects composed of patterns or gradients;
								passing objects of those ArtColor::Type values will result in RGB black with opacity of 0.
				*/
				Color(const ArtColor& c_);
		
				/**
					\brief	Color destructor
					\author	GW
					\date	08/2013
				*/
				virtual ~Color();
				
				/**
					\brief	Lightens the color by a percentage
					\author	GW
					\date	08/2013

					\param	percentage_		Amount to lighten the color brightness, in range [0.0, 1.0]
				*/
				virtual Color lighten(const double percentage_) const;

				/**
					\brief	Darkens the color by a percentage
					\author	GW
					\date	08/2013

					\param	percentage_		Amount to darken the color brightness, in range [0.0, 1.0]
				*/
				virtual Color darken(const double percentage_) const;

				/**
					\brief		Sets a Color object from another Color object
					\author		GW
					\date		08/2013

					\param		rhs_	Color object to copy values from
					\returns	The target object on the lefthand side of the = operator, with its values updated
				*/
				virtual Color& operator=(const Color& rhs_);
				
				/**
					\brief		Tests whether a given Color object is the same as another
					\author		GW
					\date		08/2013

					\param		rhs_	Color to compare against (righthand side of equality operator)
					\returns	true for the target and rhs_ being the same color, false otherwise
				*/
				virtual bool operator==(const Color& rhs_) const;
				
				/**
					\brief		Tests whether a given Color object is not the same as another
					\author		GW
					\date		08/2013

					\param		rhs_	Color to compare against (righthand side of inequality operator)
					\returns	true for the target and rhs_ being different colors, false otherwise
				*/
				virtual bool operator!=(const Color& rhs_) const;

				/**
					\brief		Static accessor to get a pure red Color object
					\author		GW
					\date		08/2013

					\returns	A Color object with its values set to pure red
				*/
				static Color red();

				/**
					\brief		Static accessor to get a pure orange Color object
					\author		GW
					\date		08/2013

					\returns	A Color object with its values set to pure orange
				*/
				static Color orange();

				/**
					\brief		Static accessor to get a pure yellow Color object
					\author		GW
					\date		08/2013

					\returns	A Color object with its values set to pure yellow
				*/
				static Color yellow();

				/**
					\brief		Static accessor to get a pure green Color object
					\author		GW
					\date		08/2013

					\returns	A Color object with its values set to pure green
				*/
				static Color green();

				/**
					\brief		Static accessor to get a pure cyan Color object
					\author		GW
					\date		08/2013

					\returns	A Color object with its values set to pure cyan
				*/
				static Color cyan();

				/**
					\brief		Static accessor to get a pure blue Color object
					\author		GW
					\date		08/2013

					\returns	A Color object with its values set to pure blue
				*/
				static Color blue();

				/**
					\brief		Static accessor to get a pure purple Color object
					\author		GW
					\date		08/2013

					\returns	A Color object with its values set to pure purple
				*/
				static Color purple();

				/**
					\brief		Static accessor to get a pure magenta Color object
					\author		GW
					\date		08/2013

					\returns	A Color object with its values set to pure magenta
				*/
				static Color magenta();

				/**
					\brief		Static accessor to get a pure white Color object
					\author		GW
					\date		08/2013

					\returns	A Color object with its values set to pure white
				*/
				static Color white();

				/**
					\brief		Static accessor to get a light gray Color object
					\author		GW
					\date		08/2013

					\returns	A Color object with its values set to 25% black
				*/
				static Color lightGray();

				/**
					\brief		Static accessor to get a neutral gray Color object
					\author		GW
					\date		08/2013

					\returns	A Color object with its values set to 50% black
				*/
				static Color gray();

				/**
					\brief		Static accessor to get a dark gray Color object
					\author		GW
					\date		08/2013

					\returns	A Color object with its values set to 75% black
				*/
				static Color darkGray();

				/**
					\brief		Static accessor to get a pure black Color object
					\author		GW
					\date		08/2013

					\returns	A Color object with its values set to pure black
				*/
				static Color black();

				/**
					\brief		Static accessor to get a Color object with opacity of 0
					\author		GW
					\date		07/2025

					\returns	A Color object with its values set to pure black and opacity of 0
				*/
				static Color clear();


			private:
				#if defined(HDI_CORE_AIP_MODE)
					friend aip::Color* __accessImpl(const Color&);
					friend Color __accessCtor(const aip::Color&);
					
					/**
						\brief	Private implementation object
					*/
					aip::Color* __impl;

					/**
						\brief	Internal use only
						\author	GW
						\date	08/2013
					*/
					Color(const aip::Color&);
				#elif defined(HDI_CORE_PSP_MODE)
					friend psp::Color* __accessImpl(const Color&);
					friend Color __accessCtor(const psp::Color&);
					
					/**
						\brief	Private implementation object
					*/
					psp::Color* __impl;

					/**
						\brief	Internal use only
						\author	GW
						\date	12/2017
					*/
					Color(const psp::Color&);
				#endif

		};
		
		typedef std::unique_ptr<Color> ColorUP;
		typedef std::shared_ptr<Color> ColorSP;
		typedef std::weak_ptr<Color> ColorWP;
		
		#if defined(HDI_CORE_AIP_MODE)
			extern aip::Color* __accessImpl(const Color&);
			extern Color __accessCtor(const aip::Color&);
		#elif defined(HDI_CORE_PSP_MODE)
			extern psp::Color* __accessImpl(const Color&);
			extern Color __accessCtor(const psp::Color&);
		#endif
	}
}

#endif
// __HDI_CORE_COLOR__
