/**
	\file
	\brief		Header file for performing hashing, encoding, or encrypting data
	\copyright	Hot Door, Inc. 2010-2025
*/

#ifndef __HDI_CORE_CRYPT__
#define __HDI_CORE_CRYPT__

#include <vector>

#include "hdicoreTypes.h"

namespace hdi
{
	namespace plat
	{
		class AESKey;
		class AESInitializationVector;
		class PublicRSAKey;
		class PrivateRSAKey;
	}

	namespace core
	{
		/**
			\brief	Organizes all AES, RSA, and other basic cryptographic functions into one namespace for convenience
		*/
		namespace crypt
		{
			/**
				\brief		Performs a rot-13 cipher (Caeser's cipher) on a string
				\author		GW
				\date		08/2013

				\param		str_	String on which the rotation will be performed
				\returns	Value of str_ argument, but with all alphabetic characters rotated by 13 positions

				\note		Only the ASCII portion of the string will be rotated.
			*/
			std::string rot13(const std::string& str_);

			/**
				\brief		Generates an MD5 hash of a given source string
				\author		GW
				\date		05/2017
				
				\param		str_	String to hash with the MD5 algorithm (binary data inside of a string is also
									acceptable)
				\returns	A string representing the hexadecimal data of the MD5 hash value
			*/
			std::string md5(const std::string& str_);

			/**
				\brief		Generates an SHA1 hash of a given source string
				\author		GW
				\date		08/2013
				
				\param		str_	String to hash with the SHA1 algorithm (binary data inside of a string is also
									acceptable)
				\returns	A string representing the hexadecimal data of the SHA1 hash value
			*/
			std::string sha1(const std::string& str_);

			/**
				\brief		Generates an SHA2-256 hash of a given source string
				\author		GW
				\date		05/2017
				
				\param		str_	String to hash with the SHA2-256 algorithm (binary data inside of a string is also
									acceptable)
				\returns	A string representing the hexadecimal data of the SHA2-256 hash value
			*/
			std::string sha2_256(const std::string& str_);

			/**
				\brief		Generates an SHA2-512 hash of a given source string
				\author		GW
				\date		05/2017
				
				\param		str_	String to hash with the SHA2-512 algorithm (binary data inside of a string is also
									acceptable)
				\returns	A string representing the hexadecimal data of the SHA2-512 hash value
			*/
			std::string sha2_512(const std::string& str_);

			/**
				\brief		Base-64 encodes any data stuffed into a string
				\author		GW
				\date		08/2013

				\param		plain_	Plain text (or binary data stuffed in a string) to base-64 encode
				\returns	The value of the plain_ argument, but base-64 encoded
			*/
			std::string base64Encode(const std::string& plain_);

			/**
				\brief		Decodes any data previously base-64 encoded
				\author		GW
				\date		08/2013

				\param		cipher_		Base-64 encoded text to decode
				\returns	The value of the cipher_ argument, but base-64 decoded
			*/
			std::string base64Decode(const std::string& cipher_);
			
			namespace aes
			{
				class Key;
				class InitVector;
			}
				
			/**
				\brief	Organizes all AES functionality into one namespace for convenience
			*/
			namespace aes
			{
				typedef std::vector<byte> ByteVector;

				/**
					\brief		Encrypts a string (or raw bytes stuffed into a string) with an AES key
					\author		GW
					\date		10/2013
					
					\param		key_			AES key to use for encryption
					\param		iv_				Initialization vector to use for this encryption call
					\param		data_			Plain data to be encrypted (in the form of a string or binary data
												inside of a string object)
					\param		b64Cipher__		Return-by-reference for the base-64 encoded cipher text
					\returns	true if the data could be encrypted, false otherwise
					
					\note		To decrypt the cipher, use the decrypt() function (or equivalent - this lib is built on
								OpenSSL and the AES standard, so any base-64 decoder and function that understands AES
								decryption can decrypt the cipher with the correct key and initialization vector).
				*/
				bool encrypt(const Key& key_, const InitVector& iv_, const std::string& data_, std::string& b64Cipher__);
				
				/**
					\brief		Decrypts a base-64 encoded string of cipher text (previously encrypted with the encrypt()
								function or equivalent)
					\author		GW
					\date		10/2013
					
					\param		key_		AES key to use for decryption
					\param		iv_			Initialization vector to use for this decryption call
					\param		b64Cipher_	Base-64 encoded cipher text to attempt to decrypt
					\param		data__		Return-by-reference for the decrypted plain text (or raw bytes stuffed into
											a string)
					\returns	true if the data could be decrypted, false otherwise
				*/
				bool decrypt(const Key& key_, const InitVector& iv_, const std::string& b64Cipher_, std::string& data__);
				
				/**
					\brief	Wraps around an AES key's raw data for some added conveniences
				*/
				class Key
				{
					public:
						/**
							\brief	Constructs a Key object from another (copy constructor)
							\author	GW
							\date	10/2013
							
							\param	key_	Key object to copy values from
						*/
						Key(const Key& key_);
						
						/**
							\brief	Constructs a Key object from a previous serialization of the key (using the
									serialize() method)
							\author	GW
							\date	10/2013
							
							\param	serializedKey_	Base-64 encoded serialization of an AES key
						*/
						Key(const std::string& serializedKey_);
					
						/**
							\brief	Constructs a new(ly generated) Key object of a given size/length
							\author	GW
							\date	10/2013
							
							\param	size_	Size/length of the key in bits (must be 128, 192, or 256)
						*/
						Key(const int32_t size_);

						/**
							\brief	Constructs a new(ly generated) Key object of a given size/length and a predetermined
									set of bytes
							\author	GW
							\date	10/2013
							
							\param	size_	Size/length of the key in bits (must be 128, 192, or 256)
							\param	bytes_	Raw bytes to use for the AES key (must contain size_/8 bytes)
						*/
						Key(const int32_t size_, const ByteVector& bytes_);

						/**
							\brief	Destructs a Key object
							\author	GW
							\date	10/2013
						*/
						virtual ~Key();
						
						/**
							\brief		Assigns one Key object to another
							\author		GW
							\date		10/2013
							
							\param		rhs_	Righthand side of the assignment operator; key object to copy values from
							\returns	A reference to the target object (lefthand side of assignment operator), but
										with its values updated
						*/
						Key& operator=(const Key& rhs_);
						
						/**
							\brief		Gets whether the object contains a valid key
							\author		GW
							\date		10/2013
							
							\returns	true if the key is valid/not empty, false otherwise
						*/
						bool valid() const;
						
						/**
							\brief		Gets the size/length of the key in bits
							\author		GW
							\date		10/2013
							
							\returns	The number of bits for the key
						*/
						int32_t length() const;
						
						/**
							\brief		Gets the raw bytes of the AES key
							\author		GW
							\date		10/2013
							
							\returns	A container of all the raw byte values of the AES key
						*/
						ByteVector bytes() const;
						
						/**
							\brief		Serializes the key in a base-64 string; useful for transmitting the string to
										another party or writing the key to disk (not recommended without encrypting it
										first!)
							\author		GW
							\date		10/2013
							
							\returns	The serialized version of the key
							
							\note		A key serialization string can be converted back into a Key object at runtime
										with the Key(const std::string&) constructor.
						*/
						std::string serialize() const;
						
						/**
							\brief		Compares two Key objects for equality
							\author		GW
							\date		10/2013
							
							\param		rhs_	Righthand side of the equality operator; Key object to compare against
							\returns	true if the two objects have the same key values, false otherwise
						*/
						bool operator==(const Key& rhs_) const;
						
						/**
							\brief		Compares two Key objects for inequality
							\author		GW
							\date		10/2013
							
							\param		rhs_	Righthand side of the inequality operator; Key object to compare against
							\returns	true if the two objects have differing key values, false otherwise
						*/
						bool operator!=(const Key& rhs_) const;


					private:
						friend plat::AESKey* __accessImpl(const Key&);

						friend bool encrypt(const Key&, const InitVector&, const std::string&, std::string&);
						friend bool decrypt(const Key&, const InitVector&, const std::string&, std::string&);
					
						/**
							\brief	Private implementation object
						*/
						plat::AESKey* __impl;

						/**
							\brief	For internal use only
							\author	GW
							\date	10/2013
						*/
						Key();
				};
		
				typedef std::unique_ptr<Key> KeyUP;
				typedef std::shared_ptr<Key> KeySP;
				typedef std::weak_ptr<Key> KeyWP;
				
				/**
					\brief	Wraps around an AES initialization vector's raw data for some added conveniences
				*/
				class InitVector
				{
					public:
						/**
							\brief	Constructs a new(ly generated) InitVector object for some random initialization
									vector values
							\author	GW
							\date	10/2013
							
							\note	Initialization vectors are used to ensure that indentical plain values encoded with
									indentical keys do not result in indentical cipher values.
							\note	Always use same vector for one encryption and decryption operation, but generate a
									new vector each time something new is encrypted with the same key (see note above!)
						*/
						InitVector();
						
						/**
							\brief	Constructs an InitVector object from another (copy constructor)
							\author	GW
							\date	10/2013
							
							\param	iv_		InitVector object to copy values from
						*/
						InitVector(const InitVector& iv_);
						
						/**
							\brief	Constructs an InitVector object from a previous serialization of the initialization
									vector (using the serialize() method)
							\author	GW
							\date	10/2013
							
							\param	serializedIV_	Base-64 encoded serialization of an AES initialization vector
						*/
						InitVector(const std::string& serializedIV_);
						
						/**
							\brief	Destructs an InitVector object
							\author	GW
							\date	10/2013
						*/
						virtual ~InitVector();
						
						/**
							\brief		Assigns one InitVector object to another
							\author		GW
							\date		10/2013
							
							\param		rhs_	Righthand side of the assignment operator; InitVector object to copy
												values from
							\returns	A reference to the target object (lefthand side of assignment operator), but
										with its values updated
						*/
						InitVector& operator=(const InitVector& rhs_);
						
						/**
							\brief		Gets the raw bytes of the AES initialization vector
							\author		GW
							\date		10/2013
							
							\returns	A container of all the raw byte values of the AES init vector
						*/
						ByteVector bytes() const;
						
						/**
							\brief		Serializes the initialization vector in a base-64 string; useful for
										transmitting the string to another party or writing the IV to disk
							\author		GW
							\date		10/2013
							
							\returns	The serialized version of the initialization vector
							
							\note		An InitVector serialization string can be converted back into an IV object at
										runtime with the InitVector(const std::string&) constructor.
						*/
						std::string serialize() const;
						
						/**
							\brief		Compares two InitVector objects for equality
							\author		GW
							\date		10/2013
							
							\param		rhs_	Righthand side of the equality operator; InitVector object to compare
							\returns	true if the two objects have the same byte values, false otherwise
						*/
						bool operator==(const InitVector& rhs_) const;
						
						/**
							\brief		Compares two InitVector objects for inequality
							\author		GW
							\date		10/2013
							
							\param		rhs_	Righthand side of the inequality operator; InitVector object to compare
							\returns	true if the two objects have differing byte values, false otherwise
						*/
						bool operator!=(const InitVector& rhs_) const;


					private:
						friend plat::AESInitializationVector* __accessImpl(const InitVector&);

						friend bool encrypt(const Key&, const InitVector&, const std::string&, std::string&);
						friend bool decrypt(const Key&, const InitVector&, const std::string&, std::string&);

						/**
							\brief	Private implementation object
						*/
						plat::AESInitializationVector* __impl;
				};
		
				typedef std::unique_ptr<InitVector> InitVectorUP;
				typedef std::shared_ptr<InitVector> InitVectorSP;
				typedef std::weak_ptr<InitVector> InitVectorWP;

				extern plat::AESKey* __accessImpl(const Key&);
				extern plat::AESInitializationVector* __accessImpl(const InitVector&);
			}
			
			namespace rsa
			{
				class PublicKey;
				class PrivateKey;
			}

			/**
				\brief	Organizes all RSA functionality into one namespace for convenience
			*/
			namespace rsa
			{
				/**
					\brief		Generates a new RSA key pair of a given size/length
					\author		GW
					\date		10/2013
					
					\param		size_		Size/length of the key in bits (must be n*1024; not recommended for n>16)
					\param		privKey__	Return-by-reference for the new private key
					\param		pubKey__	Return-by-reference for the new public key
					\returns	true if the key pair could be generated, false otherwise
					
					\note		A minimum key size of 2048 bits is recommended.
				*/
				bool generateKeyPair(const int32_t size_, PrivateKey& privKey__, PublicKey& pubKey__);

				/**
					\brief		Encrypts a string (or raw bytes stuffed into a string) with a public RSA key
					\author		GW
					\date		10/2013
					
					\param		key_			Public RSA key to use for encryption
					\param		data_			Plain data to be encrypted (in the form of a string or binary data put
												inside of a string object)
					\param		b64Cipher__		Return-by-reference for the base-64 encoded cipher text
					\returns	true if the data could be encrypted, false otherwise
					
					\note		To decrypt the cipher, use the decryptWithPrivateKey() function (or equivalent - this
								lib is built on OpenSSL and the RSA standard, so any base-64 decoder and function that
								understands RSA private key decryption can decrypt the cipher with the correct key).
				*/
				bool encryptWithPublicKey(const PublicKey& key_, const std::string& data_, std::string& b64Cipher__);

				/**
					\brief		Decrypts a base-64 encoded string of cipher text (previously encrypted with the
								encryptWithPublicKey() function or equivalent)
					\author		GW
					\date		10/2013
					
					\param		key_		Private RSA key to use for decryption
					\param		b64Cipher_	Base-64 encoded cipher text to attempt to decrypt
					\param		data__		Return-by-reference for the decrypted plain text (or raw bytes stuffed into
											a string)
					\returns	true if the data could be decrypted, false otherwise
				*/
				bool decryptWithPrivateKey(const PrivateKey& key_, const std::string& b64Cipher_, std::string& data__);

				/**
					\brief		Encrypts a string (or raw bytes stuffed into a string) with a private RSA key
					\author		GW
					\date		10/2013
					
					\param		key_			Private RSA key to use for encryption
					\param		data_			Plain data to be encrypted (in the form of a string or binary data put
												inside of a string object)
					\param		b64Cipher__		Return-by-reference for the base-64 encoded cipher text
					\returns	true if the data could be encrypted, false otherwise
					
					\note		To decrypt the cipher, use the decryptWithPublicKey() function (or equivalent - this lib
								is built on OpenSSL and the RSA standard, so any base-64 decoder and function that
								understands RSA private key decryption can decrypt the cipher with the correct key).
					\note		Typically, sensitive data is not encrypted with a private key because anyone with the
								public key can decrypt it. However, "signing" (encrypting) some token data with a
								private key is a common practice to prove to a holder of the public key that the true
								owner of the private key e.g. received a message and is responding to it (as only the
								holder of the private key could do this).
				*/
				bool encryptWithPrivateKey(const PrivateKey& key_, const std::string& data_, std::string& b64Cipher__);

				/**
					\brief		Decrypts a base-64 encoded string of cipher text (previously encrypted with the
								encryptWithPrivateKey() function or equivalent)
					\author		GW
					\date		10/2013
					
					\param		key_		Public RSA key to use for decryption
					\param		b64Cipher_	Base-64 encoded cipher text to attempt to decrypt
					\param		data__		Return-by-reference for the decrypted plain text (or raw bytes stuffed into
								a string)
					\returns	true if the data could be decrypted, false otherwise
				*/
				bool decryptWithPublicKey(const PublicKey& key_, const std::string& b64Cipher_, std::string& data__);
			
				/**
					\brief	Wraps around a public RSA key's raw data for some added conveniences
				*/
				class PublicKey
				{
					public:
						/**
							\brief	Constructs an empty PublicKey object
							\author	GW
							\date	10/2013
							
							\note	To test if a PublicKey object is empty/valid, call valid() on it
							\note	Empty PublicKey objects do not relate to any actual public RSA key; they are
									designed to be "receivers" of some other PublicKey object via the overloaded
									assignment operator. Empty PublicKey objects are useless until such time (though it
									is safe to call any of their methods).
						*/
						PublicKey();
						
						/**
							\brief	Constructs a PublicKey object from another (copy constructor)
							\author	GW
							\date	10/2013
							
							\param	key_	PublicKey object to copy values from
						*/
						PublicKey(const PublicKey& key_);
						
						/**
							\brief	Constructs a PublicKey object from a previous serialization of the key (using the
									serialize() method)
							\author	GW
							\date	10/2013
							
							\param	serializedKey_	Base-64 encoded serialization of a public RSA key
						*/
						PublicKey(const std::string& serializedKey_);

						/**
							\brief	Destructs a PublicKey object
							\author	GW
							\date	10/2013
						*/
						virtual ~PublicKey();
						
						/**
							\brief		Assigns one PublicKey object to another
							\author		GW
							\date		10/2013
							
							\param		rhs_	Righthand side of the assignment operator; public key object to copy
												values from
							\returns	A reference to the target object (lefthand side of assignment operator), but
										with its values updated
						*/
						PublicKey& operator=(const PublicKey& rhs_);
						
						/**
							\brief		Gets whether the object contains a valid key
							\author		GW
							\date		10/2013
							
							\returns	true if the key is valid/not empty, false otherwise
						*/
						bool valid() const;

						/**
							\brief		Gets the length of the key in bits
							\author		GW
							\date		10/2013
							
							\returns	The number of bits for the public key
						*/
						int32_t length() const;

						/**
							\brief		Serializes the public key in a base-64 string; useful for transmitting the
										string to another party or writing the key to disk
							\author		GW
							\date		10/2013
							
							\returns	The serialized version of the key
							
							\note		A key serialization string can be converted back into a PublicKey object at
										runtime with the PublicKey(const std::string&) constructor.
						*/
						std::string serialize() const;
						
						/**
							\brief		Compares two PublicKey objects for equality
							\author		GW
							\date		10/2013
							
							\param		rhs_	Righthand side of the equality operator; PublicKey object to compare
							\returns	true if the two objects have the same key values, false otherwise
						*/
						bool operator==(const PublicKey& rhs_) const;
						
						/**
							\brief		Compares two PublicKey objects for inequality
							\author		GW
							\date		10/2013
							
							\param		rhs_	Righthand side of the inequality operator; PublicKey object to compare
							\returns	true if the two objects have differing key values, false otherwise
						*/
						bool operator!=(const PublicKey& rhs_) const;


					private:
						friend bool generateKeyPair(const int32_t, PrivateKey&, PublicKey&);
						friend bool encryptWithPublicKey(const PublicKey&, const std::string&, std::string&);
						friend bool decryptWithPrivateKey(const PrivateKey&, const std::string&, std::string&);
						friend bool encryptWithPrivateKey(const PrivateKey&, const std::string&, std::string&);
						friend bool decryptWithPublicKey(const PublicKey&, const std::string&, std::string&);
					
						/**
							\brief	Private implementation object
						*/
						plat::PublicRSAKey* __impl;
				};
		
				typedef std::unique_ptr<PublicKey> PublicKeyUP;
				typedef std::shared_ptr<PublicKey> PublicKeySP;
				typedef std::weak_ptr<PublicKey> PublicKeyWP;
				
				/**
					\brief	Wraps around a private RSA key's raw data for some added conveniences
				*/
				class PrivateKey
				{
					public:
						/**
							\brief	Constructs an empty PrivateKey object
							\author	GW
							\date	10/2013
							
							\note	To test if a PrivateKey object is empty/valid, call valid() on it
							\note	Empty PrivateKey objects do not relate to any actual private RSA key; they are
									designed to be "receivers" of some other PrivateKey object via the overloaded
									assignment operator. Empty PrivateKey objects are useless until such time (though it
									is safe to call any of their methods).
						*/
						PrivateKey();
						
						/**
							\brief	Constructs a PrivateKey object from another (copy constructor)
							\author	GW
							\date	10/2013
							
							\param	key_	PrivateKey object to copy values from
						*/
						PrivateKey(const PrivateKey& key_);
						
						/**
							\brief	Constructs a PrivateKey object from a previous serialization of the key (using the
									serialize() method)
							\author	GW
							\date	10/2013
							
							\param	serializedKey_	Base-64 encoded serialization of a private RSA key
						*/
						PrivateKey(const std::string& serializedKey_);

						/**
							\brief	Destructs a PrivateKey object
							\author	GW
							\date	10/2013
						*/
						virtual ~PrivateKey();
						
						/**
							\brief		Assigns one PrivateKey object to another
							\author		GW
							\date		10/2013
							
							\param		rhs_	Righthand side of the assignment operator; private key object to copy
												values from
							\returns	A reference to the target object (lefthand side of assignment operator), but
										with its values updated
						*/
						PrivateKey& operator=(const PrivateKey& rhs_);
						
						/**
							\brief		Gets whether the object contains a valid key
							\author		GW
							\date		10/2013
							
							\returns	true if the key is valid/not empty, false otherwise
						*/
						bool valid() const;

						/**
							\brief		Gets the size/length of the key in bits
							\author		GW
							\date		10/2013
							
							\returns	The number of bits for the private key
						*/
						int32_t length() const;

						/**
							\brief		Serializes the private key in a base-64 string; useful for transmitting the
										string to another party (not recommended for private keys!) or writing the key
										to disk
							\author		GW
							\date		10/2013
							
							\returns	The serialized version of the key
							
							\note		A key serialization string can be converted back into a PrivateKey object at
										runtime with the PrivateKey(const std::string&) constructor.
						*/
						std::string serialize() const;
						
						/**
							\brief		Compares two PrivateKey objects for equality
							\author		GW
							\date		10/2013
							
							\param		rhs_	Righthand side of the equality operator; PrivateKey object to compare
							\returns	true if the two objects have the same key values, false otherwise
						*/
						bool operator==(const PrivateKey& rhs_) const;
						
						/**
							\brief		Compares two PrivateKey objects for inequality
							\author		GW
							\date		10/2013
							
							\param		rhs_	Righthand side of the inequality operator; PrivateKey object to compare
							\returns	true if the two objects have differing key values, false otherwise
						*/
						bool operator!=(const PrivateKey& rhs_) const;


					private:
						friend bool generateKeyPair(const int32_t, PrivateKey&, PublicKey&);
						friend bool encryptWithPublicKey(const PublicKey&, const std::string&, std::string&);
						friend bool decryptWithPrivateKey(const PrivateKey&, const std::string&, std::string&);
						friend bool encryptWithPrivateKey(const PrivateKey&, const std::string&, std::string&);
						friend bool decryptWithPublicKey(const PublicKey&, const std::string&, std::string&);

						/**
							\brief	Private implementation object
						*/
						plat::PrivateRSAKey* __impl;
				};
		
				typedef std::unique_ptr<PrivateKey> PrivateKeyUP;
				typedef std::shared_ptr<PrivateKey> PrivateKeySP;
				typedef std::weak_ptr<PrivateKey> PrivateKeyWP;
			}
		}
	}
}

#endif
// __HDI_CORE_CRYPT__
