/**
	\file
	\brief		Header file for general Illustrator custom art manipulation
	\copyright	Hot Door, Inc. 2010-2025
*/

#ifndef __HDI_CORE_CUSTOM_ART__
#define __HDI_CORE_CUSTOM_ART__

#if defined(HDI_CORE_AIP_MODE)

#include "hdicoreTypes.h"

namespace hdi
{
	namespace core
	{
		class Art;
	
		/**
			\brief	Allows for custom art "group", user-editable art, and user-viewable "result" art manipulation
		*/
		class CustomArt
		{
			public:
				/**
					\brief	Destructs a CustomArt object
					\author	GW
					\date	07/2017
				*/
				~CustomArt();
			
				/**
					\brief		Gets the art underlying the CustomArt that the user can edit in Illustrator
					\author		GW
					\date		07/2017
					
					\returns	A pointer to the user-editable art, or NULL for error
					
					\note		The editable art does not show visibly in the Illustrator document except for when the
								user has selected the CustomArt or has moused-over the CustomArt (then its path and
								segment annotations will show).
				*/
				std::unique_ptr<Art> editableArt() const;
			
				/**
					\brief		Gets the art underlying the CustomArt that the user can see in Illustrator
					\author		GW
					\date		07/2017
					
					\returns	A pointer to the user-viewable art, or NULL for error
					
					\note		The viewable art need not be the same as the editable art, and it cannot be selected/
								modified in the Illustrator document by the user.
				*/
				std::unique_ptr<Art> viewableArt() const;
			
				/**
					\brief	Marks the CustomArt as dirty, so that Illustrator will fire the
							CustomArtUpdateViewableArtMessageType shortly
					\author	GW
					\date	07/2017
					
					\note	The correct time to update your user-viewable art is in response to
							CustomArtUpdateViewableArtMessageType.
				*/
				void markDirty();
			
				/**
					\brief	Marks the CustomArt as clean, so that Illustrator will not fire the
							CustomArtUpdateViewableArtMessageType even if the user-editable art has been changed
					\author	GW
					\date	07/2017
				*/
				void markClean();
			
				/**
					\brief	Even if the user-editable art will be changed (or has been changed), this will prevent the
							CustomArtUpdateViewableArtMessageType message from firing until the next time Illustrator
							messages your plugin (and you again change the user-editable art)
					\author	GW
					\date	07/2017
				*/
				void silenceNextUpdate();
			
				/**
					\brief		Gets whether your CustomArt is allowed to be a clipping mask
					\author		GW
					\date		07/2017
					
					\returns	true if clipping via your CustomArt is allowed
				*/
				bool canBeClipMask() const;
			
				/**
					\brief	Sets whether your CustomArt is allowed to be a clipping mask
					\author	GW
					\date	07/2017
				
					\param	canBe_	true if clipping via your CustomArt is allowed
				*/
				void setCanBeClipMask(const bool canBe_);


			private:
				// Only Art can construct a new CustomArt object
				friend class Art;

				/**
					\brief	Art to which this custom art belongs
				*/
				void* __data;
				
				/**
					\brief	Empty CustomArt objects cannot be constructed
					\author	GW
					\date	07/2017
				*/
				CustomArt();

				/**
					\brief	Constructs a new CustomArt object from a given Art object
					\author	GW
					\date	07/2017

					\param	art_	Pointer for the existing art we're wrapping around
				*/
				CustomArt(void* const art_);
			
				/**
					\brief	CustomArt objects cannot be copied
					\author	GW
					\date	07/2017
				*/
				CustomArt(const CustomArt&);
			
				/**
					\brief	CustomArt objects cannot be copied
					\author	GW
					\date	07/2017
				*/
				CustomArt& operator=(const CustomArt&);
		};
	}
}

#endif
// HDI_CORE_AIP_MODE

#endif
// __HDI_CORE_CUSTOM_ART__
