/**
	\file
	\brief		Header file for handling a day of the year and time of day
	\copyright	Hot Door, Inc. 2010-2025
*/

#ifndef __HDI_CORE_DATETIME__
#define __HDI_CORE_DATETIME__

#include "hdicoreDate.h"
#include "hdicoreTime.h"

namespace hdi
{
	namespace plat
	{
		class DateTime;
	}
	
	namespace core
	{
		/**
			\brief	Allows for acquisition and manipulation of a date (day of year) and time (of day)
		*/
		class DateTime
		{
			public:
				/**
					\brief	Constructs a DateTime object with a default value (1970-01-01 00:00:00)
					\author	GW
					\date	11/2014
				*/
				DateTime();
				
				/**
					\brief	Constructs a DateTime object from another existing DateTime object
					\author	GW
					\date	11/2014
					
					\param	dt_		Existing object to copy values from
				*/
				DateTime(const DateTime& dt_);
				
				/**
					\brief	Constructs a DateTime object from existing Date and Time objects
					\author	GW
					\date	11/2014
					
					\param	d_	Existing Date object
					\param	t_	Existing Time object
				*/
				DateTime(const Date& d_, const Time& t_);
				
				/**
					\brief	Constructs a DateTime object from a POSIX timestamp
					\author	GW
					\date	11/2014
					
					\param	ts_		Timestamp to use
				*/
				explicit DateTime(const int32_t ts_);
				
				/**
					\brief	Constructs a DateTime object from a POSIX timestamp
					\author	GW
					\date	11/2014
					
					\param	ts_		Timestamp to use
				*/
				explicit DateTime(const int64_t ts_);
				
				/**
					\brief	Constructs a DateTime object from a POSIX timestamp
					\author	GW
					\date	11/2014
					
					\param	ts_		Timestamp to use
				*/
				explicit DateTime(const double ts_);
				
				/**
					\brief	Constructs a DateTime object from a string representation of a datetime
					\author	GW
					\date	11/2014
					
					\param	dt_	String representation of a datetime
					
					\note	The expected format is e.g. "2014-11-25 14:34:56.789". A variety of delimiters can be used
							for the date (i.e. '-', '.', '/', and ',') and time (i.e. ':', '-', '.', and ',').
							Strings that do not follow this format have the same result as using the DateTime() ctor.
				*/
				DateTime(const std::string& dt_);
				
				/**
					\brief	Constructs a DateTime object from a given year, month, day, hour, minute, second, and
							microsecond
					\author	GW
					\date	11/2014
					
					\param	year_	Year, in the range [1970,9999]
					\param	month_	Month of the given year, in the range [1,12]
					\param	day_	Day of the month, in the range [1,31]
					\param	hour_	Hour of the day, in the range [0,23]
					\param	min_	Minute of the given hour, in the range [0,59]
					\param	sec_	Second of the given minute, in the range [0,59]
					\param	us_		Microsecond of the given second, in the range [0,999999]
					
					\note	Values below or above the expected range will be forced to the minimum or maximum value,
							respectively.
				*/
				DateTime(
					const int16_t year_,
					const int16_t month_,
					const int16_t day_,
					const int16_t hour_,
					const int16_t min_,
					const int16_t sec_ = 0,
					const int32_t us_ = 0
				);
				
				/**
					\brief		Constructs a DateTime object from the current system time
					\author		GW
					\date		11/2014
					
					\param		us_		Whether to use microsecond resolution
					\returns	A DateTime object for the current UTC time
				*/
				static DateTime Current(const bool us_ = false);
				
				/**
					\brief	Destructs a DateTime object
					\author	GW
					\date	11/2014
				*/
				virtual ~DateTime();
				
				/**
					\brief		Overloaded assignment operator to copy values from one DateTime object to another
					\author		GW
					\date		11/2014
					
					\param		rhs_	Righthand side of the operator, DateTime object to copy values from
					\returns	The target DateTime object, but with values updated to match that of rhs_
				*/
				virtual DateTime& operator=(const DateTime& rhs_);
				
				/**
					\brief		Overloaded equality operator to determine if two DateTime objects are equal
					\author		GW
					\date		11/2014
					
					\param		rhs_	Righthand side of the operator, DateTime object to compare
					\returns	true if the two objects represent the same date and time
				*/
				virtual bool operator==(const DateTime& rhs_) const;
				
				/**
					\brief		Overloaded inequality operator to determine if two DateTime objects differ
					\author		GW
					\date		11/2014
					
					\param		rhs_	Righthand side of the operator,
					\returns	true if the two objects represent different days or times
				*/
				virtual bool operator!=(const DateTime& rhs_) const;
				
				/**
					\brief		Overloaded less-than operator to determine if one DateTime object is earlier in time than
								the other
					\author		GW
					\date		02/2015
					
					\param		rhs_	Righthand side of the operator; DateTime object that might be later in time than
										the target (i.e. lefthand side)
					\returns	true if the target (i.e. lefthand side) represents a date earlier than rhs_
				*/
				virtual bool operator<(const DateTime& rhs_) const;
				
				/**
					\brief		Overloaded less-than-or-equal operator to determine if one DateTime object is earlier in
								time than, or equal to, the other
					\author		GW
					\date		02/2015
					
					\param		rhs_	Righthand side of the operator; DateTime object that might be later in time than,
										or equal to, the target (i.e. lefthand side)
					\returns	true if the target (i.e. lefthand side) represents a date earlier than, or equal to, rhs_
				*/
				virtual bool operator<=(const DateTime& rhs_) const;
				
				/**
					\brief		Overloaded greater-than operator to determine if one DateTime object is later in time
								than the other
					\author		GW
					\date		02/2015
					
					\param		rhs_	Righthand side of the operator; DateTime object that might be earlier in time
										than the target (i.e. lefthand side)
					\returns	true if the target (i.e. lefthand side) represents a date later than rhs_
				*/
				virtual bool operator>(const DateTime& rhs_) const;
				
				/**
					\brief		Overloaded greater-than-or-equal operator to determine if one DateTime object is later
								in time than, or equal to, the other
					\author		GW
					\date		02/2015
					
					\param		rhs_	Righthand side of the operator; DateTime object that might be earlier in time
										than, or equal to, the target (i.e. lefthand side)
					\returns	true if the target (i.e. lefthand side) represents a date later than, or equal to, rhs_
				*/
				virtual bool operator>=(const DateTime& rhs_) const;
				
				/**
					\brief		Overloaded subtraction operator to determine the duration between two DateTime objects
					\author		GW
					\date		11/2014
					
					\param		rhs_	Righthand side of the operator, DateTime object to "substract" from the lefthand
										side
					\returns	The duration between the two days and times
				*/
				virtual Duration operator-(const DateTime& rhs_) const;
				
				/**
					\brief		Overloaded substraction operator to determine the resultant date and time before the
								target
					\author		GW
					\date		11/2014
					
					\param		rhs_	Righthand side of the operator, Duration object to substract from the lefthand
										side
					\returns	The date and time that would result from subtracting the duration from the target
				*/
				virtual DateTime operator-(const Duration& rhs_) const;
				
				/**
					\brief		Overloaded subtraction-assignment operator to determine and assign the resultant date
								and time before the initial target
					\author		GW
					\date		11/2014
					
					\param		rhs_	Righthand side of the operator, Duration object to substract from the lefthand
										side
					\returns	The target DateTime object, but with its values updated to reflect the substraction of
								the given duration
				*/
				virtual DateTime& operator-=(const Duration& rhs_);
				
				/**
					\brief		Overloaded addition operator to determine the resultant date and time after the target
					\author		GW
					\date		11/2014
					
					\param		rhs_	Righthand side of the operator, Duration object to add to the lefthand side
					\returns	The date and time that would result from adding the duration to the target
				*/
				virtual DateTime operator+(const Duration& rhs_) const;
				
				/**
					\brief		Overloaded addition-assignment operator to determine and assign the resultant date and
								time after the initial target
					\author		GW
					\date		11/2014
					
					\param		rhs_	Righthand side of the operator, Duration object to add to the lefthand side
					\returns	The target DateTime object, but with its values updated to reflect the addition of the
								given duration
				*/
				virtual DateTime& operator+=(const Duration& rhs_);
				
				/**
					\brief		Gets the date for the DateTime object
					\author		GW
					\date		11/2014
					
					\returns	The date of the target DateTime
					
					\note		Modifying the returned Date object will not affect the target DateTime object. If you
								make changes to the Date object, the target DateTime object must be updated with the
								setDate() method.
				*/
				Date date() const;
				
				/**
					\brief	Sets the date for the DateTime object
					\author	GW
					\date	11/2014
					
					\param	d_	New date for the target DateTime
				*/
				void setDate(const Date& d_);
				
				/**
					\brief		Gets the time for the DateTime object
					\author		GW
					\date		11/2014
					
					\returns	The time of the target DateTime
					
					\note		Modifying the returned Time object will not affect the target DateTime object. If you
								make changes to the Time object, the target DateTime object must be updated with the
								setTime() method.
				*/
				Time time() const;
				
				/**
					\brief	Sets the time for the DateTime object
					\author	GW
					\date	11/2014
					
					\param	t_	New time for the target DateTime
				*/
				void setTime(const Time& t_);
				
				/**
					\brief		Formats the target DateTime object into a string representation
					\author		GW
					\date		11/2014
					
					\param		format_		Input string representing the format for the output string
					\returns	An output string for the target object's date and time, formatted as indicated
					
					\note		For a description of the supported formatting characters, see the notes for the
								Date::format() and Time::format() methods.
				*/
				virtual std::string format(const std::string& format_ = "Y-m-d H:i:s") const;


			private:
				/**
					\brief	Private implementation object
				*/
				plat::DateTime* __impl;
		};
		
		typedef std::unique_ptr<DateTime> DateTimeUP;
		typedef std::shared_ptr<DateTime> DateTimeSP;
		typedef std::weak_ptr<DateTime> DateTimeWP;
	}
}

/**
	\brief		Adds a duration to a date and time
	\author		GW
	\date		11/2014
	
	\param		lhs_	Duration to use to offset the date and time
	\param		rhs_	DateTime to be offset by a duration
	\returns	A new DateTime object for the result of the offset
*/
hdi::core::DateTime operator+(const hdi::core::Duration& lhs_, const hdi::core::DateTime& rhs_);

#endif
// __HDI_CORE_DATETIME__
