/**
	\file
	\brief		Header file for a class to handle dispatching plugin messages
	\copyright	Hot Door, Inc. 2010-2025
*/

#ifndef __HDI_CORE_DISPATCHER__
#define __HDI_CORE_DISPATCHER__

#include <vector>

#include "hdicoreTypes.h"

namespace hdi
{
	namespace core
	{
		class Callback;
		class Message;
		class PrefData;
		class Timer;
		
		#if defined(HDI_CORE_AIP_MODE)
			class Annotator;
			class Notifier;
			class ToolMessage;
		#endif

		/**
			\brief	Class through which all messages ultimately pass; holds registered callbacks for messages, notifiers,
					timers, etc. and calls them at the proper time
		*/
		class Dispatcher
		{
			public:
				#if defined(HDI_CORE_AIP_MODE)
					typedef std::vector< std::shared_ptr<Notifier> > NotifierVector;
				#endif
			
				/**
					\brief	Describes a variety of dispatch-related errors
				*/
				enum Error
				{
					NoDispatchError					= 0,
					UnhandledMessageError			= 10,	// Not a "real error", just that a received message was not handled
					UnknownMessageTypeError			= 20,
					RedundantMessageReceivedError	= 30
				};
				
				/**
					\brief	Describes which keys can be monitored for presses at anytime
				*/
				enum VirtualKey
				{
					// ==== Locale independent keys ====
					CommandKey		= 10,			// Only useful on Mac
					OptionKey		= 20,
					AltKey			= OptionKey,
					ControlKey		= 30,
					ShiftKey		= 40,
					CapsLockKey		= 50,
					
					SpaceKey		= 110,
					TabKey			= 100,
					ReturnKey		= 120,			// Same as enter key; no way to differentiate on all platforms
					
					EscapeKey		= 200,
					ClearKey		= 210,
					DeleteKey		= 220,
					ForwardDeleteKey= 230,
					
					UpArrowKey		= 300,
					DownArrowKey	= 310,
					LeftArrowKey	= 320,
					RightArrowKey	= 330,
					
					HomeKey			= 400,
					EndKey			= 410,
					PageUpKey		= 420,
					PageDownKey		= 430,
					HelpKey			= 440,
					
					F1Key			= 500,
					F2Key			= 510,
					F3Key			= 520,
					F4Key			= 530,
					F5Key			= 540,
					F6Key			= 550,
					F7Key			= 560,
					F8Key			= 570,
					F9Key			= 580,
					F10Key			= 590,
					F11Key			= 600,
					F12Key			= 610,
					F13Key			= 620,
					F14Key			= 630,
					F15Key			= 640,
					F16Key			= 650,
					F17Key			= 660,
					F18Key			= 670,
					F19Key			= 680,
					F20Key			= 690,
					
					NumPad0Key		= 800,
					NumPad1Key		= 810,
					NumPad2Key		= 820,
					NumPad3Key		= 830,
					NumPad4Key		= 840,
					NumPad5Key		= 850,
					NumPad6Key		= 860,
					NumPad7Key		= 870,
					NumPad8Key		= 880,
					NumPad9Key		= 890,
					
					NumPadDecimalKey	= 1000,
					NumPadPlusKey		= 1010,
					NumPadMinusKey		= 1020,
					NumPadMultiplyKey	= 1030,
					NumPadDivideKey		= 1040,
					// ==== End locale independent keys ====
					
					// ==== Locale dependent keys ====
					LD_0Key				= 2010,
					LD_1Key				= 2020,
					LD_2Key				= 2030,
					LD_3Key				= 2040,
					LD_4Key				= 2050,
					LD_5Key				= 2060,
					LD_6Key				= 2070,
					LD_7Key				= 2080,
					LD_8Key				= 2090,
					LD_9Key				= 2100,

					LD_AKey				= 2110,
					LD_BKey				= 2120,
					LD_CKey				= 2130,
					LD_DKey				= 2140,
					LD_EKey				= 2150,
					LD_FKey				= 2160,
					LD_GKey				= 2170,
					LD_HKey				= 2180,
					LD_IKey				= 2190,
					LD_JKey				= 2200,
					LD_KKey				= 2210,
					LD_LKey				= 2220,
					LD_MKey				= 2230,
					LD_NKey				= 2240,
					LD_OKey				= 2250,
					LD_PKey				= 2260,
					LD_QKey				= 2270,
					LD_RKey				= 2280,
					LD_SKey				= 2290,
					LD_TKey				= 2300,
					LD_UKey				= 2310,
					LD_VKey				= 2320,
					LD_WKey				= 2330,
					LD_XKey				= 2340,
					LD_YKey				= 2350,
					LD_ZKey				= 2360,

					LD_SlashKey			= 2370,	// "/" on US keyboard
					LD_BackslashKey		= 2380,	// "\" on US keyboard

					LD_CommaKey			= 2390,	// "," on US keyboard
					LD_PeriodKey		= 2400,	// "." on US keyboard
					LD_SemicolonKey		= 2410,	// ";" on US keyboard

					LD_MinusKey			= 2420,	// "-" on US keyboard
					LD_EqualKey			= 2430,	// "=" on US keyboard

					LD_LeftBracketKey	= 2440,	// "[" on US keyboard
					LD_RightBracketKey	= 2450,	// "]" on US keyboard

					LD_GraveKey			= 2460,	// "`" on US keyboard
					LD_QuoteKey			= 2470	// "'" on US keyboard
					// ==== End locale dependent keys ====
				};

				/**
					\brief		Registers a new callback for a given message type
					\author		GW
					\date		09/2013

					\param		type_		Type of message to which callback_ will be associated (only one callback per
											message is allowed)
					\param		callback_	Callback to perform when the caller/selector pair associated with type_ is
											received in a message
					\returns	true if the callback was registered successfully

					\note		Many of the available message types are already automatically handled by the hdi::core
								lib. See the MessageType enum docs for more information.
				*/
				bool registerMessageCallback(const MessageType type_, const Callback& callback_);

				/**
					\brief	Unregisters a callback for a given message type
					\author	GW
					\date	09/2013

					\param	type_	Type of message from which the caller is unsubscribing
				*/
				void unregisterMessageCallback(const MessageType type_);

				/**
					\brief		Registers a new callback for a given caller/selector pair
					\author		GW
					\date		04/2014

					\param		caller_		Caller ID string
					\param		selector_	Selector ID string for caller_
					\param		callback_	Callback to perform when the caller/selector pair is received in a message
					\returns	true if the caller was registered successfully
					
					\note		Only one callback per caller/selector pair is allowed.
					\note		If the caller/selector pair is already known by the hdi::core lib, then this function
								will bail. Always try to use the version of this method that takes a MessageType
								argument instead; if the message you're interested in is not built into the app (or
								you know a special third-party plugin will send the message to you) then utilize this
								method.
					\note		Any message associated with the caller/selector pair will always be processed as a
								CustomMessage object, so be aware when calling the lastMessage() method from within your
								callback.
				*/
				bool registerMessageCallback(
					const std::string& caller_,
					const std::string& selector_,
					const Callback& callback_
				);

				/**
					\brief	Unregisters a callback for a given caller/selector pair
					\author	GW
					\date	04/2014

					\param	caller_		Caller ID string from which the caller is unsubscribing
					\param	selector_	Selector ID string for caller_
					
					\note	Only use this method if you originally registered the message callback with the
							registerMessageCallback() method variant that takes a caller/selector pair.
				*/
				void unregisterMessageCallback(const std::string& caller_, const std::string& selector_);
			
				#if defined(HDI_CORE_PSP_MODE)
					/**
						\brief	Registers a new callback for a given caller/selector numeric pair
						\author	GW
						\date	12/2017

						\param	caller_		Photoshop-compatible caller enum value
						\param	selector_	Photoshop-compatible selector int value
						\param	callback_	Callback to perform when the caller/selector pair is received in a message

						\note	Only one callback per caller/selector pair is allowed.
						\note	If the caller/selector pair is already known by the hdi::core lib, then this function
								will bail. Always try to use the version of this method that takes a MessageType
								argument instead; if the message you're interested in is not built into the app (or
								you know a special third-party plugin will send the message to you) then utilize this
								method.
						\note	Any message associated with the caller/selector pair will always be processed as a
								CustomMessage object, so be aware when calling the lastMessage() method from within your
								callback.
					*/
					bool registerMessageCallback(
						const CallerType caller_,
						const int16_t selector_,
						const Callback& callback_
					);

					/**
						\brief		Unregisters a callback for a given caller/selector numeric pair
						\author		GW
						\date		12/2017

						\param		caller_		Photoshop-compatible caller enum value
						\param		selector_	Photoshop-compatible selector int value
						\returns	A pointer to the Callback object that was unregistered
						
						\note	Only use this method if you originally registered the message callback with the
								registerMessageCallback() method variant that takes a caller/selector pair.
					*/
					void unregisterMessageCallback(const CallerType caller_, const int16_t selector_);
			
					/**
						\brief		Gets a previously-registered pre- hdi_core "play" event callback
						\author		GW
						\date		02/2018
						
						\returns	The callback previously registered via the registerCoreEventPreCallback() function
					*/
					Callback* const coreEventPreCallback() const;
			
					/**
						\brief	Registers a new callback to be called whenever your plugin is being messaged from another
								hdi_core plugin, via the Photoshop::playCoreEvent() method
						\author	GW
						\date	02/2018
						
						\param	callback_	Callback to perform BEFORE the normal message callback (i.e. the one registered
											with the registerMessageCallback() method)
					*/
					void registerCoreEventPreCallback(const Callback& callback_);
			
					/**
						\brief		Gets a previously-registered post- hdi_core "play" event callback
						\author		GW
						\date		02/2018
						
						\returns	The callback previously registered via the registerCoreEventPostCallback() function
					*/
					Callback* const coreEventPostCallback() const;
			
					/**
						\brief	Registers a new callback to be called whenever your plugin is being messaged from another
								hdi_core plugin, via the Photoshop::playCoreEvent() method
						\author	GW
						\date	02/2018
						
						\param	callback_	Callback to perform AFTER the normal message callback (i.e. the one registered
											with the registerMessageCallback() method)
					*/
					void registerCoreEventPostCallback(const Callback& callback_);
			
					/**
						\brief		Gets the message data from the other hdi_core plugin, which it sent via the
									Photoshop::playCoreEvent() method
						\author		GW
						\date		02/2018
						
						\returns	A pointer to the message data sent from the other plugin
						
						\note		The return value is only valid if this method is being called from a registered
									message or hdi_core event callback.
					*/
					PrefData* const coreEventMessage() const;
			
					/**
						\brief	Sets the response data for the other hdi_core plugin, which it will receive via the
								Photoshop::playCoreEvent() method
						\author	GW
						\date	02/2018
						
						\param	response_	Response data for the other plugin, if any
						
						\note	This method is only valid if it is being called from a registered message or hdi_core
								event callback.
					*/
					void setCoreEventResponse(const PrefData& response_);
				#endif
				// HDI_CORE_PSP_MODE
				
				/**
					\brief		Notifies the dispatcher of an incoming message, causing it to call the appropriate
								callback or return UnhandledMessageError in the case no callback is registered
					\author		GW
					\date		09/2013

					\param		type_		Type of message that is being received
					\param		msg_		Message data sent along with the message; this will automatically be
											processed by the dispatcher and made available by the lastMessage() method
					\returns	Error that occurred during event dispatching (essentially unused at this point)

					\note		You can acquire the most recent message by calling lastMessage()
					\note		It would be an extreme rarity for this method to be called directly.
					\note		For Photoshop this also handles known action plugin messages.
				*/
				Error message(const MessageType type_, void* const msg_);
				
				/**
					\brief		Notifies the dispatcher of an incoming (custom) message, causing it to call the
								appropriate callback or return UnhandledMessageError in the case no callback is
								registered
					\author		GW
					\date		04/2014

					\param		caller_		Caller ID string
					\param		selector_	Selector ID string for caller_
					\param		msg_		Message data sent along with the message; this will automatically be
											processed by the dispatcher and made available by the lastMessage() method
					\returns	Error that occurred during event dispatching (essentially unused at this point)

					\note		You can acquire the most recent message by calling lastMessage()
					\note		It would be an extreme rarity for this method to be called directly.
					\note		Essentially any caller/selector pair that is not known to the internal PluginMain()
								function is forwarded to this method (i.e. any caller/selector pair that is not built
								into the app), such that a developer can receive custom messages.
					\note		For Photoshop this also handles custom action plugin messages.
				*/
				Error message(const std::string& caller_, const std::string& selector_, void* const msg_);
			
				#if defined(HDI_CORE_PSP_MODE)
					/**
						\brief		Notifies the dispatcher of an incoming (custom) Photoshop-style message, causing it
									to call the appropriate callback or return UnhandledMessageError in the case no
									callback is registered
						\author		GW
						\date		12/2017

						\param		caller_		Photoshop-compatible caller enum value
						\param		selector_	Photoshop-compatible selector ID
						\param		msg_		Param block struct sent along with the selector

						\note		You can acquire the most recent message by calling lastMessage()
						\note		It would be an extreme rarity for this method to be called directly.
						\note		Essentially any Photoshop-style selector that is not known to the internal *PluginMain()
									function is forwarded to this method (i.e. any selector that is not built into the
									app), such that a developer can receive custom messages.
					*/
					Error message(const CallerType caller_, const int16_t selector_, void* const msg_);
			
					/**
						\brief		Gets the result code pointer for the current message, which Photoshop will process
									after you have finished processing said message
						\author		GW
						\date		01/2018
						
						\returns	A pointer to the result code, which you can manipulate if needed
						
						\note		Action plugins do not have a result code for their messages, so this method will
									return NULL in that case.
					*/
					int16_t* const resultCode();
				#endif
				// HDI_CORE_PSP_MODE

				/**
					\brief		Gets a Message object that wraps around the last message data received
					\author		GW
					\date		09/2013

					\returns	A Message object describing the last message sent by the app (or the current message if
								you are calling this from inside a callback that was executed by the dispatcher itself)

					\note		If you are calling this from inside a message callback, then it will be the current
								message relating to the aforementioned callback (e.g. the returned Message object can
								be safely modified in the case of app messages that have data output fields). Once your
								message callback has finished, however, not all data in a Message subclass will be
								available (which subclasses and which data varies greatly between each message type, so
								do as much processing as possible inside your message callback)!
					\note		The return value is a pointer to the internal Message object; it will be modified if you
								assign anything in it, which may affect subsequent messages. Be careful!
				*/
				Message* const lastMessage();

				/**
					\brief	Executes a given callback after a delay
					\author	GW
					\date	08/2013

					\param	callback_	Callback object to execute on delay
					\param	delay_		Number of seconds to wait before executing (approximately - the OS may wait longer)
				*/
				void scheduleCallback(const Callback& callback_, const double delay_ = 0.0);

				/**
					\brief	Registers a Timer object
					\author	GW
					\date	09/2013

					\param	timer_	New timer, whose callback will be called when the proper notification is received by
									the dispatcher
				*/
				void registerTimer(const Timer& timer_);

				/**
					\brief		Unregisters a given timer
					\author		GW
					\date		09/2013

					\param		timer_	Timer to be unregistered
					\returns	true if the timer was removed, false otherwise
				*/
				bool unregisterTimer(const Timer& timer_);

				#if defined(HDI_CORE_AIP_MODE)
					/**
						\brief		Gets the last tool message received, regardless of whether subsequent non-tool messages
									have been received
						\author		GW
						\date		09/2013

						\returns	The last tool-related message received by the dispatcher

						\note 		The return value is a pointer to the internal ToolMessage object; it will be modified if
									you assign anything in it, which may affect subsequent tools/panels/etc. Be careful!
						\note		Generally, you shouldn't need to call this method. Only use it if you know what you're
									doing. If a specific piece of functionality provided by the app is not handled by this
									class (or related classes), then it should probably be added to the library.
					*/
					ToolMessage* const lastToolMessage();

					/**
						\brief	Registers a Notifier object for a given Illustrator notifier subscription
						\author	GW
						\date	09/2013

						\param	notifier_	New notifier, whose callback will be called when the proper notification is
											received by the dispatcher

						\note	If multiple notifiers are registered for the same type of notification, their callbacks will
								be executed in the order of registration.
					*/
					void registerNotifier(const Notifier& notifier_);

					/**
						\brief		Gets registered notifiers by their type
						\author		GW
						\date		09/2013

						\param		type_		Type of the notifiers to find
						\returns	notifier__	All notifiers whose type matches the type_ argument
					*/
					NotifierVector registeredNotifiersOfType(const NotifierType type_);

					/**
						\brief		Unregisters a given notifier
						\author		GW
						\date		09/2013

						\param		notifier_	Notifier to be unregistered
						\returns	true if the notifier was removed, false otherwise
					*/
					bool unregisterNotifier(const Notifier& notifier_);
			
					/**
						\brief	Registers an Annotator object
						\author	GW
						\date	09/2013

						\param	annotator_	New annotator, whose callback will be called when the proper notification is
											received by the dispatcher
					*/
					void registerAnnotator(const Annotator& annotator_);

					/**
						\brief		Unregisters a given annotator
						\author		GW
						\date		09/2013

						\param		annotator_	Annotator to be unregistered
						\returns	true if the annotator was removed, false otherwise
					*/
					bool unregisterAnnotator(const Annotator& annotator_);
				#endif
				// HDI_CORE_AIP_MODE
				
				/**
					\brief		Creates an app-wide key-down monitor that executes a callback when a specific key has
								been pressed
					\author		GW
					\date		03/2014
					
					\param		key_	Key that should be monitored for pressing
					\param		cb_		Callback to execute when key_ has been pressed (may be called repeatedly if the
								user holds down the key)
					\returns	A unique ID for the monitor that was created, or 0 for error

					\note		This method is not required to check for modifier keys during tool use. See the
								hdi::core::Tool::shiftKeyDown(), controlKeyDown(), optionKeyDown(), and commandKeyDown()
								methods for more information.
					\note		Response time from the key being pressed and the callback being executed varies between
								each platform and version of the app. This is handled by the OS itself, so there's not
								much to be done if you find the response time to be too slow :(
				*/
				uint32_t addKeyDownMonitor(const VirtualKey key_, const Callback& cb_);
				
				/**
					\brief		Creates an app-wide key-up monitor that executes a callback when a specific key has
								been released
					\author		GW
					\date		10/2022
					
					\param		key_	Key that should be monitored for release
					\param		cb_		Callback to execute when key_ has been release
					\returns	A unique ID for the monitor that was created, or 0 for error

					\note		Response time from the key being released and the callback being executed varies between
								each platform and version of the app. This is handled by the OS itself, so there's not
								much to be done if you find the response time to be too slow :(
				*/
				uint32_t addKeyUpMonitor(const VirtualKey key_, const Callback& cb_);
				
				/**
					\brief		Gets an existing key-down monitor, if any
					\author		GW
					\date		03/2014
					
					\param		key_	Virtual key in question
					\param		id__	Return-by-reference for the monitor's ID number
					\param		cb__	Return-by-reference for the monitor's callback
					\returns	true if the monitor exists, false otherwise
				*/
				bool getKeyDownMonitor(
					const VirtualKey key_,
					uint32_t& id__,
					std::unique_ptr<Callback>& cb__
				) const;
				
				/**
					\brief		Gets an existing key-up monitor, if any
					\author		GW
					\date		10/2022
					
					\param		key_	Virtual key in question
					\param		id__	Return-by-reference for the monitor's ID number
					\param		cb__	Return-by-reference for the monitor's callback
					\returns	true if the monitor exists, false otherwise
				*/
				bool getKeyUpMonitor(
					const VirtualKey key_,
					uint32_t& id__,
					std::unique_ptr<Callback>& cb__
				) const;
				
				/**
					\brief	Removes a key-down monitor previously created with addKeyDownMonitor()
					\author	GW
					\date	03/2014
					
					\param	id_		ID for the keypress monitor (returned by earlier call to addKeyDownMonitor() method)
				*/
				void removeKeyDownMonitor(const uint32_t id_);

				/**
					\brief	Removes a key-up monitor previously created with addKeyUpMonitor()
					\author	GW
					\date	10/2022
					
					\param	id_		ID for the keypress monitor (returned by earlier call to addKeyUpMonitor() method)
				*/
				void removeKeyUpMonitor(const uint32_t id_);
				
				/**
					\brief		Gets whether the given key is down
					\author		GW
					\date		10/2022
					
					\param		key_	The key in question
					\param		down__	Return-by-reference for whether the key is down; true means down, false means up
					\returns	true if the key down state could be acquired, false otherwise
					
					\note		This method is not suitable to check for modifier keys during tool use. See the
								hdi::core::Tool::shiftKeyDown(), controlKeyDown(), optionKeyDown(), and commandKeyDown()
								methods for more information.
					\note		This method can only perform its job if a key monitor has been registered (either up or
								down) for the given key, and if the user has actually pressed the key since such time. If
								not, then it will return false.
				*/
				bool isKeyDown(const VirtualKey key_, bool& down__);
		


			private:
				// Only Plugin can construct a Dispatcher object
				friend class Plugin;

				/**
					\brief	Private implementation data
				*/
				void* __data;

				/**
					\brief	Private constructor for the Dispatcher class
					\author	GW
					\date	09/2013
				*/
				Dispatcher();
				
				/**
					\brief	Unused
					\author	GW
					\date	02/2015
				*/
				Dispatcher(const Dispatcher&);

				/**
					\brief	Destructs a Dispatcher object by deleting all registered message callbacks, notifiers, etc.
					\author	GW
					\date	09/2013
				*/
				~Dispatcher();
				
				/**
					\brief	Unused
					\author	GW
					\date	02/2015
				*/
				Dispatcher& operator=(const Dispatcher&);
		};
	}
}

#endif
// __HDI_CORE_DISPATCHER__
