/**
	\file
	\brief		Header file for plugin user interface divider line widgets
	\copyright	Hot Door, Inc. 2010-2025
*/

#ifndef __HDI_CORE_DIVIDER__
#define __HDI_CORE_DIVIDER__

#include "hdicoreWidget.h"

namespace hdi
{
	namespace core
	{
		/**
			\brief	Divider class for UI divider lines
		*/
		class Divider : public Widget
		{
			public:
				/**
					\brief	Describes the direction of the line in a Divider object
				*/
				enum Direction
				{
					UnknownDirection	= 0,
					Horizontal			= 10,
					Vertical			= 20
				};

				/**
					\brief	Constructs an empty Divider object
					\author	GW
					\date	09/2013
					
					\note	To test if a Divider object is empty, call isEmpty() on it
					\note	Empty Divider objects do not relate to any actual UI widget; they are designed to be
							"receivers" of some other Divider object via the overloaded assignment operator. Empty
							Divider objects are useless until such time (though it is safe to call any of their methods).
				*/
				Divider();

				/**
					\brief	Constructs a new Divider object from an existing Divider object (copy constructor)
					\author	GW
					\date	09/2013

					\param	div_	Existing Divider object
				*/
				Divider(const Divider& div_);

				/**
					\brief	Constructs a divider line widget
					\author	GW
					\date	09/2013

					\param	loc_		Top-left of the divider, in 1x resolution coordinates
					\param	length_		Length of the divider
					\param	dir_		Horizontal or vertical line?
				*/
				Divider(const Point& loc_, const double length_, const Direction dir_ = Horizontal);
				
				/**
					\brief	Divider destructor
					\author	GW
					\date	09/2013
				*/
				virtual ~Divider();

				/**
					\brief		Allows one Divider object to be assigned from another
					\author		GW
					\date		10/2013

					\param		rhs_	Righthand side of the = operator; the object to copy values from
					\returns	The target Divider object, but with its value updated to match that of rhs_
				*/
				virtual Divider& operator=(const Divider& rhs_);
				
				/**
					\brief		Convenience method to clone a Divider object on the heap
					\author		GW
					\date		10/2013

					\returns	A pointer to the new Divider object
					
					\note		If you subclass Divider, you MUST overload this method yourself! If you don't and/or
								your clone() method does not return an instance of your Divider subclass, you will
								experience "object slicing" when adding the widget to a window.
								
					\warning	The caller must manage the memory for the returned Divider object.
				*/
				virtual Divider* clone() const;
				
				/**
					\brief		Convenience method to duplicate a Divider object, creating a new and identical UI
								element to the target (but not belonging to the same containing window)
					\author		GW
					\date		11/2013
					
					\returns	A pointer to the new Divider object (and new UI element)
					
					\warning	The caller must manage the memory for the returned Divider object.
				*/
				virtual Divider* duplicate() const;

				/**
					\brief		Returns DividerType, as dividers are always dividers!
					\author		GW
					\date		09/2013
					
					\returns	DividerType
				*/
				virtual Type type() const;

				/**
					\brief		Gets the direction that the line is going
					\author		GW
					\date		09/2013

					\returns	Vertical for a vertical line, Horizontal otherwise
				*/
				virtual Direction direction() const;

				/**
					\brief		Always gets "", as dividers cannot have text
					\author		GW
					\date		09/2013
					
					\returns	""
				*/
				virtual std::string text() const;

				/**
					\brief	Prevents setting the text, as dividers cannot have text
					\author	GW
					\date	09/2013
				*/
				virtual void setText(const std::string&);

				/**
					\brief		Returns true, because dividers are always enabled
					\author		GW
					\date		09/2013

					\returns	true
				*/
				virtual bool enabled();

				/**
					\brief	Overrides Widget::setEnabled() because dividers are always enabled
					\author	GW
					\date	09/2013

					\param	enable_		true to enable, false to disable (ignored)
				*/
				virtual void setEnabled(const bool enable_);
		};
		
		typedef std::unique_ptr<Divider> DividerUP;
		typedef std::shared_ptr<Divider> DividerSP;
		typedef std::weak_ptr<Divider> DividerWP;
	}
}

#endif
// __HDI_CORE_DIVIDER__
