/**
	\file
	\brief		Header file for manipulating application documents
	\copyright	Hot Door, Inc. 2010-2025
*/

#ifndef __HDI_CORE_DOCUMENT__
#define __HDI_CORE_DOCUMENT__

#include "hdicorePoint.h"
#include "hdicoreSize.h"
#include "hdicoreTypes.h"

namespace hdi
{
	#if defined(HDI_CORE_AIP_MODE)
		namespace aip
		{
			class Document;
		}
	#elif defined(HDI_CORE_PSP_MODE)
		namespace psp
		{
			class Document;
		}
	#endif
	
	namespace core
	{
		class Layer;
	
		#if defined(HDI_CORE_AIP_MODE)
			class ArtStyle;
			class Symbol;
		#elif defined(HDI_CORE_PSP_MODE)
			class XMPMeta;
		#endif
	
		/**
			\brief	Allows one to interact with a document and its metadata (differs from CurrentDocument class)
		*/
		class Document
		{
			public:
				#if defined(HDI_CORE_AIP_MODE)
					/**
						\brief	Represents options when creating a new Illustrator document
					*/
					class Preset
					{
						public:
							/**
								\brief	Controls the preview mode of a new document
							*/
							enum PreviewMode
							{
								PreviewModeDefault		= 10,
								PreviewModePixel		= 20,
								PreviewModeOverprint	= 30
							};

							/**
								\brief	Controls the transparency grid color of a new document
							*/
							enum TransparencyGrid
							{
								TransparencyGridNone		= 0,
								TransparencyGridLightGray	= 10,
								TransparencyGridGray		= 20,
								TransparencyGridDarkGray	= 30,
								TransparencyGridRed			= 40,
								TransparencyGridOrange		= 50,
								TransparencyGridGreen		= 60,
								TransparencyGridBlue		= 70,
								TransparencyGridPurple		= 80
							};

							/**
								\brief	Controls the raster resolution of a new document
							*/
							enum RasterResolution
							{
								RasterResolutionScreen	= 10,	// 72 ppi
								RasterResolutionMedium	= 20,	// 150 ppi
								RasterResolutionHigh	= 30	// 300 ppi
							};

							/**
								\brief	Controls how artboards are placed when creating a new document.
								
								\note	Items that end in "LR" are placed from left-to-right; items ending in "RL" are
										placed from right-to-left
							*/
							enum ArtboardLayout
							{
								ArtboardLayoutColumn			= 10,
								ArtboardLayoutRowLR				= 20,
								ArtboardLayoutRowRL				= 30,
								ArtboardLayoutGridByRowLR		= 40,
								ArtboardLayoutGridByColumnLR	= 50,
								ArtboardLayoutGridByRowRL		= 60,
								ArtboardLayoutGridByColumnRL	= 70
							};

							/**
								\brief	Maximum number of artboards that Illustrator documents will support
							*/
							static const uint32_t maxArtboardCount = 1000;
						
							/**
								\brief	UTF-8 title of the document. Empty ("") will result in a title like "Untitled-1", etc.
							*/
							std::string title;
							
							/**
								\brief	Width and height of the document artboard
							*/
							Size size;
							
							/**
								\brief	Number of artboards in the document. Cannot exceed maxArtboardCount.
							*/
							uint32_t artboardCount;

							/**
								\brief	Controls how the artboards are placed
							*/
							ArtboardLayout artboardLayout;
							
							/**
								\brief	How many points each artboard should be spaced from its neighbor
							*/
							double artboardSpacing;

							/**
								\brief	How many artboards should appear in each row or column before wrapping occurs
								
								\note	When artboardLayout is ArtboardLayoutGridByRowLR, ArtboardLayoutGridByColumnLR,
										ArtboardLayoutGridByRowRL, or ArtboardLayoutGridByColumnRL then the valid range is
										[1,artboardCount-1]. If the value specified is out of this range, then the default
										value of sqrt(artboardCount) is used instead.
								\note	When artboardLayout is ArtboardLayoutColumn, ArtboardLayoutRowLR, or
										ArtboardLayoutRowRL then the value is ignored and taken as 1
							*/
							uint32_t artboardRowOrColumnSize;

							/**
								\brief	Controls the color model of the document
							*/
							DocumentColorModel colorModel;

							/**
								\brief	Controls the ruler units of the document
							*/
							RulerUnits rulerUnits;

							/**
								\brief	Controls the preview mode of the document
							*/
							PreviewMode previewMode;

							/**
								\brief	Controls the transparency grid color of the document
							*/
							TransparencyGrid transparencyGrid;

							/**
								\brief	Controls the raster resolution of the document
							*/
							RasterResolution rasterResolution;
							
							/**
								\brief	Constructs a new Preset object with default values
								\author	GW
								\date	08/2013
							*/
							Preset();
							
							/**
								\brief	Destructs a Preset object
								\author	GW
								\date	08/2013
							*/
							virtual ~Preset();
					};
				#endif
				// HDI_CORE_AIP_MODE

				/**
					\brief	Constructs an empty Document object
					\author	GW
					\date	08/2013
					
					\note	To test if a Document object is empty, call isEmpty() on it
					\note	Empty Document objects do not relate to any actual app document; they are designed to be
							"receivers" of some other Document object via the overloaded assignment operator. Empty
							Document objects are useless until such time (though it is safe to call any of their methods)
				*/
				Document();

				/**
					\brief	Document copy constructor
					\author	GW
					\date	08/2013

					\param	doc_	Existing Document object to copy
				*/
				Document(const Document& doc_);
				
				#if defined(HDI_CORE_AIP_MODE)
					/**
						\brief	Constructs a new Document object from an AIDocumentHandle
						\author	GW
						\date	08/2015
						
						\param	aiDoc_	Raw Illustrator AIDocumentHandle
					*/
					Document(const AIDocumentHandle aiDoc_);

					/**
						\brief	Creates a new document, and makes it current, from a startup preset
						\author	GW
						\date	08/2013

						\param	preset_			Name of a startup preset to use when creating the new document, as UTF-8
						\param	showDialog_		When true, shows the new document dialog to query user for parameters
					*/
					Document(const std::string& preset_, const bool showDialog_);

					/**
						\brief	Creates a new document, and makes it current, from new document parameters
						\author	GW
						\date	08/2013

						\param	preset_			Parameters for the new document
						\param	showDialog_		When true, shows the new document dialog to query user for parameters
					*/
					Document(const Preset& preset_, const bool showDialog_);
				#elif defined(HDI_CORE_PSP_MODE)
					/**
						\brief	Constructs a new Document object from a Photoshop document ID
						\author	GW
						\date	12/2017
						
						\param	psDoc_	Raw Photoshop document ID
					*/
					Document(const int32_t psDoc_);
				#endif

				/**
					\brief	Document destructor
					\author	GW
					\date	08/2013
				*/
				virtual ~Document();
			
				/**
					\brief		Assigns one Document object to another
					\author		GW
					\date		08/2013

					\param		rhs_	Existing Document object to copy values from
					\returns	The target Document object, but with its values updated to match that of the rhs_ argument
				*/
				virtual Document& operator=(const Document& rhs_);

				#if defined(HDI_CORE_AIP_MODE)
					/**
						\brief		Gets the document handle around which the target object is wrapped
						\author		GW
						\date		11/2013
						
						\returns	AIDocumentHandle for this object

						\note		Generally, you shouldn't need to call this method. Only use it if you know what you're
									doing. If a specific piece of functionality provided by Illustrator is not handled by
									this class (or related classes), then it should probably be added to the hdi_core library.
					*/
					virtual AIDocumentHandle aiDocumentHandle() const;
				#elif defined(HDI_CORE_PSP_MODE)
					/**
						\brief		Gets the document ID around which the target object is wrapped
						\author		GW
						\date		12/2017
						
						\returns	Document ID for this object

						\note		Generally, you shouldn't need to call this method. Only use it if you know what you're
									doing. If a specific piece of functionality provided by Photoshop is not handled by
									this class (or related classes), then it should probably be added to the hdi_core library.
					*/
					virtual int32_t psDocumentID() const;
				#endif

				/**
					\brief		Gets whether the target Document object is empty (constructed with the default ctor)
					\author		GW
					\date		08/2013
					
					\returns	true if the target Document object is empty, false otherwise
				*/
				virtual bool isEmpty() const;

				/**
					\brief	Saves the target document
					\author	GW
					\date	08/2013
				*/
				virtual void save() const;

				/**
					\brief	Closes the target document
					\author	GW
					\date	08/2013
				*/
				virtual void close() const;

				/**
					\brief	Activates (makes current) the target document
					\author	GW
					\date	08/2013
				*/
				virtual void activate() const;

				/**
					\brief	Prints the target document
					\author	GW
					\date	08/2013

					\param	showDialog_		When true, shows the print dialog to query the user for parameters
				*/
				virtual void print(const bool showDialog_) const;

				/**
					\brief		Gets the document URL
					\author		GW
					\date		08/2013

					\returns	The URL (location) of the document, as UTF-8
				*/
				virtual std::string url() const;

				/**
					\brief		Gets whether the document exists (is open or in the clipboard)
					\author		GW
					\date		08/2013

					\returns	true if the document exists, or false if it is closed
				*/
				virtual bool exists() const;
			
				/**
					\brief		Checks whether one Document object represents the same document as another
					\author		GW
					\date		08/2013

					\param		rhs_	Existing Document object to compare with
					\returns	true if the represented documents are the same, false otherwise
				*/
				virtual bool operator==(const Document& rhs_) const;
			
				/**
					\brief		Checks whether one Document object does not represent the same document as another
					\author		GW
					\date		08/2013

					\param		rhs_	Existing Document object to compare with
					\returns	true if the represented documents are not the same, false otherwise
				*/
				virtual bool operator!=(const Document& rhs_) const;
				
				#if defined(HDI_CORE_AIP_MODE)
					/**
						\brief		Gets the number of art styles in the document
						\author		GW
						\date		11/2013

						\returns	A count of the number of art styles in the target document
					*/
					uint32_t artStyleCount() const;
					
					/**
						\brief		Gets a given art style, by its index, in the target doc
						\author		GW
						\date		11/2013

						\param		index_	Index number of the style in question
						\returns	The ArtStyle object with the specified index
					*/
					std::unique_ptr<ArtStyle> artStyleAtIndex(const uint32_t index_) const;
					
					/**
						\brief		Gets the number of symbol definitions in the document
						\author		GW
						\date		11/2013

						\returns	A count of the number of symbol definitions in the target document
					*/
					uint32_t symbolDefCount() const;
					
					/**
						\brief		Gets a given symbol definition, by its index, in the target doc
						\author		GW
						\date		11/2013

						\param		index_	Index number of the symbol def in question
						\returns	The Symbol object with the specified index
					*/
					std::unique_ptr<Symbol> symbolDefAtIndex(const uint32_t index_) const;
				#endif
				// HDI_CORE_AIP_MODE
			
				#if defined(HDI_CORE_PSP_MODE)
					/**
						\brief		Gets the center zoomed point of the target document
						\author		GW
						\date		12/2017
						
						\returns	The document's center zoom point
					*/
					virtual Point center() const;
				
					/**
						\brief		Gets the width and height of the target document
						\author		GW
						\date		12/2017
						
						\returns	The document's width and height
					*/
					virtual Size dimensions() const;
				
					/**
						\brief	Sets the width and height of the target document
						\author	GW
						\date	12/2017
						
						\param	s_		The document's new width and height
						\param	ii_		The interpolation option for resizing the document contents
					*/
					virtual void setDimensions(const Size& s_, const ImageInterpolation ii_);
				
					/**
						\brief		Gets the color space of the target document
						\author		GW
						\date		12/2017
						
						\returns	The document's color space
					*/
					virtual ColorSpace colorSpace() const;
				
					/**
						\brief	Sets the color space of the target document
						\author	GW
						\date	12/2017
						
						\param	cs_		The document's new color space
						\param	ml_		Whether to merge the document's layers before converting
					*/
					virtual void setColorSpace(const ColorSpace cs_, const bool ml_);
				
					/**
						\brief		Gets the resolution (PPI) of the target document
						\author		GW
						\date		12/2017
						
						\returns	The document's resolution
					*/
					virtual double resolution() const;
				
					/**
						\brief	Sets the resolution (PPI) of the target document
						\author	GW
						\date	12/2017
						
						\param	res_	The document's new resolution
						\param	ii_		The interpolation option for resizing the document contents
					*/
					virtual void setResolution(const double res_, const ImageInterpolation ii_);

					/**
						\brief		Gets the bit depth of the target document
						\author		GW
						\date		12/2017
						
						\returns	The document's depth
					*/
					virtual int16_t depth() const;

					/**
						\brief	Sets the bit depth of the target document
						\author	GW
						\date	12/2017
						
						\param	d_	The document's new bit depth (values other than 8, 16, and 32 will be ignored)
					*/
					virtual void setDepth(const int16_t d_);

					/**
						\brief		Gets the ruler origin point of the target document
						\author		GW
						\date		12/2017
						
						\returns	The document's ruler origin
					*/
					virtual Point rulerOrigin() const;
			
					/**
						\brief		Gets whether the smart guides are visible in the document
						\author		GW
						\date		12/2017
						
						\returns	true for the smart guides being visible, false otherwise
					*/
					virtual bool smartGuidesVisible() const;
			
					/**
						\brief		Gets whether the guides are visible in the document
						\author		GW
						\date		12/2017
						
						\returns	true for the guides being visible, false otherwise
					*/
					virtual bool guidesVisible() const;
		
					/**
						\brief		Gets whether the rulers are visible in the document
						\author		GW
						\date		12/2017
						
						\returns	true for the rulers being visible, false otherwise
					*/
					virtual bool rulersVisible() const;
			
					/**
						\brief		Gets whether the document has a background layer
						\author		GW
						\date		12/2017
						
						\returns	true for the presence of a background layer, false otherwise
					*/
					virtual bool hasBackgroundLayer() const;

					/**
						\brief		Gets the background layer, if one exists
						\author		GW
						\date		12/2017

						\returns	A Layer object for the background layer for the document, or NULL if none
					*/
					virtual std::unique_ptr<Layer> backgroundLayer() const;
				
					/**
						\brief		Gets the zoom level of the document
						\author		GW
						\date		12/2017
						
						\returns	The document zoom level, where e.g. 2.0 means 200% zoom
					*/
					virtual double zoomLevel() const;
				
					/**
						\brief		Gets the pixel scale factor of the document
						\author		GW
						\date		12/2017
						
						\returns	The document pixel scale factor
					*/
					virtual double pixelScaleFactor() const;
				
					/**
						\brief		Gets the color profile name string for the document
						\author		GW
						\date		12/2017
						
						\returns	Color profile name
					*/
					virtual std::string colorProfile() const;

					/**
						\brief		Gets the name of the file format of the target document
						\author		GW
						\date		12/2017
						
						\returns	The document's file format name
					*/
					virtual std::string formatName() const;

					/**
						\brief		Gets a count of the layers in the target doc
						\author		GW
						\date		12/2017

						\returns	A count of the number of layers in the target document
					*/   
					virtual uint32_t layerCount() const;

					/**
						\brief		Gets the first layer in the list in the target doc
						\author		GW
						\date		12/2017

						\returns	A Layer object for the first layer in the document
					*/
					virtual std::unique_ptr<Layer> firstLayer() const;

					/**
						\brief		Gets the last layer in the list in the target doc
						\author		GW
						\date		12/2017

						\returns	A Layer object for the last layer in the document
					*/
					virtual std::unique_ptr<Layer> lastLayer() const;

					/**
						\brief		Gets a given layer, by its name, in the target doc (case-sensitive)
						\author		GW
						\date		12/2017

						\param		title_	Title of the layer in question
						\returns	A Layer object with the specified title, or NULL if none
					*/
					virtual std::unique_ptr<Layer> layerByTitle(const std::string& title_) const;

					/**
						\brief		Gets a given layer, by its index, in the target doc
						\author		GW
						\date		12/2017

						\param		index_	Index number of the layer in question
						\returns	A Layer object with the specified index, or NULL if out of bounds
					*/
					virtual std::unique_ptr<Layer> layerAtIndex(const uint32_t index_) const;

					/**
						\brief		Inserts a new layer at a given position in the layer list in the target document
						\author		GW
						\date		12/2017

						\param		pos_	Position (index) at which to place the new layer
						\returns	A Layer object located at the specified position
					*/
					virtual std::unique_ptr<Layer> insertLayer(const uint32_t pos_);

					/**
						\brief		Gets the currently selected layer of the target document
						\author		GW
						\date		12/2017

						\returns	A Layer object for the currently selected layer, or NULL if none

						\note		The caller does not need to manage the memory for the returned Layer object.
					*/
					virtual std::unique_ptr<Layer> currentLayer();

					/**
						\brief	Sets the currently selected layer in the target doc
						\author	GW
						\date	12/2017

						\param	layer_	The layer to select
					*/
					virtual void setCurrentLayer(const Layer& layer_);
			
					/**
						\brief		Gets the count of channels in the document
						\author		GW
						\date		02/2018
						
						\returns	Channel count, or 0 if an error occurred
					*/
					virtual uint32_t channelCount() const;
			
					/**
						\brief		Gets the name of the indicated channel
						\author		GW
						\date		02/2018
						
						\param		ci_		Index of the desired channel
						\returns	The channel name, of "" if an error occurred
					*/
					virtual std::string channelName(const uint32_t ci_) const;
			
					/**
						\brief		Gets whether the indicated channel is visible
						\author		GW
						\date		02/2018
						
						\param		ci_		Index of the desired channel
						\returns	true if visible, false otherwise
					*/
					virtual bool channelVisible(const uint32_t ci_) const;
			
					/**
						\brief	Selects the given channel to make it current
						\author	GW
						\date	02/2018
						
						\param	ci_		Index of the desired channel
					*/
					virtual void setCurrentChannel(const uint32_t ci_);
			
					/**
						\brief		Gets the XMP metadata as an XMPMeta object
						\author		GW
						\date		02/2018
					 
						\returns	A pointer to an XMPMeta object for the document's XMP metadata, or NULL for none/error
					 
						\warning	Action plugins cannot access the XMP metadata.
					*/
					virtual std::unique_ptr<XMPMeta> getXMP() const;
			
					/**
						\brief		Sets the XMP metadata from an XMPMeta object
						\author		GW
						\date		02/2018
					 
						\param		xmp_	An XMPMeta object for the document's XMP metadata
					 
						\warning	Action plugins cannot modify the XMP metadata.
					*/
					virtual void setXMP(const XMPMeta& xmp_);
				#endif
				// HDI_CORE_PSP_MODE


			private:
				#if defined(HDI_CORE_AIP_MODE)
					friend aip::Document* __accessImpl(const Document&);
					friend Document __accessCtor(aip::Document*&);
					
					/**
						\brief	Internal use only
						\author	GW
						\date	08/2013
					*/
					Document(aip::Document*&);
				#elif defined(HDI_CORE_PSP_MODE)
					friend psp::Document* __accessImpl(const Document&);
					friend Document __accessCtor(psp::Document*&);
					
					/**
						\brief	Internal use only
						\author	GW
						\date	12/2017
					*/
					Document(psp::Document*&);
				#endif

				/**
					\brief	Private implementation data
				*/
				void* __data;
				
				/**
					\brief	Internal use only
					\author	GW
					\date	08/2013
				*/
				void* __impl() const;
		};
		
		typedef std::unique_ptr<Document> DocumentUP;
		typedef std::shared_ptr<Document> DocumentSP;
		typedef std::weak_ptr<Document> DocumentWP;
		
		#if defined(HDI_CORE_AIP_MODE)
			extern aip::Document* __accessImpl(const Document&);
			extern Document __accessCtor(aip::Document*&);
		#elif defined(HDI_CORE_PSP_MODE)
			extern psp::Document* __accessImpl(const Document&);
			extern Document __accessCtor(psp::Document*&);
		#endif
	}
}

#endif
// __HDI_CORE_DOCUMENT__
