#ifndef __HDI_CORE_DRAGGABLE_DIVIDER__
#define __HDI_CORE_DRAGGABLE_DIVIDER__

#include "hdicoreCustomWidget.h"

namespace hdi
{
	namespace core
	{
		/**
			\brief	Class for draggable divider lines, which are of a standard length, and used for resizing e.g. the
					height of the two widgets immediately above and below the divider
		*/
		class DraggableDivider : public CustomWidget
		{
			public:
				/**
					\brief	Describes the direction of the line in a DraggableDivider object
				*/
				enum Direction
				{
					UnknownDirection	= 0,
					Horizontal			= 10,
					Vertical			= 20
				};
			
				/**
					\brief	Constructs an empty DraggableDivider object
					\author	GW
					\date	11/2015
					
					\note	To test if a DraggableDivider object is empty, call isEmpty() on it
					\note	Empty DraggableDivider objects do not relate to any actual UI widget; they are designed to be
							"receivers" of some other DraggableDivider object via the overloaded assignment operator.
							Empty DraggableDivider objects are useless until such time (though it is safe to call any of
							their methods).
				*/
				DraggableDivider();

				/**
					\brief	Constructs a new DraggableDivider object from an existing DraggableDivider object (copy
							constructor)
					\author	GW
					\date	11/2015

					\param	widget_		Existing DraggableDivider object
				*/
				DraggableDivider(const DraggableDivider& widget_);

				/**
					\brief	DraggableDivider constructor, taking an argument for the location and direction
					\author	GW
					\date	11/2015

					\param	loc_	Top-left location of the divider, in 1x resolution coordinates
					\param	dir_	Horizontal or vertical line?
				*/
				DraggableDivider(const Point& loc_, const Direction dir_);
				
				/**
					\brief	DraggableDivider destructor
					\author	GW
					\date	11/2015
				*/
				virtual ~DraggableDivider();

				/**
					\brief		Allows one DraggableDivider object to be assigned from another
					\author		GW
					\date		11/2015

					\param		rhs_	Righthand side of the = operator; the object to copy values from
					\returns	The target DraggableDivider object, but with its value updated to match that of rhs_
				*/
				virtual DraggableDivider& operator=(const DraggableDivider& rhs_);
				
				/**
					\brief		Convenience method to clone a DraggableDivider object on the heap
					\author		GW
					\date		11/2015

					\returns	A pointer to the new DraggableDivider object
					
					\note		If you subclass DraggableDivider, you MUST overload this method yourself! If you don't
								and/or your clone() method does not return an instance of your DraggableDivider subclass,
								you will experience "object slicing" when adding the widget to a window.
								
					\warning	The caller must manage the memory for the returned DraggableDivider object.
				*/
				virtual DraggableDivider* clone() const;
				
				/**
					\brief		Convenience method to duplicate a DraggableDivider object, creating a new and identical
								UI element to the target (but not belonging to the same containing window)
					\author		GW
					\date		11/2015
					
					\returns	A pointer to the new DraggableDivider object (and new UI element)
					
					\warning	The caller must manage the memory for the returned DraggableDivider object.
				*/
				virtual DraggableDivider* duplicate() const;

				/**
					\brief		Gets the direction that the line is going
					\author		GW
					\date		11/2015

					\returns	Vertical for a vertical line, Horizontal otherwise
				*/
				virtual Direction direction() const;

				/**
					\brief		Always gets "", as dividers cannot have text
					\author		GW
					\date		11/2015
					
					\returns	""
				*/
				virtual std::string text() const;

				/**
					\brief	Prevents setting the text, as dividers cannot have text
					\author	GW
					\date	11/2015
				*/
				virtual void setText(const std::string&);

				/**
					\brief		Returns true, because dividers are always enabled
					\author		GW
					\date		11/2015

					\returns	true
				*/
				virtual bool enabled();

				/**
					\brief	Overrides Widget::setEnabled() because dividers are always enabled
					\author	GW
					\date	11/2015

					\param	enable_		true to enable, false to disable (ignored)
				*/
				virtual void setEnabled(const bool enable_);
			
				/**
					\brief		Gets the callback for the widget being moved during a mouse drag event
					\author		GW
					\date		11/2015
					
					\returns	A pointer to the currently registered move callback
				*/
				virtual Callback* const movedCallback() const;
			
				/**
					\brief	Sets the callback for the widget being moved during a mouse drag event
					\author	GW
					\date	11/2015
					
					\param	cb_		New callback (target and action) for the widget moving due to the user dragging it
									to one side or another
				*/
				virtual void setMovedCallback(const Callback& cb_);


			private:
				// All of these are made private because they are unnecessary for this subclass
				void redraw();
				ModifierKey getClickDragModifiers() const;
				bool getClickDragLocations(Point& start__, Point& end__) const;
				bool isMouseOver() const;
				Callback* const mouseOverCallback() const;
				void setMouseOverCallback(const Callback& cb_);
				Callback* const mouseMoveCallback() const;
				void setMouseMoveCallback(const Callback& cb_);
				Callback* const mouseOutCallback() const;
				void setMouseOutCallback(const Callback& cb_);
				Callback* const mouseDownCallback() const;
				void setMouseDownCallback(const Callback& cb_);
				Callback* const mouseDragCallback() const;
				void setMouseDragCallback(const Callback& cb_);
				Callback* const mouseUpCallback() const;
				void setMouseUpCallback(const Callback& cb_);
				Callback* const drawCallback() const;
				void setDrawCallback(const Callback& cb_);
				bool clearsOnRedraw() const;
				void setClearsOnRedraw(const bool clear_);
				void setDrawColor(const Color& color_);
				void drawLine(const Point& from_, const Point& to_);
				void drawRect(const Rect& rect_, const bool fill_ = false);
				void drawPolygon(const PointVector& points_, const bool fill_ = false);
		};
		
		typedef std::unique_ptr<DraggableDivider> DraggableDividerUP;
		typedef std::shared_ptr<DraggableDivider> DraggableDividerSP;
		typedef std::weak_ptr<DraggableDivider> DraggableDividerWP;
	}
}

#endif
// __HDI_CORE_DRAGGABLE_DIVIDER__
