/**
	\file
	\brief		Header file for basic Illustrator drawing functionality
	\copyright	Hot Door, Inc. 2010-2025
*/

#ifndef __HDI_CORE_DRAW__
#define __HDI_CORE_DRAW__

#if defined(HDI_CORE_AIP_MODE)

#include <memory>
#include <vector>

#include "hdicoreTypes.h"

namespace hdi
{
	namespace core
	{
		class Art;
		class ArtboardPoint;
		class ArtboardRect;
		class ArtColor;
		class Font;
		class PathStyle;
		
		/**
			\brief	Organizes all primitive shape drawing functionality into one namespace for convenience
		*/
		namespace draw
		{
			typedef std::vector<ArtboardPoint> ArtboardPointVector;
		
			/**
				\brief		Draws a single segment point at a given location in the current document
				\author		GW
				\date		08/2013

				\param		loc_		Location for the segment point
				\param		group_		Group the new art will go in (NULL for top of layer list)
				\returns	The new Art object, or NULL for error
			 
			 	\warning	This function does not assume responsibility of the memory for the group_ argument.
			*/
			std::unique_ptr<Art> point(const ArtboardPoint& loc_, const Art* const group_ = NULL);

			/**
				\brief		Draws a line in group_ from start_ to end_
				\author		GW
				\date		08/2013

				\param		start_		Start point for the line
				\param		end_		End point for the line
				\param		style_		Style for the new line
				\param		group_		Group the new art will go in (NULL for top of layer list)
				\returns	The new Art object, or NULL for error
			 
			 	\warning	This function does not assume responsibility of the memory for the group_ argument.
			*/
			std::unique_ptr<Art> line(
				const ArtboardPoint& start_,
				const ArtboardPoint& end_,
				const PathStyle& style_,
				const Art* const group_ = NULL
			);

			/**
				\brief		Draws a line in group_ from start_ to end_
				\author		GW
				\date		08/2013

				\param		start_		Start point for the line
				\param		end_		End point for the line
				\param		group_		Group the new art will go in (NULL for top of layer list)
				\returns	The new Art object, or NULL for error
			 
			 	\warning	This function does not assume responsibility of the memory for the group_ argument.
			*/
			std::unique_ptr<Art> line(
				const ArtboardPoint& start_,
				const ArtboardPoint& end_,
				const Art* const group_ = NULL
			);

			/**
				\brief		Draws the rectangle indicated by rect_ in group_
				\author		GW
				\date		08/2013

				\param		rect_		Rectangle to draw
				\param		style_		Style for the new rect
				\param		group_		Group the new art will go in (NULL for top of layer list)
				\returns	The new Art object, or NULL for error
			 
			 	\warning	This function does not assume responsibility of the memory for the group_ argument.
			*/
			std::unique_ptr<Art> rect(const ArtboardRect& rect_, const PathStyle& style_, const Art* const group_ = NULL);
			
			/**
				\brief		Draws the rectangle indicated by rect_ in group_
				\author		GW
				\date		08/2013

				\param		rect_		Rectangle to draw
				\param		group_		Group the new art will go in (NULL for top of layer list)
				\returns	The new Art object, or NULL for error
			 
			 	\warning	This function does not assume responsibility of the memory for the group_ argument.
			*/
			std::unique_ptr<Art> rect(const ArtboardRect& rect_, const Art* const group_ = NULL);

			/**
				\brief		Draws the rounded rectangle indicated by rect_ and radius_ in group_
				\author		GW
				\date		08/2013

				\param		rect_		Rounded rectangle to draw
				\param		radius_		Radius for the corners of the rectangle
				\param		style_		Style for the new rect
				\param		group_		Group the new art will go in (NULL for top of layer list)
				\returns	The new Art object, or NULL for error
			 
			 	\warning	This function does not assume responsibility of the memory for the group_ argument.
			*/
			std::unique_ptr<Art> roundedRect(
				const ArtboardRect& rect_,
				const double radius_,
				const PathStyle& style_,
				const Art* const group_ = NULL
			);
			
			/**
				\brief		Draws the rounded rectangle indicated by rect_ and radius_ in group_
				\author		GW
				\date		08/2013

				\param		rect_		Rounded rectangle to draw
				\param		radius_		Radius for the corners of the rectangle
				\param		group_		Group the new art will go in (NULL for top of layer list)
				\returns	The new Art object, or NULL for error
			 
			 	\warning	This function does not assume responsibility of the memory for the group_ argument.
			*/
			std::unique_ptr<Art> roundedRect(
				const ArtboardRect& rect_,
				const double radius_,
				const Art* const group_ = NULL
			);

			/**
				\brief		Draws the chamfered rectangle indicated by rect_ and dist_ in group_
				\author		GW
				\date		08/2013

				\param		rect_		Chamfered rectangle to draw
				\param		dist_		Distance inward from the corners at which the chamfer will begin
				\param		style_		Style for the new rect
				\param		group_		Group the new art will go in (NULL for top of layer list)
				\returns	The new Art object, or NULL for error
			 
			 	\warning	This function does not assume responsibility of the memory for the group_ argument.
			*/
			std::unique_ptr<Art> chamferedRect(
				const ArtboardRect& rect_,
				const double dist_,
				const PathStyle& style_,
				const Art* const group_ = NULL
			);
			
			/**
				\brief		Draws the chamfered rectangle indicated by rect_ and dist_ in group_
				\author		GW
				\date		08/2013

				\param		rect_		Chamfered rectangle to draw
				\param		dist_		Distance inward from the corners at which the chamfer will begin
				\param		group_		Group the new art will go in (NULL for top of layer list)
				\returns	The new Art object, or NULL for error
			 
			 	\warning	This function does not assume responsibility of the memory for the group_ argument.
			*/
			std::unique_ptr<Art> chamferedRect(
				const ArtboardRect& rect_,
				const double dist_,
				const Art* const group_ = NULL
			);

			/**
				\brief		Draws a polygon from a series of points
				\author		GW
				\date		08/2013

				\param		points_		Vector of points for the polygon
				\param		style_		Style for the new art
				\param		group_		Group the new art will go in (NULL for top of layer list)
				\returns	The new Art object, or NULL for error
			 
			 	\warning	This function does not assume responsibility of the memory for the group_ argument.
			*/
            std::unique_ptr<Art> polygon(
				const ArtboardPointVector& points_,
				const PathStyle& style_,
				const Art* const group_ = NULL
			);
			
			/**
				\brief		Draws a polygon from a series of points
				\author		GW
				\date		08/2013

				\param		points_		Vector of points for the polygon
				\param		group_		Group the new art will go in (NULL for top of layer list)
				\returns	The new Art object, or NULL for error
			 
			 	\warning	This function does not assume responsibility of the memory for the group_ argument.
			*/
            std::unique_ptr<Art> polygon(const ArtboardPointVector& points_, const Art* const group_ = NULL);
			
			/**
				\brief		Draws a regular polygon of a given radius with a given number of sides at a given point
				\author		GW
				\date		06/2015
				
				\param		center_		Center point of the polygon
				\param		radius_		Radius of the polygon, in points
				\param		sides_		Number of sides for the polygon, in range [3,100]
				\param		style_		Style for the new art
				\param		group_		Group the new art will go in (NULL for top of layer list)
				\returns	The new Art object, or NULL for error
			 
			 	\warning	This function does not assume responsibility of the memory for the group_ argument.
			*/
			std::unique_ptr<Art> regularPolygon(
				const ArtboardPoint& center_,
				const double radius_,
				const int16_t sides_,
				const PathStyle& style_,
				const Art* const group_ = NULL
			);
			
			/**
				\brief		Draws a regular polygon of a given radius with a given number of sides at a given point
				\author		GW
				\date		06/2015
				
				\param		center_		Center point of the polygon
				\param		radius_		Radius of the polygon, in points
				\param		sides_		Number of sides for the polygon, in range [3,100]
				\param		group_		Group the new art will go in (NULL for top of layer list)
				\returns	The new Art object, or NULL for error
			 
			 	\warning	This function does not assume responsibility of the memory for the group_ argument.
			*/
			std::unique_ptr<Art> regularPolygon(
				const ArtboardPoint& center_,
				const double radius_,
				const int16_t sides_,
				const Art* const group_ = NULL
			);
			
			/**
				\brief		Draws a regular polygon with a given number of sides at a given point, extending to a given
							vertex point (the center often being the click point and the vertex often being the cursor)
				\author		GW
				\date		06/2015
				
				\param		center_		Center point of the polygon
				\param		vertex_		Vertex point for the polygon (controls radius)
				\param		sides_		Number of sides for the polygon, in range [3,100]
				\param		style_		Style for the new art
				\param		group_		Group the new art will go in (NULL for top of layer list)
				\returns	The new Art object, or NULL for error
			 
			 	\warning	This function does not assume responsibility of the memory for the group_ argument.
			*/
			std::unique_ptr<Art> regularPolygon(
				const ArtboardPoint& center_,
				const ArtboardPoint& vertex_,
				const int16_t sides_,
				const PathStyle& style_,
				const Art* const group_ = NULL
			);
			
			/**
				\brief		Draws a regular polygon with a given number of sides at a given point, extending to a given
							vertex point (the center often being the click point and the vertex often being the cursor)
				\author		GW
				\date		06/2015
				
				\param		center_		Center point of the polygon
				\param		vertex_		Vertex point for the polygon (controls radius)
				\param		sides_		Number of sides for the polygon, in range [3,100]
				\param		group_		Group the new art will go in (NULL for top of layer list)
				\returns	The new Art object, or NULL for error
			 
			 	\warning	This function does not assume responsibility of the memory for the group_ argument.
			*/
			std::unique_ptr<Art> regularPolygon(
				const ArtboardPoint& center_,
				const ArtboardPoint& vertex_,
				const int16_t sides_,
				const Art* const group_ = NULL
			);
			
			/**
				\brief		Draws an ellipse from source_ of width_ and height_ in group_ (centered_ determines where
							source_ is... see arguments list)
				\author		GW
				\date		08/2013

				\param		source_		Source point for the ellipse; centered_ determines whether this is the top-left
										or the center of the ellipse
				\param		width_		Width of the ellipse
				\param		height_		Height of the ellipse
				\param		centered_	Whether the ellipse is centered on source_
				\param		style_		Style for the new ellipse
				\param		group_		Group the new art will go in (NULL for top of layer list)
				\returns	The new Art object, or NULL for error
			 
			 	\warning	This function does not assume responsibility of the memory for the group_ argument.
			*/
			std::unique_ptr<Art> ellipse(
				const ArtboardPoint& source_,
				const double width_,
				const double height_,
				const bool centered_,
				const PathStyle& style_,
				const Art* const group_ = NULL
			);
			
			/**
				\brief		Draws an ellipse from source_ of width_ and height_ in group_ (centered_ determines where
							source_ is... see arguments list)
				\author		GW
				\date		08/2013

				\param		source_		Source point for the ellipse; centered_ determines whether this is the top-left
										or the center of the ellipse
				\param		width_		Width of the ellipse
				\param		height_		Height of the ellipse
				\param		centered_	Whether the ellipse is centered on source_
				\param		group_		Group the new art will go in (NULL for top of layer list)
				\returns	The new Art object, or NULL for error
			 
			 	\warning	This function does not assume responsibility of the memory for the group_ argument.
			*/
			std::unique_ptr<Art> ellipse(
				const ArtboardPoint& source_,
				const double width_,
				const double height_,
				const bool centered_,
				const Art* const group_ = NULL
			);

			/**
				\brief		Draws a star with a given number of points, radii, etc. at a given point
				\author		GW
				\date		08/2013

				\param		center_			Center point for the star
				\param		points_			Number of points (tips) in the star
				\param		innerRadius_	Radius of the star body
				\param		outerRadius_	Radius of the star tips
				\param		style_			Style for the new star
				\param		group_			Group the new art will go in (NULL for top of layer list)
				\returns	The new Art object, or NULL for error
			 
			 	\warning	This function does not assume responsibility of the memory for the group_ argument.
			*/
			std::unique_ptr<Art> star(
				const ArtboardPoint& center_,
				const uint16_t points_,
				const double innerRadius_,
				const double outerRadius_,
				const PathStyle& style_,
				const Art* const group_ = NULL
			);
			
			/**
				\brief		Draws a star with a given number of points, radii, etc. at a given point
				\author		GW
				\date		08/2013

				\param		center_			Center point for the star
				\param		points_			Number of points (tips) in the star
				\param		innerRadius_	Radius of the star body
				\param		outerRadius_	Radius of the star tips
				\param		group_			Group the new art will go in (NULL for top of layer list)
				\returns	The new Art object, or NULL for error
			 
			 	\warning	This function does not assume responsibility of the memory for the group_ argument.
			*/
			std::unique_ptr<Art> star(
				const ArtboardPoint& center_,
				const uint16_t points_,
				const double innerRadius_,
				const double outerRadius_,
				const Art* const group_ = NULL
			);

			/**
				\brief		Draws an equilateral triangle of a given (base) width at a given point
				\author		GW
				\date		08/2013

				\param		center_		Center point for the triangle
				\param		width_		Width of the base of the triangle
				\param		style_		Style for the new triangle
				\param		group_		Group the new art will go in (NULL for top of layer list)
				\returns	The new Art object, or NULL for error
			 
			 	\warning	This function does not assume responsibility of the memory for the group_ argument.
			*/
			std::unique_ptr<Art> triangle(
				const ArtboardPoint& center_,
				const double width_,
				const PathStyle& style_,
				const Art* const group_ = NULL
			);
			
			/**
				\brief		Draws an equilateral triangle of a given (base) width at a given point
				\author		GW
				\date		08/2013

				\param		center_		Center point for the triangle
				\param		width_		Width of the base of the triangle
				\param		group_		Group the new art will go in (NULL for top of layer list)
				\returns	The new Art object, or NULL for error
			 
			 	\warning	This function does not assume responsibility of the memory for the group_ argument.
			*/
			std::unique_ptr<Art> triangle(
				const ArtboardPoint& center_,
				const double width_,
				const Art* const group_ = NULL
			);

			/**
				\brief		Draws an arc segment (of a circle), whichever direction is the shortest length
				\author		GW
				\date		12/2023

				\param		center_		Center point of the circle (of which the arc is a segment)
				\param		start_		First point at which the arc segment begins, also controls the radius (distance
										from center_)
				\param		end_		Last point at which the arc segment ends
				\param		style_		Style for the new arc
				\param		group_		Group the new art will go in (NULL for top of layer list)
				\returns	The new Art object, or NULL for error
			 
			 	\warning	This function does not assume responsibility of the memory for the group_ argument.
			*/
			std::unique_ptr<Art> shortestArc(
				const ArtboardPoint& center_,
				const ArtboardPoint& start_,
				const ArtboardPoint& end_,
				const PathStyle& style_,
				const Art* const group_ = NULL
			);
			
			/**
				\brief		Draws an arc segment (of a circle), whichever direction is the shortest length
				\author		GW
				\date		12/2023

				\param		center_		Center point of the circle (of which the arc is a segment)
				\param		start_		First point at which the arc segment begins, also controls the radius (distance
										from center_)
				\param		end_		Last point at which the arc segment ends
				\param		group_		Group the new art will go in (NULL for top of layer list)
				\returns	The new Art object, or NULL for error
			 
			 	\warning	This function does not assume responsibility of the memory for the group_ argument.
			*/
			std::unique_ptr<Art> shortestArc(
				const ArtboardPoint& center_,
				const ArtboardPoint& start_,
				const ArtboardPoint& end_,
				const Art* const group_ = NULL
			);

			/**
				\brief		Draws an arc segment (of a circle), whichever direction is the longest length
				\author		GW
				\date		12/2023

				\param		center_		Center point of the circle (of which the arc is a segment)
				\param		start_		First point at which the arc segment begins, also controls the radius (distance
										from center_)
				\param		end_		Last point at which the arc segment ends
				\param		style_		Style for the new arc
				\param		group_		Group the new art will go in (NULL for top of layer list)
				\returns	The new Art object, or NULL for error
			 
			 	\warning	This function does not assume responsibility of the memory for the group_ argument.
			*/
			std::unique_ptr<Art> longestArc(
				const ArtboardPoint& center_,
				const ArtboardPoint& start_,
				const ArtboardPoint& end_,
				const PathStyle& style_,
				const Art* const group_ = NULL
			);
			
			/**
				\brief		Draws an arc segment (of a circle), whichever direction is the longest length
				\author		GW
				\date		12/2023

				\param		center_		Center point of the circle (of which the arc is a segment)
				\param		start_		First point at which the arc segment begins, also controls the radius (distance
										from center_)
				\param		end_		Last point at which the arc segment ends
				\param		group_		Group the new art will go in (NULL for top of layer list)
				\returns	The new Art object, or NULL for error
			 
			 	\warning	This function does not assume responsibility of the memory for the group_ argument.
			*/
			std::unique_ptr<Art> longestArc(
				const ArtboardPoint& center_,
				const ArtboardPoint& start_,
				const ArtboardPoint& end_,
				const Art* const group_ = NULL
			);

			/**
				\brief		Draws an arc segment (of a circle)
				\author		GW
				\date		08/2013

				\param		center_		Center point of the circle (of which the arc is a segment)
				\param		start_		First point at which the arc segment begins, also controls the radius (distance
										from center_)
				\param		end_		Last point at which the arc segment ends
				\param		style_		Style for the new arc
				\param		group_		Group the new art will go in (NULL for top of layer list)
				\returns	The new Art object, or NULL for error
			 
				\deprecated	This function should not be used and will eventually be removed; use the shortestArc() or
							longestArc() function instead.
				
			 	\warning	This function does not assume responsibility of the memory for the group_ argument.
			*/
			std::unique_ptr<Art> arc(
				const ArtboardPoint& center_,
				const ArtboardPoint& start_,
				const ArtboardPoint& end_,
				const PathStyle& style_,
				const Art* const group_ = NULL
			);
			
			/**
				\brief		Draws an arc segment (of a circle)
				\author		GW
				\date		08/2013

				\param		center_		Center point of the circle (of which the arc is a segment)
				\param		start_		First point at which the arc segment begins, also controls the radius (distance
										from center_)
				\param		end_		Last point at which the arc segment ends
				\param		group_		Group the new art will go in (NULL for top of layer list)
				\returns	The new Art object, or NULL for error
			 
				\deprecated	This function should not be used and will eventually be removed; use the shortestArc() or
							longestArc() function instead.
				
			 	\warning	This function does not assume responsibility of the memory for the group_ argument.
			*/
			std::unique_ptr<Art> arc(
				const ArtboardPoint& center_,
				const ArtboardPoint& start_,
				const ArtboardPoint& end_,
				const Art* const group_ = NULL
			);

			/**
				\brief		Draws text at a given point of a given font, size, justification, color, etc.
				\author		GW
				\date		08/2013

				\param		text_		Value of the string to draw, as UTF-8
				\param		where_		Point at which to place it
				\param		fontName_	Name of the font to use (if not found it will be drawn with the default font for
										the current locale), as UTF-8
				\param		fontSize_	Font size in points (in range 0.1 to 1296.0)
				\param		just_		Justification
				\param		color_		Color for the art
				\param		group_		Group the new art will go in (NULL for top of layer list)
				\returns	The new Art object, or NULL for error
			 
			 	\warning	This function does not assume responsibility of the memory for the group_ argument.
			*/
			std::unique_ptr<Art> text(
				const std::string& text_,
				const ArtboardPoint& where_,
				const std::string& fontName_,
				const double fontSize_,
				const ParagraphJustification just_,
				const ArtColor& color_,
				const Art* const group_ = NULL
			);

			/**
				\brief		Draws text at a given point of a given font, size, justification, color, etc.
				\author		GW
				\date		08/2013

				\param		text_		Value of the string to draw, as UTF-8
				\param		where_		Point at which to place it
				\param		fontName_	Font to use
				\param		fontSize_	Font size in points (in range 0.1 to 1296.0)
				\param		just_		Justification
				\param		color_		Color for the art
				\param		group_		Group the new art will go in (NULL for top of layer list)
				\returns	The new Art object, or NULL for error
			 
			 	\warning	This function does not assume responsibility of the memory for the group_ argument.
			*/
			std::unique_ptr<Art> text(
				const std::string& text_,
				const ArtboardPoint& where_,
				const Font& font_,
				const double fontSize_,
				const ParagraphJustification just_,
				const ArtColor& color_,
				const Art* const group_ = NULL
			);
		}
	}
}

#endif
// HDI_CORE_AIP_MODE

#endif
// __HDI_CORE_DRAW__
