/**
	\file
	\brief		Header file for storing entry string values or image IDs
	\copyright	Hot Door, Inc. 2010-2025
*/

#ifndef __HDI_CORE_ENTRY_VALUE__
#define __HDI_CORE_ENTRY_VALUE__

#include "hdicoreImageIDs.h"
#include "hdicoreSVGIDs.h"

namespace hdi
{
	namespace core
	{
		/**
			\brief	Allows for a *Entry class to store either a string value or PNGI image ID for a row/column
		*/
		class EntryValue
		{
			public:
				/**
					\brief	Indicates the type of value stored in an EntryValue object
				*/
				enum ValueType
				{
					UnknownValueType	= 0,
					StringValueType		= 10,
					ImageValueType		= 20,
					VectorValueType		= 30
				};
				
				/**
					\brief	EntryValue copy constructor
					\author	GW
					\date	02/2015
					
					\param	ev_		Existing EntryValue object to copy values from
				*/
				EntryValue(const EntryValue& ev_);
				
				/**
					\brief	Constructs an EntryValue object with a string value
					\author	GW
					\date	02/2015
					
					\param	str_	String value for the entry
				*/
				explicit EntryValue(const std::string& str_);
			
				/**
					\brief	Constructs an EntryValue object with PNGI image IDs
					\author	GW
					\date	02/2015
				 
					\param	ids_	Image ID values for the entry
				*/
				EntryValue(const ImageIDs& ids_);
			
				/**
					\brief	Constructs an EntryValue object with SVG image IDs
					\author	GW
					\date	12/2018
				 
					\param	ids_	SVG ID values for the entry
				*/
				EntryValue(const SVGIDs& ids_);
				
				/**
					\brief	EntryValue destructor
					\author	GW
					\date	02/2015
				*/
				~EntryValue();
				
				/**
					\brief		Overloaded assignment operator to copy values from one EntryValue object to another
					\author		GW
					\date		02/2015
					
					\param		rhs_	Righthand side of the assignment
					\returns	The lefthand side of the assignment, but with its values updated to match rhs_
				*/
				EntryValue& operator=(const EntryValue& rhs_);
				
				/**
					\brief		Overloaded equality operator to compare two EntryValue objects to one another
					\author		GW
					\date		02/2015
					
					\param		rhs_	Righthand side of the comparison
					\returns	true if the lefthand side and righthand side of the comparison have the same entry values
				*/
				bool operator==(const EntryValue& rhs_) const;
				
				/**
					\brief		Overloaded inequality operator to compare two EntryValue objects to one another
					\author		GW
					\date		02/2015
					
					\param		rhs_	Righthand side of the comparison
					\returns	true if the lefthand side and righthand side of the comparison have differing entry values
				*/
				bool operator!=(const EntryValue& rhs_) const;
				
				/**
					\brief		Gets the type of value representing by the target object
					\author		GW
					\date		02/2015
					
					\returns	The type of value stored within the target
				*/
				ValueType type() const;
				
				/**
					\brief		Gets the string value for the target object
					\author		GW
					\date		02/2015
					
					\returns	The string value if the type is StringValueType (as UTF-8), "" otherwise
				*/
				std::string stringValue() const;
				
				/**
					\brief		Sets the string value for the target object
					\author		GW
					\date		02/2015
					
					\param		str_	New string value for the target object, as UTF-8
					\returns	true if the type is StringValueType, false otherwise
				*/
				bool setStringValue(const std::string& str_);
			
				/**
					\brief		Gets the image ID values for the target object
					\author		GW
					\date		02/2015
				 
					\returns	The image IDs
				 
					\note		The return value should be ignored if the type() method does not return ImageValueType.
				*/
				ImageIDs imageIDs() const;
			
				/**
					\brief		Sets the image ID values for the target object
					\author		GW
					\date		02/2015
				 
					\param		ids_	New ID values for the target object
					\returns	true if the type is ImageValueType, false otherwise
				*/
				bool setImageIDs(const ImageIDs& ids_);
			
				/**
					\brief		Gets the SVG ID values for the target object
					\author		GW
					\date		12/2018
				 
					\returns	The vector IDs
				 
					\note		The return value should be ignored if the type() method does not return VectorValueType.
				*/
				SVGIDs svgIDs() const;
			
				/**
					\brief		Sets the SVG ID values for the target object
					\author		GW
					\date		12/2018
				 
					\param		ids_	New ID values for the target object
					\returns	true if the type is VectorValueType, false otherwise
				*/
				bool setSVGIDs(const SVGIDs& ids_);


			private:
				/**
					\brief	Stores the type of value that the object represents
				*/
				ValueType __type;
				
				/**
					\brief	Stores the string value if __type is StringValueType
				*/
				std::string __strVal;
			
				/**
					\brief	Stores the image ID values if __type is ImageValueType
				*/
				ImageIDs __imageIDs;
			
				/**
					\brief	Stores the SVG ID values if __type is VectorValueType
				*/
				SVGIDs __svgIDs;
			
				/**
					\brief	Unused
					\author	GW
					\date	02/2015
				*/
				EntryValue();
		};
		
		typedef std::unique_ptr<EntryValue> EntryValueUP;
		typedef std::shared_ptr<EntryValue> EntryValueSP;
		typedef std::weak_ptr<EntryValue> EntryValueWP;
	}
}

#endif
// __HDI_CORE_ENTRY_VALUE__
