/**
	\file
	\brief		Header file for Illustrator file format manipulation
	\copyright	Hot Door, Inc. 2010-2025
*/

#ifndef __HDI_CORE_FILE_FORMAT__
#define __HDI_CORE_FILE_FORMAT__

#if defined(HDI_CORE_AIP_MODE)

#include <vector>

#include "hdicoreTypes.h"

namespace hdi
{
	namespace core
	{
		/**
			\brief	Describes a file format that Illustrator can either read or write
		*/
		class FileFormat
		{
			public:
				typedef std::vector<std::string> ExtVector;
				
				/**
					\brief		Used to specify the capabilities of a file format
					\details	A subset of options are used with the FileFormat constructor that creates wholly new
								Illustrator file formats. These options specify which operations are supported, and are
								also sent in a FileFormatMessage to indicate the requested action. Additional options
								can be set in the FileFormatGoMessageType message to provide more information about the
								requested operation.

					\note		The read, embed (import/place), and link operations are very similar in execution. In
								all cases, Illustrator creates an empty artwork and send the plug-in a message to read
								the file into the artwork. Special handling is required only if the plug-in needs to do
								different things depending on the type of operation; for example, the Photoshop file
								format plug-in has options for preserving Photoshop's layers if it is not linking.
				*/
				enum Options
				{
					UnknownOptions					= 0,
					
					// Read the file, creating artwork in a new document. The format is included in the File > Open file
					// types. Use when adding a format.
					ReadOption						= 1 << 0,
					
					// Write the documentís contents to a file in a non-Illustrator format. The format is included in
					// the File > Export file types. Use when adding a format.
					ExportOption					= 1 << 1,
					
					// Write the documentís contents to a file in a format from which is can be read back into
					// Illustrator without loss of data. The format is included in the File > Save As file types. Use
					// when adding a format.
					WriteOption						= 1 << 9,
					
					// Read the file and embed artwork to the current document. The format is included in the File >
					// Import file types. Use when adding a format.
					ImportArtOption					= 1 << 2,
					
					// Read the file and embed artwork to the current document. This is the same as ImportArtOption. Use
					// when adding a format.
					PlaceArtOption					= 1 << 3,
					
					// Allows "Template" to be checked in the Place dialog when linking or embedding a file of this type,
					// so the art is placed in a template layer. Use when adding a format.
					ConvertTemplateOption			= 1 << 7,
					
					// Make a link from the contents of a file to an Illustrator document. Use when adding a format.
					LinkArtOption					= 1 << 8,
					
					// Unused!
					ImportStylesOption				= 1 << 4,
					
					// Unused!
					SuppliesPrintRecordOptionOption	= 1 << 5,
					
					// Makes this the default format for all documents. If specified by more than one plug-in, the last
					// one becomes the default. Use when adding a format.
					IsDefaultOptionOption			= 1 << 6,
					
					// The plug-in will not respond to the FileFormatCheckMessageType message. (For example, the
					// PhotoShop adapter plug-in always returns that no error occurred. Use when adding a format.
					NoAutoCheckFormatOption			= 1 << 10,
					
					// Read the file, creating artwork in a new template layer in the current document. Not used for
					// adding a format.
					CreateTemplateLayerOption		= 1 << 11,
					
					// Handle the extended data passed in a FileFormatGoMessageType message for a placement request. Use
					// when adding a format.
					HasExtendedDataOption			= 1 << 12,
					
					// This file format supplies its own startup objects (colors, patterns, and so on), Illustrator does
					// not copy the startup file. Use when adding a format.
					SkipStartupObjectsOptionOption	= 1 << 13,
					
					// Disable warning dialogs upon read and write. Not used for adding a format.
					NoWarningOptionOption			= 1 << 14,
					
					// Write the current document to a copy of the file it was loaded from. Not used for adding a format.
					SaveCopyOptionOption			= 1 << 15,
					
					// Prevents this file format from appearing in the file selection menu of the Open and Place dialogs.
					// Use when adding a format.
					SuppressUIOption				= 1 << 21,
					
					// Set in combination with \c #kFileFormatWrite for a Save As operation, to distinguish it from Save.
					// Not used for adding a format.
					WriteAsOption					= 1 << 22,
					
					// Always receive the Check message, even for operations this plug-in does not support. Allows an
					// opportunity to explicitly reject operations on files matching this plug-in's type. Use when
					// adding a format.
					CheckAlwaysOption				= 1 << 23,
					
					// Handle additional parameters passed in the FileFormatMessage::actionParam() method. These
					// supplement the usual parameters of the file format, and may not be complete. Can be used, for
					// instance, for scripting or optimizing. If set in the FileFormatGoMessageType message for a plug-in
					// that does not handle the option, you can ignore it. Not used for adding a format.
					ContainsPartialParametersOption	= 1 << 24,
					
					// Import only the SLO composite fonts. Do not import artwork or other global objects, and do not
					// perform font fauxing. Not used for adding a format.
					ImportCompositeFontsOption		= 1 << 25,
					
					// Treat the file as stationary - that is, open a copy with an Untitled name. Use only in conjunction
					// with ReadOption. Not used for adding a format.
					OpenUntitledCopyOption			= 1 << 26,
					
					// An option for the native (PGF) AI File Format Writer, telling it to write out only the indicated
					// palettes and the global objects, directly or indirectly. Not used for adding a format.
					WriteSwatchLibraryOption		= 1 << 27,
					
					// An option for the native (PGF) AI File Format Writer, telling it to write out only the indicated
					// palettes and the global objects, directly or indirectly. Not used for adding a format.
					WriteBrushLibraryOption			= 1 << 28,
					
					// An option for the native (PGF) AI File Format Writer, telling it to write out only the indicated
					// palettes and the global objects, directly or indirectly. Not used for adding a format.
					WriteStyleLibraryOption			= 1 << 29,
					
					// An option for the native (PGF) AI File Format Writer, telling it to write out only the indicated
					// palettes and the global objects, directly or indirectly. Not used for adding a format.
					WriteSymbolLibraryOption		= 1 << 30
				};
				
				/**
					\brief	Used to specify the extended capabilities of a file format
				*/
				enum ExtendedOptions
				{
					NoExtendedOptions					= 0,
					SaveMultiArtboardsExtendedOption	= 1 << 0
				};
				
				/**
					\brief	Wraps information needed to add a new file format
				*/
				struct AddData
				{
					/**
						\brief	Mac 4-char code file identifier. Plug-ins on Windows can ignore this.
					*/
					int32_t type;
					
					/**
						\brief	Description string to be shown in Illustrator's Save and Open dialogs, as UTF-8
					*/
					std::string title;
					
					/**
						\brief	Indicates the order of file format menu items, if the plug-in registers more than one.
						
						\note	Use 0 if only registering a single file format.
					*/
					int32_t titleOrder;
					
					/**
						\brief	Lists the DOS-style file extensions that are filtered in the Open dialog or appended to
								the file name in the Save dialog.
						
						\note	Each extension can be up to 5 characters in length, and a total of five 5-char
								extensions are supported (or six 4-char extensions, or eight 3-char extensions, or ten
								2-char extensions, or sixteen 1-char extensions). Any combination of extension lengths
								can be mixed, but their total length cannot exceed 31 characters combined (minus a
								single-character internal delimeter that is used between each extension).
					*/
					ExtVector extensions;
					
					/**
						\brief	AddData constructor, setting default values for all members
						\author	GW
						\date	02/2015
					*/
					AddData();
					
					/**
						\brief	AddData copy constructor, copying values from an existing AddData object
						\author	GW
						\date	02/2015
						
						\param	d_	Existing AddData object
					*/
					AddData(const AddData& d_);
					
					/**
						\brief	AddData destructor
						\author	GW
						\date	02/2015
					*/
					virtual ~AddData();
					
					/**
						\brief		Overloaded assignment operator, copying values from one AddData object to another
						\author		GW
						\date		02/2015
						
						\param		rhs_	Righthand side of the assignment
						\returns	The lefthand side of the assignment, but with its values updated
					*/
					AddData& operator=(const AddData& rhs_);
				};
		
				typedef std::unique_ptr<AddData> AddDataUP;
				typedef std::shared_ptr<AddData> AddDataSP;
				typedef std::weak_ptr<AddData> AddDataWP;

				/**
					\brief	Constructs an empty FileFormat object
					\author	GW
					\date	12/2014
					
					\note	To test if a FileFormat object is empty, call isEmpty() on it
					\note	Empty FileFormat objects do not relate to any actual Illustrator file format; they are
							designed to be "receivers" of some other FileFormat object via the overloaded assignment
							operator. Empty FileFormat objects are useless until such time (though it is safe to call
							any of their methods)
				*/
				FileFormat();
			
				/**
					\brief	Constructs a new FileFormat object from an existing FileFormat object (copy constructor)
					\author	GW
					\date	12/2014

					\param	ff_		Existing FileFormat object
				*/
				FileFormat(const FileFormat& ff_);

				/**
					\brief	Constructs a new FileFormat object from an AIFileFormatHandle
					\author	GW
					\date	12/2014
					
					\param	aiFileFormat_	Raw Illustrator AIFileFormatHandle
				*/
				FileFormat(const AIFileFormatHandle aiFileFormat_);
			
				/**
					\brief	Constructs a new FileFormat object from a name and some configuration options
					\author	GW
					\date	12/2014

					\param	name_		Unique name for the file format, as UTF-8
					\param	data_		Data that further describes the file format to be added
					\param	opt_		File format operations that this plug-in supports; a bitwise OR of the Options
										enum values marked as being applicable to adding a new format
					\param	extOpt_		Extended capabilities of the new format
				*/
				FileFormat(
					const std::string& name_,
					const AddData& data_,
					const Options opt_,
					const ExtendedOptions extOpt_
				);

				/**
					\brief	FileFormat object destructor
					\author	GW
					\date	12/2014
				*/
				virtual ~FileFormat();
				
				/**
					\brief		Assigns one FileFormat object to another
					\author		GW
					\date		12/2014

					\param		rhs_	Existing FileFormat object to copy values from
					\returns	The target FileFormat object, but with its values updated to match that of the rhs_
								argument
				*/
				virtual FileFormat& operator=(const FileFormat& rhs_);

				/**
					\brief		Gets whether the target FileFormat object is empty (constructed with the default ctor)
					\author		GW
					\date		12/2014
					
					\returns	true if the target FileFormat object is empty, false otherwise
				*/
				virtual bool isEmpty() const;

				/**
					\brief		Gets the file format handle around which the target object is wrapped
					\author		GW
					\date		12/2014
					
					\returns	AIFileFormatHandle for this object

					\note		Generally, you shouldn't need to call this method. Only use it if you know what you're
								doing. If a specific piece of functionality provided by Illustrator is not handled by
								this class (or related classes), then it should probably be added to the hdi_core library.
				*/
				virtual AIFileFormatHandle aiFileFormatHandle() const;
				
				/**
					\brief		Tests whether a given FileFormat object is the same as another
					\author		GW
					\date		12/2014

					\param		rhs_	FileFormat to compare against (righthand side of equality operator)
					\returns	true for the target and rhs_ being the same file format, false otherwise
				*/
				virtual bool operator==(const FileFormat& rhs_) const;
				
				/**
					\brief		Tests whether a given FileFormat object is not the same as another
					\author		GW
					\date		12/2014

					\param		rhs_	FileFormat to compare against (righthand side of inequality operator)
					\returns	true for the target and rhs_ being different file formats, false otherwise
				*/
				virtual bool operator!=(const FileFormat& rhs_) const;
				
				/**
					\brief		Gets the unique internal name of the target file format
					\author		GW
					\date		12/2014
					
					\returns	The format name, as UTF-8
				*/
				std::string name() const;
				
				/**
					\brief		Gets the display name of the target file format
					\author		GW
					\date		12/2014
					
					\returns	The format title, as UTF-8
				*/
				std::string title() const;
				
				/**
					\brief		Gets the Mac 4-char code file type identifier
					\author		GW
					\date		12/2014
					
					\returns	The file Mac ID
				*/
				int32_t macType() const;
				
				/**
					\brief		Gets the supported file extensions of the target file format
					\author		GW
					\date		12/2014
					
					\returns	The file extension strings, as UTF-8
				*/
				ExtVector extensions() const;
				
				/**
					\brief		Gets the options flags of the target file format
					\author		GW
					\date		12/2014
					
					\returns	A bitwise OR of Options enum values
				*/
				Options options() const;
				
				/**
					\brief	Sets the options flags of the target file format
					\author	GW
					\date	12/2014
					
					\param	opt_	New options flags; a bitwise OR of Options enum values
				*/
				void setOptions(const Options opt_);


			private:
				friend class Illustrator;
				friend void* __accessImpl(const FileFormat&);

				/**
					\brief	Private implementation data
				*/
				void* __data;
				
				/**
					\brief	Internal use only
					\author	GW
					\date	12/2014
				*/
				void* __impl() const;
				
				/**
					\brief	Internal use only
					\author	GW
					\date	12/2014
				*/
				bool __add(std::string&, AddData&, Options&, ExtendedOptions&) const;
		};
		
		typedef std::unique_ptr<FileFormat> FileFormatUP;
		typedef std::shared_ptr<FileFormat> FileFormatSP;
		typedef std::weak_ptr<FileFormat> FileFormatWP;
		
		extern void* __accessImpl(const FileFormat&);
	}
}

hdi::core::FileFormat::Options operator|(
	const hdi::core::FileFormat::Options lhs_,
	const hdi::core::FileFormat::Options rhs_
);

hdi::core::FileFormat::Options& operator|=(
	hdi::core::FileFormat::Options& lhs__,
	const hdi::core::FileFormat::Options rhs_
);

#endif
// HDI_CORE_AIP_MODE

#endif
// __HDI_CORE_FILE_FORMAT__
